/* myerhtml.c - Myer output: HTML format     -*- coding: latin-1 -*-

   2003 by Jonathan Yavner.  GPL license; see file COPYING for details.
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <math.h>

#include "myer.h"

#include "myerhtml.h"

/* Although this is also included by myer.h, we explicitly include it here
   to increase the marginal costs for the calls to concat and xmemdup. */
#include "libiberty.h"

#define DOCTYPE "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\" \"http://www.w3.org/TR/html40/strict.dtd\">"
#define ALLCH    0x7fffffff
#define BEFORECH 0x7ffffffe
#define AFTERCH  -1

int OutLine, OutCol, OutTab;
FILE *InFile, *OutFile;
static char *SrcFilename;


/* ----------------------------------------------------------------------------
   Copy raw text to HTML, converting required entities and dealing with
   tabs, until ENDLINE/ENDCOL is seen.  If endline==BEFORECH, endcol is a
   sentinel character; copying stops just *before* the sentinel.  If
   endline==AFTERCH, endcol is the sentinel; copying stops just *after*
   coping the sentinel.
*/
static void
HTML_Out( int endline, int endcol )
{
	int ch = 0, prevch;

	while (endline > OutLine
	       || (endline == OutLine && endcol > OutCol)
	       || (endline == AFTERCH && endcol != ch)) {
		prevch = ch;
		ch = getc(InFile);
		switch (ch) {
		case EOF:
			return;
		case '\n':
			OutCol = OutTab = 1;
			++OutLine;
			fputs( "<br>\n", OutFile );
			continue;
		case '\t': /* Fixed 8-char tab stops */
			while (OutTab & 7) {
				fputs( "&nbsp;", OutFile );
				++OutTab;
			}
			/* Fall through */
		case ' ':
			if (isspace(prevch)) {
				fputs( "&nbsp;", OutFile );
			} else {
				putc( ' ', OutFile );
			}
			break;
		case '<':
			fputs( "&lt;",  OutFile );
			break;
		case '>':
			fputs( "&gt;",  OutFile );
			break;
		case '&':
			fputs( "&amp;", OutFile );
			break;
		default:
			if (endline == BEFORECH && endcol == ch) {
				ungetc( ch, InFile );
				return;
			}
			putc( ch, OutFile );
		}
		++OutTab;
		++OutCol;
	}
} /* HTML_Out */


/* ----------------------------------------------------------------------------
   Select an RGB color for a coupling/cohesion pair.
*/
static RGB_t
ConvertToRGB( float coupling, float cohesion )
{
	RGB_t rgb;

	cohesion = pow( cohesion, 1.0/DEFAULT_GAMMA_COHESION );

	if (coupling == COUPLING_IFDEF) {
		/* Code is #ifdef'ed out */
		rgb = DEFAULT_COLOR_IFDEF;
	} else if (coupling == COUPLING_FUNLOCAL) {
		/* Nothing interesting to say about these locals */
		rgb = DEFAULT_COLOR_LOCAL;
	} else if (coupling == COUPLING_MODLOCAL) {
		/* Uncoupled local variable */
		rgb = DEFAULT_COLOR_MODLOCAL(cohesion);
	} else {
		coupling = pow( coupling, 1.0/DEFAULT_GAMMA_COUPLING );
		rgb = DEFAULT_COLOR_GLOBAL(coupling,cohesion);
	}

	return rgb;
} /* ConvertToRGB */


/* ----------------------------------------------------------------------------
   Process one segment of spots.  A segment is either the whole file or a
   sequence of spots embedded within another spot (e.g., a macro call).
   'end' is the first spot that is *not* part of the segment.
*/
static void
myerHtml_Segment( CHAIN_t *start, CHAIN_t *end )
{
	RGB_t rgb;

	while (start != end) {
		SPOT_t *s = start->spot;
		if (!s) {
			fprintf( stderr, "%s: File changed since previous reading?\n",
				 SrcFilename );
			exit( 1 );
		}
		if (s->kind == SPOT_FunDef) {
			/* Skip these for now, until we come up with some
			   sort of background color for them. */
			start = start->next;
			continue;
		}
		HTML_Out( s->line, s->col );
		if (s->kind == SPOT_MacroDef) {
			/* The SPOT_MacroDef is like SPOT_FunDef, but the
			   coupling/cohesion values refer to the macro's
			   name, whose position is not specified.  Figure
			   out now where the macro name is */
			HTML_Out( AFTERCH, 'f' ); /* Process #def */
			HTML_Out( AFTERCH, 'e' ); /* Process ine */
			HTML_Out( BEFORECH, *s->decl->name );
			s->endline = OutLine;
			s->endcol  = OutCol + strlen(s->decl->name);
		}
		rgb = ConvertToRGB( s->coupling, s->cohesion );
		while (start->next->spot
		       && start->next->spot->line    == s->line
		       && start->next->spot->col     == s->col
		       && start->next->spot->endline == s->endline
		       && start->next->spot->endcol  == s->endcol) {
			/* For overlays due to macros, take the worst case
			   of all the coupling/cohesion values that are
			   hiding behind the macro.  FIXME: Is this really
			   the best thing to do? It makes assert() very
			   bright because there's always a generated string
			   behind it. */
			RGB_t x = ConvertToRGB( start->next->spot->coupling,
						start->next->spot->cohesion );
			if (rgb.r < x.r) {
				rgb.r = x.r;
			}
			if (rgb.g < x.g) {
				rgb.g = x.g;
			}
			if (rgb.b < x.b) {
				rgb.b = x.b;
			}
			start = start->next;
		}
		if (s->kind == SPOT_ConstRef) {
			fprintf( OutFile, DEFAULT_HTML_STARTCONST );
		}
		fprintf( OutFile, "<font color=\"#%02x%02x%02x\">",
			 rgb.r, rgb.g, rgb.b );
		/* Find macro arguments other embedded spots */
		CHAIN_t *to = start->next;
		while (to->spot
		       && (to->spot->line < s->endline
			   || (to->spot->line == s->endline
			       && to->spot->col < s->endcol))) {
			to = to->next;
		}
		if (start->next != to) {
			myerHtml_Segment( start->next, to );
			start = to->prev;
		}
		HTML_Out( s->endline, s->endcol );
		fprintf( OutFile, "</font>" );
		if (s->kind == SPOT_ConstRef) {
			fprintf( OutFile, DEFAULT_HTML_ENDCONST );
		}
		start = start->next;
	}
} /* myerHtml_Segment */


/* ----------------------------------------------------------------------------
   Produce index.html to summarize the other output files.
*/
void
myerHtml_Index()
{
	char *fname = concat( Output_Directory, "index.html", 0 );
	FILE *out = fopen( fname, "w" );
	char *prev = 0;
	int prevlen = -1;

	if (!out) {
		perror( fname );
		exit( 1 );
	}

	if (Verbose) {
		fprintf( stderr, "[6] > %s\n", fname );
	}

	fprintf( out, "%s\n<html>\n", DOCTYPE );

	fprintf( out, "<body><i>Myer - %s</i><br>\n", My_Ctime() );

	for (int sid = 1; sid < NUM_IDS(WantedSIDs); ++sid) {
		SRC_t *src = WantedSIDs[sid];
		char *lastslash = strrchr( src->name, '/' );
		int newlen = (lastslash ? (lastslash-src->name+1) : 0);

		/* Group the files by directory */
		if (newlen != prevlen || strncmp( prev, src->name, newlen )) {
			if (prev) {
				fprintf( out, "</ul>\n" );
			}
			prev = src->name;
			prevlen = newlen;
			fprintf( out, "\n<h3>%.*s:</h3><ul>\n",
				 newlen-1, src->name );
		}

		/* Create outname for source file. Change slashes to
		   underscores. For reduced confusion, also change the
		   extension dot to an underscore */
		int len = strlen(src->name);
		src->outname = xmemdup(src->name,len+1,len+6);
		for (char *ptr = src->outname; *ptr; ++ptr) {
			if (*ptr == '/') {
				*ptr = '_';
			}
		}
		char *lastdot = strrchr( src->outname, '.' );
		if (lastdot && !strchr( lastdot, '/' )) {
			*lastdot = '_';
		}
		strcat( src->outname, ".html" );

		fprintf( out, "<li><a href=\"%s\">%s</a></li>\n",
			 src->outname, &src->name[newlen] );
	}

	if (prev) {
		fprintf( out, "</ul>\n" );
	}

	fprintf( out, "</body></html>\n" );

	fflush( out );
	if (ferror(out)) {
		perror( fname );
		exit( 1 );
	}
	fclose( out );
	free( fname );
} /* myerHtml_Index */


/* ----------------------------------------------------------------------------
   Convert one file to HTML, with coloring to indicate coupling/cohesion.
*/
void
myerHtml_Process( SRC_t *src )
{
	char *outname = concat( Output_Directory, src->outname, 0 );
	const RGB_t bg = DEFAULT_COLOR_BG;
	const RGB_t fg = DEFAULT_COLOR_FG;

	if (Verbose) {
		fprintf( stderr, "[6] > %s\n", outname );
	}

	if (!(InFile = fopen( src->name, "r" ))) {
		perror( src->name );
		exit( 1 );
	}
	if (!(OutFile = fopen( outname, "w" ))) {
		perror( outname );
		exit( 1 );
	}

	fprintf( OutFile, "%s\n<html><head><title>Myer - %s</title></head>\n",
		 DOCTYPE, src->name );
	fprintf( OutFile, "<body bgcolor=\"#%02x%02x%02x\">\n",
		 bg.r, bg.g, bg.b );
	fprintf( OutFile, "<font color=\"#%02x%02x%02x\"><tt>\n",
		 fg.r, fg.g, fg.b );

	OutLine = OutCol = OutTab = 1;
	SrcFilename = src->name;
	myerHtml_Segment( src->spots.next, &src->spots );

	assert( !feof(InFile) );
	HTML_Out( ALLCH, 1357924680 ); /* 2nd arg doesn't matter */
	assert( feof(InFile) );
	fclose( InFile );

	fprintf( OutFile, "</tt></font></body></html>\n" );

	fflush( OutFile );
	if (ferror(OutFile)) {
		perror( src->outname );
		exit( 1 );
	}
	fclose( OutFile );

	free( outname );
} /* myerHtml_Process */
