/* Myer.h - Global declarations for Myer        -*- coding: latin-1 -*-

    2003 by Jonathan Yavner.  GPL license; see file COPYING for details.
*/

#ifndef MYER_H
#define MYER_H

#include "libiberty.h"

/* Allocate an object of TYPE, set to all-zeroes */
#define NEW_(type) (type *) xcalloc( 1, sizeof(type) )

typedef struct COMPILATION_UNIT COMPILATION_UNIT_t;
typedef struct TOKEN_HASH       TOKEN_HASH_t;

typedef struct CHAIN CHAIN_t;
typedef struct SPOT  SPOT_t;
typedef struct SRC   SRC_t;
typedef struct FUNC  FUNC_t;
typedef struct DECL  DECL_t;


/* ----------------------------------------------------------------------------
   Structure of a compilation unit.
*/

enum TOKENCLASS {
	/* Only these are used in DECL_t.class */
	TOKEN_VARIABLE  = 'v',
	TOKEN_FUNCTION  = 'f',
	TOKEN_PARAMETER = 'p',
	TOKEN_FIELD     = 'F',
	TOKEN_STRUCT    = 's', /* Also union and enum types */
	TOKEN_TYPEDEF   = 't',
	TOKEN_LABEL     = 'L',
	TOKEN_ENUMCONST = 'E',
	/* Above and below are used in TOKEN_HASH_t.class */
	TOKEN_CHARCONST = 'C',
	TOKEN_NUMBCONST = 'N',
	TOKEN_STRCONST  = 'S',
	TOKEN_MACRODEF  = 'D',
	/* Below are used only by the phaseX intermediate files */
	TOKEN_COMMENT   = '*',
	TOKEN_FUNCSTART = '+',
	TOKEN_FUNCEND   = '-',
	TOKEN_SKIP      = '/', /* Code excluded by preprocessor conditional */
	TOKEN_ENDSTMT   = ';',
	TOKEN_FILEPOP   = '<', /* End of #include */
	TOKEN_DEF       = '=', /* Def-spot for variables and types */
	TOKEN_FILEPUSH  = '>', /* #include */
	TOKEN_FILESKIP  = 'I', /* Redundant #include that was skipped */
	TOKEN_BUILTIN   = 'B', /* Last UID for compiler built-in stuff */
	TOKEN_PUBLIC    = 'P', /* Preceding var or func is a public symbol */
	TOKEN_KEYWORD   = 'K',
	TOKEN_MACROREF  = 'M',
	TOKEN_DECLSPOT  = 'd', /* Extern-decl-spot for variables and functions*/
	TOKEN_EQUATEUID = 'e', /* Two UID's refer to the same object */
	TOKEN_ENDINPUT  = 'q',
	TOKEN_FILECHG   = '|', /* #line */
	/* Below are used for phase2 => phase3 intermediate files */
	TOKEN_DECL      = ':', /* An instantiation of a func/var/type/etc */
	TOKEN_NEWFILE   = '@', /* Start of info on a new file */
	/* Phase-3/4 intermediate files */
	TOKEN_SRCMARK   = '#', /* Marking on a source file */
	TOKEN_NUMREFS   = '&', /* Number of refs to this include file */
	TOKEN_PHASE4    = '%'  /* Separator between phase2/3 and phase4 part */
};

enum SPOTKIND { /* Kinds of spots that might be colorized */
	SPOT_Skip        = TOKEN_SKIP,
	SPOT_IncludeFile = TOKEN_FILEPOP,
	SPOT_ConstRef    = TOKEN_CHARCONST,
	SPOT_MacroDef    = TOKEN_MACRODEF,
	SPOT_MacroRef    = TOKEN_MACROREF,
	SPOT_FunDef      = TOKEN_FUNCTION,
	SPOT_Def         = TOKEN_DEF,
	SPOT_Decl        = TOKEN_DECLSPOT,
	SPOT_Ref         = 'r'
};

enum INCLTYPE { /* Kinds of special "include file" type things */
	I_FUNC,	 /* Global functions and predefined macros */
	I_VAR,	 /* Variables */
	I_TYPE,  /* Types and structs */
	I_CHAR,	 /* Character constants 'f' */
	I_STR,	 /* String constants "f" */
	I_NUMB,	 /* Numeric constants 0xf */
	I_LOCAL, /* Decls declared within current file */
	I_MAX
};

#define INCLTYPE_NAMES "fvtCSN" /* Names for incltypes, just for cuteness */

struct CHAIN { /* Chain of things */
	CHAIN_t *prev, *next;
	union {
		SPOT_t *spot;
		FUNC_t *func;
		void *generic;
		int sid;
	}; /* Anonymous union */
};

struct SPOT { /* An interesting spot in the source text */
	enum SPOTKIND kind;
	int line, col, endline, endcol;
	DECL_t *decl;
	float coupling, cohesion;
#define COUPLING_IFDEF    -1.0 /* Flag value for SPOT_Skip */
#define COUPLING_FUNLOCAL -2.0 /* Flag value for function-local identifiers */
#define COUPLING_MODLOCAL -3.0 /* Flag value for module-local identifiers */
	union {
		int sid;		/* For SPOT_IncludeFile */
		CHAIN_t *dlink; /* For others except SPOT_Skip, this
				     SPOT's link in its DECL's chain */
	} ref;
};


/* Poor man's sparse matrix: We group decls into blocks of 256, allocating
   blocks only if some decl in that block is referenced by the source file.
   If nonsparse, phase-4 would need >1GB of virtual memory to analyse gcc!
 */
typedef int INTBLOCK_t[256];

struct SRC { /* Source file */
	char *name;	      /* Official filename from gcc */
	char *outname;	      /* Filename in output directory */
	dev_t device;
	ino_t inode;	      /* Device & inode, unique name of file */
	int baseline;	      /* Line for last file change, or -1 if finished */
#define INCONSISTENT_CONTENTS -12345678 /* Flag baseline for myermerge */
#define WANTED_FOR_OUTPUT     -23456789 /* Flag baseline for myercalc */
	int fileline;	      /* File line that matches `baseline' */
	int startline;	      /* Preprocessor line number for line 1 of file */
	int numfuncs;	      /* Number of SPOT_FunDef's */
	int numinclude;       /* Number of SPOT_IncludeFile's in this src */
	int numrefs;	      /* Number of files that #include this one */
	INTBLOCK_t **incldecls;/* Include file number for each decl (-1 = not
				 referenced from this file, 0 = undef
				 global, 1 = current file */
	INTBLOCK_t **declrefs; /* Total use-counts for each decl. */
	int *inclrefs;	      /* Array [numinclude+I_MAX] giving total number of
				 decl's referenced from each include file. */
	int *inclsids;	      /* Array [numinclude+I_MAX] giving SID for
				 each include file. */
	int sid;	      /* For pretty printing of phase-2 output */
	CHAIN_t spots;	      /* Interesting spots in this file.
				 spots.  spots.spot must be 0 */
	CHAIN_t includes;     /* Spots with kind==SPOT_IncludeFile */
	CHAIN_t funcs;	      /* A FUNC_t for each function.  Root node
				 describes outside-of-function area. */
};

struct FUNC { /* A function */
	int uid;
	INTBLOCK_t **declrefs; /* Per-function use-counts for each decl. */
	int *inclrefs;	      /* Array [numinclude+I_MAX] giving number of decl
				 references to each include file. */
};

struct DECL { /* Info on one UID */
	char *name; /* Input filename */
	enum TOKENCLASS class;
	CHAIN_t refs; 	      /* Spots where referenced or declared */
	CHAIN_t decls;	      /* SIDs where declared */
	CHAIN_t *def;	      /* Spot where defined, or first spot where
				 declared, or 0 for undefined ref */
	int defsid;	      /* Source file containing 'def', or 0 */
	int numfiles;	      /* Number of files (processed in phase 4) that
				 have references to this decl */
	int numrefs;	      /* Number of references to this decl in
				 phase-4 processed files. */
	int uid;	      /* For pretty output from phase 2*/
	unsigned int ispublic : 1; /* Same in all compilation units */
	unsigned int isglobal : 1; /* Same throughout one compilation unit */
	unsigned int defwaiting : 1; /* See myerToken_From_Phase2_File */
	unsigned int wasmerged : 1; /* See myerMerge */
	unsigned int deleted : 1; /* Has been merged with another */
};

struct COMPILATION_UNIT { /* Info on one compilation unit */
	SRC_t **sids;
	DECL_t **uids, **fake_uids;
	struct htab *names;
};


/* ----------------------------------------------------------------------------
   Phase-1 stuff.
*/

struct INPUT_FORMAT { /* Description of a .myerN input file format */
	char class;
	int  numread;    /* Expected return from scanf */
	const char *fmt; /* Format string for scanf */
};

struct htab *Names_hash;

void myerParse_From_File( char *filename );


/* ----------------------------------------------------------------------------
   Phase-2 stuff.
*/

typedef struct TOKEN {
	enum TOKENCLASS class;
	int line, col, endline, endcol;
	TOKEN_HASH_t *hashref;
	int uid;
	char *name;
} TOKEN_t;

int NextSID;

void		   myerToken( TOKEN_t *tok );
COMPILATION_UNIT_t myerToken_Read( char *filename, int lastch );
COMPILATION_UNIT_t myerToken_From_File( char **filename );


/* ----------------------------------------------------------------------------
   Phase 3.
*/

int myerMerge_From_File( char **filename );

SRC_t  **AllSIDs;
DECL_t **AllUIDs;
int NextAllSID;
struct htab *AllHash;


/* ----------------------------------------------------------------------------
   Phase 4.
*/

int  myerSum_From_File( char **filename );
void myerSum_To_File( void );
void myerSum_Prescan( void );
void myerSum_Process( SRC_t * );

SRC_t **WantedSIDs;
int SpecialSIDs[I_MAX];



/* ----------------------------------------------------------------------------
   Phase 5.
*/

struct SCALE_FACTORS {
	float global, file, func;
};

/* Scaling factors.  These specify how the marginal cost formulae for the
   current function, current module, and all modules in program are to be
   combined.  The three numbers should add up to 1.0 */
#define DEFAULT_SCALE_COUPLING { 0.35, 0.50, 0.15 }
			    /* Global, file, function */
#define DEFAULT_SCALE_COHESION { 0.35, 0.50, 0.15 }

int  myerCalc_From_File( char **filename );
void myerCalc_To_File( void );
void myerCalc_Process( SRC_t * );

extern struct SCALE_FACTORS ScaleCoupling, ScaleCohesion;


/* ----------------------------------------------------------------------------
   Phase 6.
*/

#define DEFAULT_OUTPUT_DIRECTORY "Myer/"

typedef struct RGB {
	int r, g, b;
} RGB_t;

char *Output_Directory;

void myerHtml_Index( void );
void myerHtml_Process( SRC_t * );


/* ----------------------------------------------------------------------------
   Arrays of source files and decls, used by phases 2 and 3.  Element [0] is
   size.
*/

void AddSID( SRC_t ***sids, SRC_t *src );
void AddUID( DECL_t ***uids, int uid, DECL_t *src );
void MergeUID( DECL_t *global, DECL_t *one );


#define NUM_IDS(array) (array ? (int) array[0] : 0)


/* ----------------------------------------------------------------------------
   Other stuff
*/

int Verbose;
int Cc1_major, Cc1_minor;


FILE *Call_cc1( const char *filename, int dump_to_stdout );
void Stat_File( const char *filename, dev_t *dev, ino_t *ino );
char *My_Ctime( void );
void Common_Write( char phase,SRC_t **sids,DECL_t **uids,DECL_t **fake_uids );
void Common_Free( COMPILATION_UNIT_t cu );

/* ----------------------------------------------------------------------------
   Hash-table entries for tokens.
*/

struct TOKEN_HASH {
	void *data;
	enum TOKENCLASS class;
	char  key[4]; /* Allocated as large as needed */
};

#define HASH_HEADER offsetof(TOKEN_HASH_t,key)

struct htab  *Token_Hash_init(void);
TOKEN_HASH_t *Token_Hash_add( struct htab *, char *,
			      enum TOKENCLASS, int insertp );
void          TokenHash( struct htab *table, TOKEN_t *t );


/* =========================================================================
   Operations on circular doubly-linked chains.
*/


/* ----------------------------------------------------------------------------
   Initialize a circular chain to point to itself.
*/
static inline void
InitChain( CHAIN_t *c )
{
	c->prev = c->next = c;
} /* InitChain */


/* ----------------------------------------------------------------------------
   Nonzero if chain is empty.
*/
static inline int
EmptyChain( CHAIN_t *c )
{
	return c->next == c;
} /* EmptyChain /*/


/* ----------------------------------------------------------------------------
   Insert an object into a chain before a given node.
*/
static inline CHAIN_t *
InsertChain( CHAIN_t *c, void *it )
{
	CHAIN_t *x = NEW_(CHAIN_t);

	x->generic    = it;
	x->next       = c;
	x->prev       = c->prev;
	x->next->prev = x;
	x->prev->next = x;

	return x;
} /* InsertChain */


/* ----------------------------------------------------------------------------
   Delete a link from its chain.
*/
static inline void *
UnlinkChain( CHAIN_t *c )
{
	void *result = c->generic;

	c->next->prev = c->prev;
	c->prev->next = c->next;
	free( c );
	return result;
} /* UnlinkChain */


/* ----------------------------------------------------------------------------
   Scan backwards through CHAIN, looking for an object that makes the EXPR
   true.  EXPR contains references to VAR, which is set to each item pointed
   to by CHAIN until a nonzero result is obtained.

   If no match is found, the item for the chain header is returned, which
   should be zero.
*/
#define FIND_CHAIN_(chain,type,var,expr) ({				\
			CHAIN_t *_c = (chain)->prev;			\
			while (_c != (chain)) {				\
				type *var = _c->generic;		\
				if (expr) {				\
					break;				\
				}					\
				_c = _c->prev;				\
			}						\
			(type *) _c->generic;				\
		})

#endif
