/* myer.c - Myer main program        -*- coding: latin-1 -*-

    2003 by Jonathan Yavner.  GPL license; see file COPYING for details.
 */

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#include <string.h>
#include <sys/stat.h>

#include "libiberty.h"
#include "hashtab.h"

#include "myer.h"

/* From myerenv.c */
extern int myerenv_count;
extern char *myerenv_list[];

struct SCALE_FACTORS ScaleCoupling = DEFAULT_SCALE_COUPLING;
struct SCALE_FACTORS ScaleCohesion = DEFAULT_SCALE_COHESION;

static char *Output_filename;

char *CC1_command;
int Verbose;

#define MAX_PHASE 6


/* ----------------------------------------------------------------------------
   Convert a filename to device/inode, to detect different filenames that
   point to the same file, such as "./x.c" and "/home/joe/x.c".
*/
void
Stat_File( const char *filename, dev_t *dev, ino_t *ino )
{
	struct stat buf;

	if (!stat(filename,&buf)) {
		*dev = buf.st_dev;
		*ino = buf.st_ino;
	} else {
		perror( filename );
		/* Create a fake ID from the filename.  Since filenames are
		   hashed, multiple occurrences of this name should have the
		   same address. */
		*dev = 0;
		*ino = (int) filename;
	}
} /* Stat_File */


/* ----------------------------------------------------------------------------
   Returns nonzero if FILENAME is an existing directory.
*/
static int
IsDir( const char *filename )
{
	struct stat buf;

	return !stat( filename, &buf )
		&& S_ISDIR(buf.st_mode);
} /* IsDir */


/* ----------------------------------------------------------------------------
   Hide the silly complexity of ctime().
*/
char *
My_Ctime()
{
	char *result;

	time_t now = time(0);
	result = ctime(&now);
	result[24] = 0; /* Get rid of stupid newline */
	return result;
} /* My_Ctime */


/* ============================================================================
   Working with arrays of items.
*/

#define SRC_ALLOC    50 /* Number of srcs  to add to allocation when needed */
#define DECL_ALLOC 1000 /* Number of decls to add to allocation when needed */


/* ----------------------------------------------------------------------------
   Adds a src to the array.  Reallocs array if needed.  Element [0] is
   number of elements in the array.
*/
void
AddSID( SRC_t ***sids, SRC_t *src )
{
	int numsids = NUM_IDS(*sids);

	while (src->sid >= numsids) { /* Need more room in SID array */
		numsids += SRC_ALLOC;
		*sids = (SRC_t **) xrealloc( *sids, numsids*sizeof(SRC_t*) );
		memset( &(*sids)[numsids-SRC_ALLOC],
			0,
			SRC_ALLOC*sizeof(SRC_t *) );
		* (int *) &(*sids)[0] = numsids;
	}
	(*sids)[src->sid] = src;
} /* AddSID */


/* ----------------------------------------------------------------------------
   Adds a decl to the array.  Reallocs array if needed.  Element [0] is
   number of elements in the array.

*/
void
AddUID( DECL_t ***uids, int uid, DECL_t *decl )
{
	int numuids = NUM_IDS(*uids);

	while (uid >= numuids) { /* Need more room in array */
		numuids += DECL_ALLOC;
		*uids = (DECL_t **) xrealloc( *uids, numuids*sizeof(DECL_t*) );
		memset( &(*uids)[numuids-DECL_ALLOC],
			0,
			DECL_ALLOC*sizeof(DECL_t *) );
		* (int *) &(*uids)[0] = numuids;
	}
	assert( !(*uids)[uid] );
	(*uids)[uid] = decl;
} /* AddUID */




/* ============================================================================
   Working with hash tables.
*/


/* ----------------------------------------------------------------------------
   Computes hash value for TOKEN_HASH_t
*/
static hashval_t
Token_Hash_compute( const TOKEN_HASH_t *entry )
{
	return htab_hash_string(entry->key) + entry->class;
} /* Token_Hash_compute */


/* ----------------------------------------------------------------------------
   Determines whether two TOKEN_HASH_t entries are equal.
*/
static int
Token_Hash_equal( const TOKEN_HASH_t *left, const TOKEN_HASH_t *right )
{
	return (left->class == right->class) && !strcmp(left->key,right->key);
} /* Token_Hash_equal */


/* ----------------------------------------------------------------------------
   Initialize a hash-table for token identifiers.
*/
htab_t
Token_Hash_init()
{
	return htab_create( 30000,
			    (htab_hash) Token_Hash_compute,
			    (htab_eq)   Token_Hash_equal,
			    (htab_del)  free );
} /* Token_Hash_init */


/* ----------------------------------------------------------------------------
   Hashes a name/class pair.  Returns a pointer to the hash-entry, or 0 if
   insertp==0 and the token is not already known.

   This method uses a double memcpy.  Could switch to obstacks (from
   Libiberty) to avoid the second copy.
*/
TOKEN_HASH_t *
Token_Hash_add( htab_t table, char *name, enum TOKENCLASS class, int insertp )
{
	/* Note: see scanf strings "%4095s" and "%4095[^\n]" in myertoken.c */
	union {
		TOKEN_HASH_t mh;
		char buf[4096+HASH_HEADER];
	} x;
	int len = strlen(name)+1;

	assert( len < 4096 );
	x.mh.class = class;
	memcpy( x.mh.key, name, len );

	TOKEN_HASH_t **find = (TOKEN_HASH_t **) htab_find_slot( table,
								&x.mh,
								insertp );
	if (!find) {
		return 0;
	}

	if (!*find) {
		/* New identifier */
		*find = xmalloc( HASH_HEADER+len );
		x.mh.data = 0;
		memcpy( *find, &x.mh, HASH_HEADER+len );
	}

	return *find;
} /* Token_Hash_add */


/* ----------------------------------------------------------------------------
   Adds the name to the hash-table, if not already done.  Sets t->hash.
   Adjusts t->name to point to the hashed string.
*/
void
TokenHash( htab_t table, TOKEN_t *t )
{
	if (!t->name) {
		/* Doesn't need doing */
		return;
	}

	if (t->hashref) {
		/* Already done. */
		return;
	}

	t->hashref = Token_Hash_add( table, t->name, 0, INSERT );
	t->name    = t->hashref->key;
} /* TokenHash */



/* ============================================================================
   Common file I/O for writing files, phases 2/3/4.
*/

/* ----------------------------------------------------------------------------
   Write a declaration to phase-2/3/4 output file.
*/
static void
Common_Write_Decl( DECL_t *d )
{
	char x;

	if (d->ispublic) {
		x = 'P';
		assert( d->isglobal );
	} else if (!d->isglobal) {
		x = 'f';
	} else {
		x = ' ';
	}

	if (d->def) {
		printf( "%c #%d\t%c%c\t@%d %d\t%s\n",
			TOKEN_DECL,
			d->uid, d->class, x,
			d->defsid, d->def->spot->line, d->name );
	} else {
		printf( "%c #%d\t%c%c\t\t%s\n",
			TOKEN_DECL,
			d->uid, d->class, x, d->name );
	}
} /* Common_Write_Decl */


/* ----------------------------------------------------------------------------
   Write phase-2/3/4 data to file.
*/
void
Common_Write( char phase, SRC_t **sids, DECL_t **uids, DECL_t **fake_uids )
{
	if (Verbose) {
		fprintf( stderr, "[%c] > (decls)\n", phase );
	}

	printf( "%c myer%c * %s  -*- tab-width: 12 -*-\n\n",
		TOKEN_COMMENT, phase, My_Ctime() );

	for (int j = 1; j < NUM_IDS(uids); ++j) {
		if (uids[j]
		    && uids[j]->uid == j
		    && !uids[j]->deleted) {
			Common_Write_Decl( uids[j] );
		}
	}
	putchar( '\n' );

	if (fake_uids) {
		for (int j = 1; j < NUM_IDS(fake_uids); ++j) {
			if (fake_uids[j]) {
				Common_Write_Decl( fake_uids[j] );
			}
		}
		putchar( '\n' );
	}

	for (int j = 1; j < NUM_IDS(sids); ++j) {
		SRC_t *src = sids[j];
		if (!src) {
			continue;
		}
		if (Verbose) {
			fprintf( stderr, "[%c] > %s\n", phase, src->name );
		}
		if (src->baseline == INCONSISTENT_CONTENTS) {
			printf( "%cInconsistent contents\n",
				TOKEN_SRCMARK );
		} else if (src->baseline == WANTED_FOR_OUTPUT) {
			printf( "%cWanted\n", TOKEN_SRCMARK );
		}
		printf( "%c%d %s\n", TOKEN_NEWFILE, src->sid, src->name );
		if (src->numrefs) {
			printf( "%c %d\n", TOKEN_NUMREFS, src->numrefs );
		}
		for (CHAIN_t *sc = src->spots.next;
		     sc != &src->spots;
		     sc = sc->next) {
			SPOT_t *spot = sc->spot;
			int target;
			char targtype;
			switch (spot->kind) {
			case TOKEN_REPARSE:
				/* These hopefully got replaced */
				continue;
			case SPOT_IncludeFile:
				targtype = '@';
				target = spot->ref.sid;
				break;
			case SPOT_Skip:
				targtype = ' ';
				target = 0;
				break;
			default:
				targtype = '#';
				target = spot->decl->uid;
			}
			printf( "%c %c%d\t%d,%d\t%d,%d\n",
				spot->kind, targtype, target,
				spot->line, spot->col,
				spot->endline, spot->endcol );
		}

		putchar( '\n' );
	}

	fflush(stdout);
	if (ferror(stdout)) {
		perror( "Output error" );
		exit( 1 );
	}
} /* Common_Write */


/* ----------------------------------------------------------------------------
   Free the memory in an INTBLOCK_t.
*/
#ifdef LEAK_CHECK
static void
Common_Free_IntBlock( INTBLOCK_t **ib )
{
	for (int j = ((NUM_IDS(AllUIDs)+255)>>8)-1; j >= 0; --j) {
		if (ib[j]) {
			free( ib[j] );
		}
	}
	free( ib );
} /* Common_Free_IntBlock */
#endif


/* ----------------------------------------------------------------------------
   Free up memory that represents a compilation unit.
*/
void
Common_Free( COMPILATION_UNIT_t cu )
{
	for (int j = 1; j < NUM_IDS(cu.sids); ++j) {
		SRC_t *src = cu.sids[j];
		if (src) {
			while (!EmptyChain( &src->spots )) {
				SPOT_t *s = UnlinkChain(src->spots.next);
				if (s->kind != SPOT_Skip
				    && s->kind != SPOT_IncludeFile) {
					UnlinkChain( s->ref.dlink );
				}
				free( s );
			}
#ifdef LEAK_CHECK
			/* When checking for leaks, clean up additional
			   pointers that occur only in the global
			   source-list. */
			if (src->inclsids) {
				free( src->inclsids );
			}
			if (src->inclrefs) {
				free( src->inclrefs );
			}
			if (src->declrefs) {
				Common_Free_IntBlock( src->declrefs );
			}
			if (src->incldecls) {
				Common_Free_IntBlock( src->incldecls );
			}
			if (src->includes.next) {
				while (!EmptyChain(&src->includes)) {
					UnlinkChain(src->includes.next);
				}
			}
			if (src->funcs.next) {
				FUNC_t *func = src->funcs.func;
				for (;;) {
					Common_Free_IntBlock( func->declrefs );
					free( func->inclrefs );
					free( func );
					if (EmptyChain(&src->funcs)) {
						break;
					}
					func = UnlinkChain( src->funcs.next );
				}
			}
			if (src->outname) {
				free( src->outname );
			}
#endif
			free( src );
		}
	}
	free( cu.sids );

	for (int j = 1; j < NUM_IDS(cu.uids); ++j) {
		DECL_t *d = cu.uids[j];
		if (d && d->uid == j) {
			while (!EmptyChain( &d->decls )) {
				UnlinkChain( d->decls.next );
			}
			free( d );
		}
	}
	free( cu.uids );

	for (int j = 1; j < NUM_IDS(cu.fake_uids); ++j) {
		DECL_t *d = cu.fake_uids[j];
		if (d) {
			while (!EmptyChain( &d->refs )) {
				UnlinkChain( d->refs.next );
			}
			while (!EmptyChain( &d->decls )) {
				UnlinkChain( d->decls.next );
			}
			free( d );
		}
	}
	free( cu.fake_uids );

	if (cu.names) {
		htab_delete( cu.names );
	}
} /* Common_Free */



/* ============================================================================
   Top-level driver.
*/


/* ----------------------------------------------------------------------------
   If Output_Filename ends with ".o", change it to a more appropriate
   extension.  This is to support Makefile rules like
	${CC} ${CFLAGS} -c -o $@ $<
   where we just change CC=myer and then run a normal make.
*/
static int
Select_Output( int phase )
{
	char suffix[6] = { 'm', 'y', 'e', 'r', phase+'0' };

	if (!Output_filename) {
		/* Just send to stdout */
		return phase;
	}

	char *extension = strrchr( Output_filename, '.' );
	if (extension && !strcmp( extension, ".o" )) {
		/* Change to selected extension. */
		if (phase == MAX_PHASE+1) {
			/* No phase specified.  Use phase 2, since we're
			   creating separate output files for each
			   compilation unit. */
			phase = 2;
			suffix[4] = '2';
		}
		extension[1] = 0;
		Output_filename = concat( Output_filename, suffix, 0 );
	}
	if (phase < MAX_PHASE) {
		/* Substitute requested file instead of stdout */
		if (!freopen( Output_filename, "w", stdout )) {
			perror( Output_filename );
			exit( 1 );
		}
	} else {
		/* Output will be a directory */
		if (Output_filename[strlen(Output_filename)-1] == '/') {
			/* Explicit directory */
			Output_Directory = Output_filename;
		} else {
			Output_Directory = concat( Output_filename,
						   "_",
						   DEFAULT_OUTPUT_DIRECTORY,
						   0 );
		}
	}
	return phase;
} /* Select_Output */


/* ----------------------------------------------------------------------------
   Call the cc1 subprogram to produce the phase-1 file.  Result is the pipe
   that will receive the phase-1 output, or 0 if want_stdout.
*/
FILE *
Call_cc1( const char *filename, int want_stdout )
{
	char *cmd = concat( CC1_command, " ", filename, 0 );
	FILE *result;

	if (want_stdout) {
		if (system(cmd)) {
			exit( 1 );
		}
		result = 0;
	} else {
		result = popen( cmd, "r" );
		if (!result) {
			perror( cmd );
			exit( 1 );
		}
	}
	free( cmd );
	return result;
} /* Call_cc1 */


/* ----------------------------------------------------------------------------
   Created WantedSIDs array, listing only the src files for which analysis
   has been requested.
*/
static void
SetUpWanted(void)
{
	int sid, j = 1;

	for (sid = 1; sid < NUM_IDS(AllSIDs); ++sid) {
		if (AllSIDs[sid] && AllSIDs[sid]->baseline==WANTED_FOR_OUTPUT) {
			++j;
		}
	}
	WantedSIDs = (SRC_t **) xmalloc( j * sizeof(SRC_t *) );
	WantedSIDs[0] = (void *) j;
	j = 0;
	for (sid = 1; sid < NUM_IDS(AllSIDs); ++sid) {
		if (AllSIDs[sid] && AllSIDs[sid]->baseline==WANTED_FOR_OUTPUT) {
			WantedSIDs[++j] = AllSIDs[sid];
		}
	}
} /* SetUpWanted */


/* ----------------------------------------------------------------------------
   Sort source files by name.  First, sort by directory path, then by
   basename within directory.
*/
static int
Compare_SRC( const SRC_t **left, const SRC_t **right )
{
	const char *lname =  (*left)->name;
	const char *rname = (*right)->name;

	const char *ldir = strrchr( lname, '/' );
	const char *rdir = strrchr( rname, '/' );

	int ldlen, rdlen;
	if (ldir) {
		ldlen = (ldir - lname) + 1;
	} else {
		ldlen = 0;
	}
	if (rdir) {
		rdlen = (rdir - rname) + 1;
	} else {
		rdlen = 0;
	}

	int result = strncmp( lname, rname, ((ldlen<rdlen)?ldlen:rdlen) );
	if (!result) {
		result = ldlen - rdlen;
	}
	if (!result) {
		result = strcmp( &lname[ldlen], &rname[rdlen] );
	}
	return result;
} /* Compare_SRC */


/* ------------------------------------------------------------------------- */
int
main( int argc, char *argv[] )
{
	int j, sid = 0, phase_desired = MAX_PHASE+1, phase_received = 0;
	char *progname, *ptr;
	COMPILATION_UNIT_t cu;
	dev_t dev;
	ino_t ino;
	static const char Period[2] = ".";

	/* Fix up shell-escape characters in the command-line arguments,
	   which will be passed to myer_cc1 via system().
	   FIXME: Why isn't there a standard library function for this? */
	for (int j = 1; j < argc; ++j) {
		char *x = argv[j];
		for (int k = 0; x[k]; ++k) {
			if (x[k] == '"'
			    || x[k] == '\''
			    || x[k] == '\\'
			    || isspace(x[k])) {
				char save[2] = { x[k], 0 };
				x[k] = 0;
				if (x == argv[j]) {
					x = concat( x, "\\", save, &x[k+1], 0 );
				} else {
					x=reconcat( x,x,"\\",save,&x[k+1],0 );
				}
				++k;
			}
		}
		if (argv[j] != x) { /* Some shell-escape was found */
			argv[j] = x;
		}

	}

	/* Setup for phase 1.  Create command line */
	ptr = strrchr( argv[0], '/' );
	if (ptr) {
		*ptr = 0;
		progname = xstrdup(argv[0]);
		/* Restore argv[0] so 'ps' will display nicely */
		*ptr = '/';
	} else {
		/* Our program doesn't have an absolute path.  Assume
		   myer_cc1 is in current directory. */
		progname = (char *) Period;
	}
	CC1_command = concat( progname,
			      "/myer_cc1"
			      " -iprefix " GCC_INSTALL
			      " -quiet"
			      " -ftabstop=1"  /* We need tabs = single chars */
			      " -dx@" /* Skip assembly, emit our phase-1 */
			      " -o /dev/null", /* Send assembly to bit bucket */
			      0 );
	for (j = 0; j < myerenv_count; ++j) {
		CC1_command = reconcat( CC1_command,
					CC1_command,
					" ",
					myerenv_list[j],
					0 );
	}

	/* Process our command-line options */
	for (j = 1; j < argc; ++j) {
		if (argv[j][0] != '-') {
			continue;
		}
		if (argv[j][1] == 'P') {
			phase_desired = strtol( &argv[j][2], &ptr, 0 );
			if (!*ptr
			    && phase_desired >= 1
			    && phase_desired <= MAX_PHASE) {
				continue;
			}
			fprintf( stderr, "Invalid -P option\n" );
			exit( 1 );
		}
		if (!strcmp( argv[j], "-v" )) {
			Verbose = 1;
			continue;
		}
		if (!strcmp( argv[j], "-c" )) {
			/* Skip this - we never run the linker */
			continue;
		}
		if (!strcmp( argv[j], "-o" )) {
			/* Intercept this - we write to a filename with
			   .myerN instead of .o */
			Output_filename = argv[++j];
			/* Prevent next pass through arglist from thinking
			   this is an input file. */
			argv[j] = "-skip";
			continue;
		}
		/* Check for inappropriate gcc options */
		assert( strcmp( argv[j], "-E" ) && strcmp( argv[j], "-S" ) );
		/* Pass this option to gcc */
		CC1_command = reconcat( CC1_command,CC1_command," ",argv[j],0 );
		continue;
	}

	Output_Directory = DEFAULT_OUTPUT_DIRECTORY;
	phase_desired = Select_Output( phase_desired );

	if (Verbose) {
		fprintf( stderr, "myer (" __DATE__ ")" );
		if (phase_desired <= MAX_PHASE) {
			fprintf( stderr, ": phase = %d", phase_desired );
		}
		if (phase_desired < MAX_PHASE) {
			fprintf( stderr, ", output = %s",
				 (Output_filename
				  ? Output_filename
				  : "<stdout>") );
		} else {
			fprintf( stderr, ", output = %s", Output_Directory );
		}
		putc( '\n', stderr );
	}

	/* Process command-line input files */
	for (j = 1; j < argc; ++j) {
		if (argv[j][0] == '-') {
			continue;
		}

		switch (phase_desired) {
		case 1:
			if (Verbose) {
				fprintf( stderr, "[1] %s\n", argv[j] );
			}
			Call_cc1( argv[j], 1 );
			break;
		case 2:
			cu = myerToken_From_File( &argv[j] );
			Common_Write( '2', cu.sids, cu.uids, cu.fake_uids );
			Common_Free( cu );
			break;
		case 3:
			phase_received = myerMerge_From_File( &argv[j] );
			if (phase_received >= 3) {
				sid = j;
				j = argc;
			}
			break;
		case 4:
			phase_received = myerSum_From_File( &argv[j] );
			if (phase_received >= 3) {
				sid = j;
				j = argc;
			}
			break;
		case 5:
		case 6:
		case MAX_PHASE+1:
			phase_received = myerCalc_From_File( &argv[j] );
			if (phase_received >= 3) {
				sid = j;
				j = argc;
			}
			break;
		default:
			; /* Checked above */
		}
	}

	if (phase_desired < 3) {
		/* Output already produced above */
		return 0;
	}

	if (phase_received >= 3) {
		do {
			/* Skip over trailing options */
			++sid;
		} while (sid < argc && argv[sid][0] == '-');
		if (sid < argc) {
			fprintf( stderr, "Extraneous arguments - the phase 3/4/5 file specifies everything\n" );
			exit( 1 );
		}
	}

	if (phase_received < 3) {
		/* Mark the desired files */
		if (Verbose) {
			fprintf( stderr, "myer: Mark the 'wanted' files\n" );
		}
		for (j = 1; j < argc; ++j) {
			if (argv[j][0] == '-') {
				continue;
			}
			Stat_File( argv[j], &dev, &ino );
			for (sid = 1; sid < NUM_IDS(AllSIDs); ++sid) {
				if (AllSIDs[sid]
				    && AllSIDs[sid]->device == dev
				    && AllSIDs[sid]->inode == ino) {
					break;
				}
			}
			if (sid >= NUM_IDS(AllSIDs)) {
				/* This file wasn't read during any
				   compilation.  It should be some sort of
				   metadata file */
				char *extension = strrchr( argv[j], '.' );
				if (extension
				    && !memcmp( extension, ".myer", 5 )) {
					continue;
				}
				fprintf( stderr,
					 "%s: No information on this file\n",
					 argv[j] );
				continue;
			}
			if (AllSIDs[sid]->baseline == INCONSISTENT_CONTENTS) {
				fprintf( stderr, "%s: File does not have same contents in all compilation units - cannot annotate\n", AllSIDs[sid]->name );
			} else {
				AllSIDs[sid]->baseline = WANTED_FOR_OUTPUT;
			}
		}
	}

	if (phase_desired == 3) {
		Common_Write( '3', AllSIDs, AllUIDs, 0 );
		return 0;
	}

	SetUpWanted();

	if (phase_received < 4) {
		myerSum_Prescan();
		for (j = 1; j < NUM_IDS(WantedSIDs); ++j) {
			myerSum_Process( WantedSIDs[j] );
		}
	}

	if (phase_desired == 4) {
		myerSum_To_File();
		return 0;
	}

	if (phase_received < 5) {
		for (j = 1; j < NUM_IDS(WantedSIDs); ++j) {
			myerCalc_Process( WantedSIDs[j] );
		}
	}

	if (phase_desired == 5) {
		myerCalc_To_File();
		return 0;
	}

	qsort( &WantedSIDs[1], NUM_IDS(WantedSIDs)-1, sizeof(SRC_t *),
	       (int (*)(const void *, const void *)) Compare_SRC );

	/* Create output directory if needed */
	if (!IsDir( Output_Directory )) {
		/* Not already a directory - create it */
		if (mkdir( Output_Directory, 0666 )) {
			perror( Output_Directory );
			exit( 1 );
		}
	}

	/* Create output files */
	myerHtml_Index();
	for (j = 1; j < NUM_IDS(WantedSIDs); ++j) {
		myerHtml_Process( WantedSIDs[j] );
	}

#ifdef LEAK_CHECK /* Define this for use with valgrind */
	for (j = 0; j < I_LOCAL; ++j) {
		free( AllSIDs[SpecialSIDs[j]]->name );
	}
	Common_Free( (COMPILATION_UNIT_t){ AllSIDs, AllUIDs, 0, AllHash } );
	if (progname != Period) {
		free( progname );
	}
	free( CC1_command );
	free( WantedSIDs );
#endif

	return 0;
} /* main - Myer */
