/* myersum.c - Produce sums to aid in calculations  -*- coding: latin-1 -*-

   2003 by Jonathan Yavner <jyavner@member.fsf.org>
   GPL license; see file COPYING for details.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include "myer.h"

/* This seems redundant because myer.h includes libiberty.h.  By explicitly
   including it here, we give the calls to xcalloc higher cohesion penalty
   because they are treated as coming from this include instead of from
   myer.h like everything else. */
#include "libiberty.h"

#define CALLOC_(type,count) (type *) xcalloc( count, sizeof(type) )


/* ----------------------------------------------------------------------------
   Prescan each source file, producing a list of #include spots.  Set up the
   special SIDs for things that are extern'ed but not #include'd.
*/
void
myerSum_Prescan()
{
	if (Verbose) {
		fprintf( stderr, "myer: Begin sums\n" );
	}
	for (int sid = 1; sid < NUM_IDS(AllSIDs); ++sid) {
		SRC_t *src = AllSIDs[sid];
		if (!src) {
			continue;
		}
		InitChain( &src->includes );
		for (CHAIN_t *sc = src->spots.next;
		     sc != &src->spots;
		     sc=sc->next) {
			if (sc->spot->kind == SPOT_IncludeFile) {
				InsertChain( &src->includes,
					     (void *) sc->spot->ref.sid );
				++AllSIDs[sc->spot->ref.sid]->numrefs;
			}
		}
	}

	/* Create the special SIDs for extern'ed things */
	for (int j = 0; j < I_LOCAL; ++j) {
		SRC_t *src = NEW_(SRC_t);
		src->name = strdup("\\.-special" );
		src->name[1] = INCLTYPE_NAMES[j];
		src->sid = ++NextAllSID;
		SpecialSIDs[j] = src->sid;
		InitChain( &src->spots );
		AddSID( &AllSIDs, src );
	}
} /* myerSum_Prescan */


/* ----------------------------------------------------------------------------
   Produce sums for one file.
*/
void
myerSum_Process( SRC_t *src )
{
	int target;
	FUNC_t *curfunc;

	if (Verbose) {
		fprintf( stderr, "[4] = %s\n", src->name );
	}
	src->numinclude = 0;
	for (CHAIN_t *ic=src->includes.next; ic!=&src->includes; ic=ic->next) {
		++src->numinclude;
	}

	int *indirect = CALLOC_( int, NUM_IDS(AllSIDs) );
	int *levels   = CALLOC_( int, NUM_IDS(AllSIDs) );

	src->inclsids = CALLOC_( int, src->numinclude+I_MAX );

	/* For each file directly or indirectly included, record the
	   explicit include through which decls from that SID will be
	   perceived by this source file, which we call "include #1". The
	   undeclared builtins are "include #0".

	   For each indirect include, we record how deeply nested the call
	   was.  If a later #include gets to the same place with a shallower
	   nesting, we match user expectations by treating the refs as
	   coming from the later include, even though the compiler doesn't.
	*/
	src->inclsids[I_LOCAL] = src->sid;
	++src->numrefs; /* Compiling this file is a reference */
	indirect[src->sid] = I_LOCAL;
	levels[src->sid] = 0;
	int block, sector, numinclude = I_MAX-1, level = 1;
	for (CHAIN_t *ic=src->includes.next; ic!=&src->includes; ic=ic->next) {
		src->inclsids[++numinclude] = ic->sid;
		CHAIN_t *todo = NEW_(CHAIN_t); /* Chain of sid*65536 + level */
		InitChain( todo );
		assert( NUM_IDS(AllSIDs) < 65536 );
		int sid = ic->sid;
		do {
			if (!indirect[sid] || levels[sid] > level) {
				indirect[sid] = numinclude;
				levels[sid] = level;
				CHAIN_t *xc;
				CHAIN_t *x1 = &AllSIDs[sid]->includes;
				for (xc = x1->next; xc != x1; xc = xc->next) {
					assert( level < 1000+I_MAX );
					target = (xc->sid << 16) + (level + 1);
					InsertChain( todo, (void *) target );
				}
			}
			target = (int) UnlinkChain( todo->next );
			sid    = target >> 16;
			level  = target & 0xffff;
		} while (sid);
	}

	/* Compute number of links between this file and others */
	src->inclrefs = CALLOC_( int, src->numinclude+I_MAX );
	src->declrefs = CALLOC_( INTBLOCK_t *, (NUM_IDS(AllUIDs)+255)>>8 );
	src->incldecls= CALLOC_( INTBLOCK_t *, (NUM_IDS(AllUIDs)+255)>>8 );
	int func_endline = 0, func_endcol = 0;

	InitChain( &src->funcs );
	src->funcs.func = NEW_(FUNC_t);
	curfunc = src->funcs.func;
	curfunc->inclrefs = CALLOC_( int, src->numinclude+I_MAX );
	curfunc->declrefs = CALLOC_( INTBLOCK_t *, (NUM_IDS(AllUIDs)+255)>>8 );

	for (CHAIN_t *sc=src->spots.next; sc!=&src->spots; sc=sc->next) {
		SPOT_t *s = sc->spot;
		if (s->kind == SPOT_Skip
		    || s->kind == SPOT_IncludeFile) {
			continue;
		}
		if (s->kind == SPOT_FunDef) {
			++src->numfuncs;
			curfunc = NEW_(FUNC_t);
			curfunc->inclrefs = CALLOC_(int,src->numinclude+I_MAX);
			curfunc->declrefs = CALLOC_(INTBLOCK_t *,
						    (NUM_IDS(AllUIDs)+255)>>8 );
			curfunc->uid = s->ref.decl->uid;
			InsertChain( &src->funcs, curfunc );
			func_endline = s->endline;
			func_endcol  = s->endcol;
			continue;
		}
		if (s->endline > func_endline
		    || (s->line == func_endline && s->col > func_endcol)) {
			curfunc = src->funcs.func;
		}
		if (!s->ref.decl->isglobal) {
			continue;
		}
		block  = s->ref.decl->uid >> 8;
		sector = s->ref.decl->uid & 255;
		if (!src->incldecls[block]) {
			/* First time for any UID in this block */
			src->declrefs[block]  = NEW_(INTBLOCK_t);
			src->incldecls[block] = NEW_(INTBLOCK_t);
			memset( src->incldecls[block], -1, sizeof(INTBLOCK_t) );
		}
		target = (*src->incldecls[block])[sector];
		if (target < 0) {
			/* First time for this decl */
			target = indirect[s->ref.decl->defsid];
			if (s->ref.decl->def
			    && s->ref.decl->def->spot->kind == SPOT_Decl) {
				/* Was declared but never defined.  Might be a
				   forward extern that's later repeated in an
				   include file. */
				target = 0;
			}
			if (!target) {
				/* Official declaration isn't one we know
				   about.  Scan all declarations to look for
				   one that's in our include files */
				for (CHAIN_t *ic = s->ref.decl->decls.next;
				     ic != &s->ref.decl->decls;
				     ic = ic->next) {
					target = indirect[ic->sid];
					if (target >= I_MAX) {
						break;
					}
					/* Another forward-ref in our target
					   file.  If not declared somewhere
					   else, treat as global-ref */
					target = 0;
				}
			}
			if (!target) {
				/* Put into different global categories
				   depending on type */
				switch (s->ref.decl->class) {
				case TOKEN_FUNCTION:
				case TOKEN_MACRODEF:
					target = I_FUNC;
					break;
				case TOKEN_VARIABLE:
				case TOKEN_FIELD:
					target = I_VAR;
					break;
				case TOKEN_TYPEDEF:
				case TOKEN_STRUCT:
					target = I_TYPE;
					break;
				case TOKEN_CHARCONST:
				case TOKEN_ENUMCONST: /* Header file refers to enum from some other header file */
					target = I_CHAR;
					break;
				case TOKEN_STRCONST:
					target = I_STR;
					break;
				case TOKEN_NUMBCONST:
					target = I_NUMB;
					break;
				case TOKEN_PARAMETER:
					/* These aren't really global */
					target = I_LOCAL;
					continue;
				default:
					assert(!"Unexpected global decl class");
				}
			}
			if (!src->inclsids[target]) {
				/* First reference to a special target */
				assert( target < I_MAX );
				src->inclsids[target] = SpecialSIDs[target];
				++AllSIDs[SpecialSIDs[target]]->numrefs;
			}
			(*src->incldecls[block])[sector] = target;
			++src->inclrefs[target];
			++s->ref.decl->numfiles;
		}
		if (!curfunc->declrefs[block]) {
			/* First time seen any decl in this block within
			   this function */
			curfunc->declrefs[block] = NEW_(INTBLOCK_t);
		}
		if (!(*curfunc->declrefs[block])[sector]) {
			/* First time seen this decl in this function */
			++curfunc->inclrefs[target];
		}
		++(*curfunc->declrefs[block])[sector];
		++(*src->declrefs[block])[sector];
		++s->ref.decl->numrefs;
	}

	free( levels );
	free( indirect );
} /* myerSum_Process */


/* ----------------------------------------------------------------------------
   Produce phase-4 intermediate file.
*/
void
myerSum_To_File()
{
	int j, k, m;
	CHAIN_t *fc;
	char type;

	Common_Write( '4', AllSIDs, AllUIDs, 0 );

	if (Verbose) {
		fprintf( stderr, "[4] > (file & function sums)\n" );
	}

	printf( "%c\n", TOKEN_PHASE4 );
	for (j = 1; j < NUM_IDS(WantedSIDs); ++j) {
		SRC_t *src = WantedSIDs[j];
		CHAIN_t *ic = &src->includes;
		printf( "\n%c%d %d %d\n", TOKEN_NEWFILE, src->sid,
			src->numinclude, src->numfuncs );
		for (k = 0; k < src->numinclude+I_MAX; ++k) {
			if (k < I_LOCAL) {
				m = SpecialSIDs[k];
				type = INCLTYPE_NAMES[k];
			} else if (k == I_LOCAL) {
				m = src->sid;
				type = ' ';
			} else {
				ic = ic->next;
				m = ic->sid;
				type = ' ';
			}
			printf( "%c @%d %d%c\n",
				SPOT_IncludeFile, m, src->inclrefs[k], type );
			for (m = 1; m < NUM_IDS(AllUIDs); ++m) {
				if (src->incldecls[m>>8]
				    && (*src->incldecls[m>>8])[m&255] == k) {
					printf( "%c\t#%d\t%d\n",
						SPOT_Ref, m,
						(*src->declrefs[m>>8])[m&255] );
				}
			}
		}
		fc = &src->funcs;
		do {
			FUNC_t *func = fc->func;
			printf( "%c #%d:", SPOT_FunDef, func->uid );
			for (k = 0; k < src->numinclude+I_MAX; ++k) {
				printf( " %d", func->inclrefs[k] );
			}
			putchar( '\n' );
			for (m = 1; m < NUM_IDS(AllUIDs); ++m) {
				if (func->declrefs[m>>8]
				    && (*func->declrefs[m>>8])[m&255]) {
					printf( "%c\t#%d\t%d\n",
						SPOT_Ref, m,
						(*func->declrefs[m>>8])[m&255]);
				}
			}
			fc = fc->next;
		} while (fc != &src->funcs);
	}

	if (Verbose) {
		fprintf( stderr, "[4] > (uid sums)\n" );
	}
	putchar( '\n' );
	for (j = 1; j < NUM_IDS(AllUIDs); ++j) {
		if (!AllUIDs[j] || !AllUIDs[j]->numfiles) {
			continue;
		}
		printf( "%c #%d\t%d\t%d\n", TOKEN_DECL, j,
			AllUIDs[j]->numfiles, AllUIDs[j]->numrefs );
	}
} /* myerSum_To_File */


/* ----------------------------------------------------------------------------
   Read a .myer4 file.
*/
static void
myerSum_Read( char *filename )
{
	COMPILATION_UNIT_t cu = myerToken_Read( filename, TOKEN_PHASE4 );
	int j, sid, uid, numinclude, numfuncs, numrefs, block, sector;
	SRC_t *src = 0;
	FUNC_t *func = 0;
	char type;

	AllSIDs = cu.sids;
	AllUIDs = cu.uids;
	AllHash = cu.names;
	assert( !cu.fake_uids );

	while ((j = getchar()) != EOF) {
		switch (j) {
		case '\n':
			continue;
		case TOKEN_NEWFILE:
			if (scanf("%d %d %d\n",&sid,&numinclude,&numfuncs)!= 3){
				goto Parse_Error;
			}
			src = AllSIDs[sid];
			assert( src && src->baseline==WANTED_FOR_OUTPUT );
			src->numinclude = numinclude;
			src->numfuncs   = numfuncs;
			src->incldecls = CALLOC_( INTBLOCK_t *,
						  (NUM_IDS(AllUIDs)+255)>>8 );
			src->declrefs = CALLOC_( INTBLOCK_t *,
						 (NUM_IDS(AllUIDs)+255)>>8 );
			src->inclrefs = CALLOC_( int, numinclude+I_MAX );
			src->inclsids = CALLOC_( int, numinclude+I_MAX );
			InitChain( &src->includes );
			InitChain( &src->funcs );
			numinclude = -1;
			break;
		case SPOT_IncludeFile:
			if (scanf(" @%d %d%c\n",&sid,&numrefs,&type) != 3) {
				goto Parse_Error;
			}
			++numinclude;
			if (numinclude < I_LOCAL) {
				assert( !sid
					|| EmptyChain(&AllSIDs[sid]->spots) );
				assert( type == INCLTYPE_NAMES[numinclude] );
			} else if (numinclude == I_LOCAL) {
				assert( sid == src->sid );
			} else {
				assert( sid );
				InsertChain( &src->includes, (void *) sid );
			}
			src->inclrefs[numinclude] = numrefs;
			src->inclsids[numinclude] = sid;
			func = 0;
			break;
		case SPOT_FunDef:
			if (scanf( " #%d:", &uid ) != 1) {
				goto Parse_Error;
			}
			if (!uid) {
				src->funcs.func = NEW_(FUNC_t);
			} else {
				InsertChain( &src->funcs, NEW_(FUNC_t) );
			}
			func = src->funcs.prev->func;
			func->uid = uid;
			func->inclrefs = CALLOC_( int, numinclude+1 );
			func->declrefs = CALLOC_( INTBLOCK_t *,
						  (NUM_IDS(AllUIDs)+255)>>8 );
			for (j = 0; j <= numinclude; ++j) {
				if (scanf( " %d", &numrefs ) != 1) {
					goto Parse_Error;
				}
				func->inclrefs[j] = numrefs;
			}
			if (getchar() != '\n') {
				goto Parse_Error;
			}
			break;
		case SPOT_Ref:
			if (scanf( " #%d %d\n", &uid, &numrefs ) != 2) {
				goto Parse_Error;
			}
			block  = uid >> 8;
			sector = uid & 255;
			if (!func) {
				if (!src->incldecls[block]) {
					src->declrefs[block] = NEW_(INTBLOCK_t);
					src->incldecls[block]= NEW_(INTBLOCK_t);
					memset( src->incldecls[block],
						-1,
						sizeof(INTBLOCK_t) );
				}
				(*src->incldecls[block])[sector] = numinclude;
				(*src->declrefs[block])[sector] = numrefs;
			} else {
				if (!func->declrefs[block]) {
					func->declrefs[block]=NEW_(INTBLOCK_t);
				}
				(*func->declrefs[block])[sector] = numrefs;
			}
			break;
		case TOKEN_DECL:
			if (scanf( " #%d %d %d", &uid, &j, &numrefs ) != 3) {
				goto Parse_Error;
			}
			AllUIDs[uid]->numfiles = j;
			AllUIDs[uid]->numrefs  = numrefs;
			break;
		default:
			goto Parse_Error;
		}
	}
	return;

 Parse_Error:
	fprintf( stderr, "Parse error in phase-4 file\n" );
	exit( 1 );
} /* myerSum_Read */


/* ----------------------------------------------------------------------------
   API for phase 4.
*/
int
myerSum_From_File( char **fnamep )
{
	char *filename = *fnamep;
	char *extension = strrchr( filename, '.' );

	if (extension && !strcmp( extension, ".myer4" )) {
		if (Verbose) {
			fprintf( stderr, "[4] < %s\n", filename );
		}
		myerSum_Read( filename );
		return 4;
	} else {
		return myerMerge_From_File( fnamep );
	}
} /* myerSum_From_File */
