/* myerparse.c - Callback for gcc to produce tokens -*- coding: latin-1 -*-

   2003 by Jonathan Yavner.  GPL license; see file COPYING for details.
*/

#include "config.h"
#include "system.h"
#include "cpplib.h"
#include "cpphash.h"
#include "tree.h"
#include "myer.h"

static cpp_hashnode Conditional_name = { .ident = { 3, (__u_char *) "#if" } };
static cpp_token Conditional = { .val = { .node = &Conditional_name } };
static cpp_token Implicit_Funcdef = { .line = -99 };
static cpp_reader *CurReader;

/* Save these until their meanings show up */
#define NUMIDENTS 32 /* Function name, old-style parameter decls, #if */
struct {
	unsigned int line, col;
	const unsigned char *name;
} Myer_idents[NUMIDENTS];

static int Endstmt_line, Endstmt_col, last_builtin;
static char *Struct_Prefix = (char *) "";

static int prev_line, prev_col;
static const char *prev_name;


/* ----------------------------------------------------------------------------
   Saves an identifier for later recall when its classification shows up.
*/
static void
myerSaveIdent( const cpp_token *tok )
{
	int j;

	for (j = 0; j < NUMIDENTS; ++j) {
		if (!Myer_idents[j].name) {
			break;
		}
	}

	if (j >= NUMIDENTS) {
		cpp_error( CurReader, "Myer: Too many postponed identifiers" );
		j = 0;
	}
	Myer_idents[j].line = tok->line;
	Myer_idents[j].col  = tok->col;
	Myer_idents[j].name = tok->val.node->ident.str;
} /* myerSaveIdent */


/* ----------------------------------------------------------------------------
   Finds an identifier by matching name.
*/
static void
myerFindIdent( const char *name, unsigned int *line, unsigned int *col )
{
	int j;

	for (j = 0; j < NUMIDENTS; ++j) {
		if (Myer_idents[j].name &&
		    !ustrcmp( (unsigned char *) name, Myer_idents[j].name )) {
			*line = Myer_idents[j].line;
			*col  = Myer_idents[j].col;
			Myer_idents[j].name = 0;
			return;
		}
	}

	cpp_error( CurReader, "Myer: No siting for identifier %s", name );
	*line = *col = 0;
} /* myerFindIdent */


/* ----------------------------------------------------------------------------
   Make sure all preceding idents are used up.
*/
static void
myerCheckIdents(void)
{
	int j;

	for (j = 0; j < NUMIDENTS; ++j) {
		if (Myer_idents[j].name) {
			cpp_error_with_line( CurReader,
					     Myer_idents[j].line,
					     Myer_idents[j].col,
				    "Myer: No meaning found for identifier %s",
					     Myer_idents[j].name );
			Myer_idents[j].name = 0;
		}
	}
} /* myerCheckIdents */


/* ------------------------------------------------------------------------- */
void
myerParse( enum MYER_PARSE type, ... )
{
	va_list ap;
	const cpp_reader *pfile;
	const cpp_token *tok;
	cpp_token temptok;
	cpp_hashnode *hnode;
	const cpp_context *ctx;
	const struct line_map *new_map;
	tree n, r;
	unsigned int j, line, col;
	int k;
	char typech;
	const char *ident;
	unsigned char *nodename;
	char resolved_filename[PATH_MAX];

	/* Example of extern-linkage. Also an example of a hack! */
	extern struct file_stack *input_file_stack;

	if (!Myer_mode) {
		return;
	}

	va_start( ap, type );

	switch (type) {
	case MYER_PARSE_NUMBCONST:
		tok = va_arg( ap, cpp_token * );
		printf( "%c %d,%d\t\t%s\n",
			TOKEN_NUMBCONST,
			tok->line, tok->col, tok->val.str.text );
		break;

	case MYER_PARSE_CHARCONST:
		tok = va_arg( ap, cpp_token * );
		printf( "%c %d,%d\t\t'%s'\n",
			TOKEN_CHARCONST,
			tok->line, tok->col, tok->val.str.text );
		break;

	case MYER_PARSE_STRCONST:
		tok = va_arg( ap, cpp_token * );
		printf( "%c %d,%d\t\t\"%s\"\n",
			TOKEN_STRCONST,
			tok->line, tok->col, tok->val.str.text );
		break;

	case MYER_PARSE_END_STATEMENT:
		Endstmt_line = CurReader->line;
		Endstmt_col  = CPP_BUF_COLUMN( CurReader->buffer,
					       CurReader->buffer->cur );
		break;

	case MYER_PARSE_IDENT:
		if (Endstmt_line) {
			myerCheckIdents();
			printf( "%c %d,%d\n",
				TOKEN_ENDSTMT, Endstmt_line, Endstmt_col );
			Endstmt_line = Endstmt_col = 0;
		}
		tok = va_arg( ap, cpp_token * );
		myerSaveIdent( tok );
		break;

	case MYER_PARSE_KEYWORD:
		n = va_arg( ap, tree );
		myerFindIdent( IDENTIFIER_POINTER(n), &line, &col );
		printf( "%c %d,%d\t\t%s\n",
			TOKEN_KEYWORD, line, col, IDENTIFIER_POINTER(n) );
		break;

	case MYER_PARSE_STRUCT_START:
		n = va_arg( ap, tree );
		if (!n) {
			ident = "";
		} else {
			ident = IDENTIFIER_POINTER(n);
		}
		Struct_Prefix = reconcat( Struct_Prefix,
					  Struct_Prefix, ident, ".",
					  0 );
		break;

	case MYER_PARSE_STRUCT_END:
		for (k = strlen(Struct_Prefix)-1; k > 0; --k) {
			if (Struct_Prefix[k-1] == '.') {
				break;
			}
		}
		if (k < 0) { /* An END with no matching START */
			abort();
		}
		Struct_Prefix[k] = 0;
		break;

	case MYER_PARSE_REF:
		n = va_arg( ap, tree );
		if (!n) {
			/* Skip unnamed types */
			va_end(ap);
			return;
		}
		r = va_arg( ap, tree );
		if (DECL_P(r)) {
			j = DECL_UID(r);
		} else {
			j = TYPE_UID(r);
		}
		if (~Myer_mode & MYER_MODE_READY) {
			/* Skip builtins */
			last_builtin = j;
			va_end(ap);
			return;
		}
		myerFindIdent( IDENTIFIER_POINTER(n), &line, &col );
		if (line == (unsigned int) -99) {
			/* Fake identifier - hide it */
			break;
		}
		switch (TREE_CODE(r)) {

		case FUNCTION_DECL: typech = TOKEN_FUNCTION;  break;
		case LABEL_DECL:    typech = TOKEN_LABEL;     break;
		case CONST_DECL:    typech = TOKEN_ENUMCONST; break;
		case TYPE_DECL:     typech = TOKEN_TYPEDEF;   break;
		case VAR_DECL:      typech = TOKEN_VARIABLE;  break;
		case PARM_DECL:     typech = TOKEN_PARAMETER; break;

		case RECORD_TYPE:
		case UNION_TYPE:
		case ENUMERAL_TYPE:
			typech = TOKEN_STRUCT;
			break;

		case FIELD_DECL:
			if (DECL_CONTEXT(r)) {
				typech = TOKEN_FIELD;
				break;
			}
			/* Special case for first declaration of a field --
			   prefix its name with that of its containing
			   struct */
			printf( "%c %d,%d\t#%d\t%d %s%s\n",
				TOKEN_FIELDDEF, line, col, j,
				strlen(IDENTIFIER_POINTER(n)),
				Struct_Prefix, IDENTIFIER_POINTER(n) );
			va_end(ap);
			return;

		default:
			cpp_error(CurReader,"Unexpected def type");
			typech = '?';
		}
		printf( "%c %d,%d\t#%d\t%s\n",
			typech, line, col, j, IDENTIFIER_POINTER(n) );
		prev_line = line;
		prev_col  = col;
		prev_name = IDENTIFIER_POINTER(n);
		break;

	case MYER_PARSE_REPARSE_TYPE:
		n = va_arg( ap, tree );
		hnode = CPP_HASHNODE(GCC_IDENT_TO_HT_IDENT(n));
		if (hnode->ident.str != (unsigned char *) prev_name) {
			/* Should be last-seen item */
			abort();
		}
		printf( "%c\t\t%s\n", TOKEN_REPARSE, hnode->ident.str );
		temptok.line = prev_line;
		temptok.col  = prev_col;
		temptok.val.node = hnode;
		myerSaveIdent( &temptok );
		break;

	case MYER_PARSE_MARK_DECL:
		r = va_arg( ap, tree );
		if (DECL_P(r)) {
			j = DECL_UID(r);
		} else {
			j = TYPE_UID(r);
		}
		printf( "%c\t#%d\n", TOKEN_DECLSPOT, j );
		break;

	case MYER_PARSE_MARK_PUBLIC:
		r = va_arg( ap, tree );
		printf( "%c\t#%d\n", TOKEN_PUBLIC, DECL_UID(r) );
		break;

	case MYER_PARSE_DEF:
		r = va_arg( ap, tree );
		if (DECL_P(r)) {
			j = DECL_UID(r);
		} else {
			j = TYPE_UID(r);
		}
		printf( "%c\t#%d\n", TOKEN_DEF, j );
		break;

	case MYER_PARSE_IMPLICIT_FUNCDEF:
		/* Hide following ref, since we've already seen the def */
		nodename = va_arg( ap, unsigned char * );
		r = va_arg( ap, tree );
		printf( "%c\t#%d\n", TOKEN_PUBLIC, DECL_UID(r) );
		Implicit_Funcdef.val.node = cpp_lookup( CurReader,
							nodename,
							ustrlen(nodename) );
		myerSaveIdent( &Implicit_Funcdef );
		break;

	case MYER_PARSE_FUNCSTART:
		r = va_arg( ap, tree );
		printf( "%c %d,%d\t#%d\t%s +++++++++++++++++++++++++\n",
			TOKEN_FUNCSTART,
			CurReader->line,
			CPP_BUF_COLUMN( CurReader->buffer,
					CurReader->buffer->cur ),
			DECL_UID(r),
			IDENTIFIER_POINTER(DECL_NAME(r)) );
		break;

	case MYER_PARSE_FUNCEND:
		r = va_arg( ap, tree );
		printf( "%c %d,%d\t#%d\t%s -------------------------\n",
			TOKEN_FUNCEND,
			CurReader->line,
			CPP_BUF_COLUMN( CurReader->buffer,
					CurReader->buffer->cur ),
			DECL_UID(r),
			IDENTIFIER_POINTER(DECL_NAME(r)) );
		break;

	case MYER_PARSE_START_SKIPPING:
		pfile = va_arg( ap, cpp_reader * );
		Conditional.line = pfile->directive_line;
		myerSaveIdent( &Conditional );
		break;

	case MYER_PARSE_END_SKIPPING:
		pfile = va_arg( ap, cpp_reader * );
		myerFindIdent( (char *) Conditional.val.node->ident.str,
			       &line, &col );
		printf( "%c %d-%d\n",
			TOKEN_SKIP, line, pfile->directive_line );
		break;

	case MYER_PARSE_MACRO_DEF:
		pfile = va_arg( ap, cpp_reader * );
		hnode = va_arg( ap, cpp_hashnode * );
		if (~Myer_mode & MYER_MODE_READY) {
			/* Skip builtins & command-line */
			return;
		}
		printf( "%c %d-%d\t\t%s\n",
			TOKEN_MACRODEF,
			pfile->directive_line,
			pfile->line-1,
			hnode->ident.str );
		break;

	case MYER_PARSE_MACRO_REF:
		pfile = va_arg( ap, cpp_reader * );
		tok   = va_arg( ap, cpp_token * );
		hnode = va_arg( ap, cpp_hashnode * );
		ctx   = pfile->context;
		/* FIXME: Is there a more reliable way to figure out where
		   the end of the macroref is? */
		if (!ctx->prev->prev) {
			/* In file */
			line = pfile->line;
			col  = CPP_BUF_COLUMN( pfile->buffer,
					       pfile->buffer->cur );
		} else if (tok[1].type != CPP_OPEN_PAREN) {
			/* Macro with no args, invoked from argument list to
			   another macro */
			line = tok->line;
			col  = tok->col + hnode->ident.len - 1;
		} else {
			/* It's a function-like macro invoked from the
			   argument list of another macro. */
			const cpp_token *ct;
			j = 0;
			for (ct = &tok[1];
			     ct->type != CPP_EOF && ct->type != CPP_PADDING;
			     ++ct) {
				if (ct->type == CPP_OPEN_PAREN) {
					++j;
				} else if (ct->type == CPP_CLOSE_PAREN) {
					--j;
				}
				if (!j) {
					break;
				}
			}
			if (ct->type != CPP_CLOSE_PAREN) {
				abort();
			}
			line = ct->line;
			col = ct->col;
		}
		printf( "%c %d,%d\t%d,%d\t%s\n",
			TOKEN_MACROREF,
			tok->line, tok->col,
			line, col,
			hnode->ident.str );
		break;

	case MYER_PARSE_SKIPINCLUDE:
		ident = va_arg( ap, char * );
		printf( "%c %d\t\t%s\n",
			TOKEN_FILESKIP, CurReader->directive_line, ident );
		break;

	case MYER_PARSE_FILE_CHANGE:
		CurReader = va_arg( ap, cpp_reader * );
		new_map = va_arg( ap, struct line_map * );
		if (!realpath( new_map->to_file, resolved_filename )) {
			/* Can't resolve? - use unresolved name */
			strcpy( resolved_filename, new_map->to_file );
		}
		switch (new_map->reason) {
		case LC_RENAME:
			if (*new_map->to_file == '<') {
				/* Skip prologue & command-line */
				break;
			}
			if (!last_builtin) {
				/* #line directive */
				printf( "%c %d\t%d\t%s\n",
					TOKEN_FILECHG,
					new_map->from_line,
					new_map->to_line,
					resolved_filename );
				break;
			}
			/* End of prologue, start of actual program */
			Myer_mode |= MYER_MODE_READY;
			printf( "%c\t\t-*- tab-width: 14 -*-\n",
				TOKEN_COMMENT );
			printf( "%c\t#%d\n",
				TOKEN_BUILTIN, last_builtin );
			last_builtin = 0;
			/* Fall through */

		case LC_ENTER:
			if (!input_file_stack) {
				/* Ignore first-time entry -- the original
				   input file will be seen again as an
				   LC_RENAME. */
				break;
			}
			printf( "%c %d\t\t%s\n",
				TOKEN_FILEPUSH,
				new_map->from_line,
				resolved_filename );
			break;
		case LC_LEAVE:
			printf( "%c %d\t%d\t%s\n",
				TOKEN_FILEPOP,
				new_map->from_line,
				new_map->to_line,
				resolved_filename );
			break;
		}
		break;

	case MYER_PARSE_END_INPUT:
		printf( "%c\n", TOKEN_ENDINPUT );
		myerCheckIdents();
		Endstmt_line = Endstmt_col = 0; /* Setup for next file */
		fflush( stdout );
		if (ferror(stdout)) {
			cpp_error_from_errno( CurReader, "Myer output" );
			abort();
		}
		break;

	default:
		cpp_error( CurReader, "Myer: Unimplemented parse code" );
	}
	va_end(ap);
} /* myerParse */
