/* myermerge.c - Merge compilation units into master list -*-coding: latin-1-*-

    2003 by Jonathan Yavner.  GPL license; see file COPYING for details.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "hashtab.h"

#include "myer.h"

static SRC_t **CurSIDs;
static DECL_t **CurUIDs, **CurFakeUIDs;
static int   NextAllUID;


/* ----------------------------------------------------------------------------
   Transfer all references to decl "one" into refs to decl "global".
*/
static void
Decl_Merge( DECL_t *global, DECL_t *one )
{
	for (CHAIN_t *rc = one->refs.next; rc != &one->refs; rc = rc->next) {
		CHAIN_t *gc = InsertChain( &global->refs, rc->spot );
		if (one->def == rc) {
			if (global->def && global->def != one->def) {
				/* Multiply defined */
				global->defsid = -1;
			} else {
				global->def = gc;
			}
		}
		rc->spot->ref.decl = global;
	}
	for (CHAIN_t *rc = one->decls.next; rc != &one->decls; rc = rc->next) {
		InsertChain( &global->decls, (void *) rc->sid );
	}

	if (one->wasmerged) {
		/* Duplicate decl was erroneously merged into master list.
		   Get rid of it. */
		one->deleted = 1;
	}
} /* Decl_Merge */


/* ----------------------------------------------------------------------------
   Transfer a DECL from the current file to the master.
*/
static void
Transfer_UID( DECL_t *d )
{
	TOKEN_HASH_t *th = Token_Hash_add( AllHash, d->name, 0, INSERT );

	d->name = th->key;
	if (d->uid < 0) {
		CurFakeUIDs[-d->uid] = 0;
	} else {
		CurUIDs[d->uid] = 0;
	}
	d->uid = ++NextAllUID;
	AddUID( &AllUIDs, d->uid, d );
	d->wasmerged = 1;
} /* Translate_UID */


/* ----------------------------------------------------------------------------
   Transfer a chain of SPOT_t from current file to master.
*/
static void
Transfer_Spots( CHAIN_t *sc )
{
	SPOT_t *s;

	while ((s = sc->spot)) {
		switch (s->kind) {
		case SPOT_Skip:
			/* Has no pointer to be updated */
			break;
		case SPOT_IncludeFile:
			s->ref.sid = CurSIDs[s->ref.sid]->sid;
			break;
		default:
			if (!s->ref.decl->wasmerged) {
				Transfer_UID( s->ref.decl );
			}
		}
		sc = sc->next;
	}
} /* Transfer_Spots */


/* ----------------------------------------------------------------------------
   Moves a spot to a different chain or different position in its chain.  If
   this was the defining spot for its decl, updates the backlink.
*/
static void
MoveSpot( CHAIN_t *old, CHAIN_t *new )
{
	SPOT_t  *x = UnlinkChain( old );
	CHAIN_t *y = InsertChain( new, x );

	if (x->kind != SPOT_IncludeFile
	    && x->ref.decl
	    && x->ref.decl->def == old) {
		x->ref.decl->def = y;
	}
} /* MoveSpot */


/* ----------------------------------------------------------------------------
   Match up a declaration in a compilation unit with one in the master.  If
   the decl has the same definition, merge it. If the master has several
   coextensive spots, the decl may match a later one.

   Result is the master spot where the decl matched (usually same as
   `master' argument, or 0 for no match.
*/
static CHAIN_t *
Match_Decls( CHAIN_t *master, CHAIN_t *one )
{
	SPOT_t *m = master->spot;
	SPOT_t *o = one->spot;
	DECL_t *md = m->ref.decl;
	DECL_t *od = o->ref.decl;

	if (od->def == md->def) {
		/* Already merged */
		return master;
	}

	if (od->def && md->def &&
	    od->defsid == md->defsid &&
	    od->def->spot->line == md->def->spot->line &&
	    od->def->spot->col == md->def->spot->col) {
		/* The decls are defined at same spot in same file */
		Decl_Merge( md, od );
		return master;
	}

	if (od->isglobal && md->isglobal
	    && !strcmp( od->name, md->name )) {
		/* Ref to like-named globals */
		if (!od->def && !md->def) {
			/* Never-defined globals.  Might as well merge them. */
			Decl_Merge( md, od );
			return master;
		}
		if (m->kind == SPOT_Ref && o->kind == SPOT_Decl
		    && md->def
		    && md->def->spot->kind == SPOT_Def) {
			/* Ref to a blind struct not defined in this file,
			   but some other file does have a def for it */
			o->kind = SPOT_Ref;
			Decl_Merge( md, od );
			return master;
		}
		if (m->kind == SPOT_Decl && o->kind == SPOT_Ref
		    && od->def
		    && od->def->spot->kind == SPOT_Def) {
			/* Ref to a defined struct that was blind in all
			   previous occurrences of this file. */
			Transfer_UID( od );
			Decl_Merge( od, md );
			m->kind = SPOT_Ref;
			return master;
		}
	}

	/* Maybe this decl matches some later coextensive spot */
	SPOT_t *try = master->next->spot;
	if (try &&
	    try->line == m->line &&
	    try->col  == m->col &&
	    try->endline == m->endline &&
	    try->endcol  == m->endcol &&
	    Match_Decls( master->next, one )) {
		/* Bubble-sort to move matching spot to proper place in chain */
		MoveSpot( master->next, master );
		return master->prev;
	}

	/* No match */
	return 0;
} /* Match_Decls */


/* ----------------------------------------------------------------------------
   A file in a compilation unit is already in the master list.  Merge the
   information from the two compilations.
*/
static void
myerMerge_File( SRC_t *master, SRC_t *one )
{
	CHAIN_t *match, *mc = master->spots.next, *oc = one->spots.next;
	int end_macrodef = 0;
	SPOT_t *prev = 0;
	static const SPOT_t Empty = {
		SPOT_Skip, 0x7fffffff, 0, 0x7fffffff, 0, 0.0, 0.0, {.sid = 0}
	};

	while (mc != &master->spots || oc != &one->spots) {
		SPOT_t *m = mc->spot, *o = oc->spot;

		if (!m) {
			m = (SPOT_t *) &Empty;
		}
		if (!o) {
			o = (SPOT_t *) &Empty;
		}

		match = mc;
		if (o->kind == SPOT_IncludeFile) {
			if (prev != o) {
				/* Update the SID, but only once.  The same
				   o might be seen several times if we're
				   skipping over MASTER items not mentioned
				   in the ONE file */
				o->ref.sid = CurSIDs[o->ref.sid]->sid;
				prev = o;
			}
		} else if (o->kind != SPOT_Skip
			   && ((m->kind == o->kind)
			       || (m->kind==SPOT_Ref && o->kind==SPOT_Decl)
			       || (o->kind==SPOT_Ref && m->kind==SPOT_Decl))) {
			match = Match_Decls( mc, oc );
			if (match) {
				/* Master spot might have been bubble-sorted */
				mc = match;
			}
		}

		if (m->kind       != o->kind
		    || m->line    != o->line
		    || m->col     != o->col
		    || m->endline != o->endline
		    || m->endcol  != o->endcol
		    || !match) {
			if (m->kind == SPOT_IncludeFile &&
			    m->endline < o->line) {
				/* Included first time, not second */
				mc = mc->next;
				continue;
			}
			if (o->endline < end_macrodef) {
				/* This cu has a mention for an identifier
				   not seen in another cu */
				if (o->kind != SPOT_IncludeFile
				    && o->kind != SPOT_Skip
				    && !o->ref.decl->wasmerged) {
					Transfer_UID( o->ref.decl );
				}
				oc = oc->next;
				MoveSpot( oc->prev, mc );
				continue;
			}
			if (m->endline < end_macrodef) {
				/* This cu fails to have a mention for an
				   identifier seen in other cu's. */
				mc = mc->next;
				continue;
			}
			if (m->kind == SPOT_Skip && m->line < o->line) {
				/* Text was skipped on previous reading */
				end_macrodef = m->endline;
				mc = mc->next;
				free( UnlinkChain(mc->prev) );
				continue;
			}
			if (o->kind == SPOT_Skip && o->line < m->line) {
				/* Text was skipped on following reading */
				end_macrodef = o->endline;
				oc = oc->next;
				continue;
			}
			goto Problem_File;
		}

		if (m->kind == SPOT_MacroDef) {
			end_macrodef = m->endline;
		} else if (m->line > end_macrodef) {
			end_macrodef = 0;
		}

		/* The spots match */
		mc = mc->next;
		oc = oc->next;
	}

	if (mc == &master->spots && oc == &one->spots) {
		return; /* Successful merge */
	}

 Problem_File: /* Didn't compile the same way in all compilation units */
	master->baseline = INCONSISTENT_CONTENTS; /* Prohibit HTML display of this file */

	/* Transfer the remainder of the current file as a concatenation to
	   the master. */
	Transfer_Spots( oc );
	while (one->spots.next != oc) {
		if (one->spots.next->spot->kind != SPOT_IncludeFile &&
		    one->spots.next->spot->ref.decl &&
		    one->spots.next->spot->ref.decl->def == one->spots.next) {
			/* Deleting the def-spot.  If this comes up later,
			   just pretend it was never defined. */
			one->spots.next->spot->ref.decl->def = 0;
		}
		/* FIXME: Should free(UnlinkChain(one->spots.next)) but this
		   causes a segfault when analyzing gcc.  Don't want to
		   chase it down right now. */
		UnlinkChain( one->spots.next );
	}
	master->spots.prev->next = one->spots.next;
	one->spots.next->prev    = master->spots.prev;
	one->spots.prev->next    = &master->spots;
	master->spots.prev       = one->spots.prev;
	InitChain( &one->spots );
} /* myerMerge_File */


/* ----------------------------------------------------------------------------
   Merge one compilation unit into the master list.
*/
static void
myerMerge( COMPILATION_UNIT_t cu )
{
	TOKEN_HASH_t *th;
	SRC_t *src;
	int j, sid;

	CurSIDs = cu.sids;
	CurUIDs = cu.uids;
	CurFakeUIDs = cu.fake_uids;

	/* Translate all source-file ID numbers. Hash the filenames ti find
	   duplicates.  FIXME: This is inconsistent for the dev/ino
	   duplicate-detection method used elsewhere in Myer. */
	for (j = 1; j < NUM_IDS(CurSIDs); ++j) {
		src = CurSIDs[j];
		if (!src) {
			continue;
		}
		th = Token_Hash_add( AllHash,src->name,TOKEN_FILEPUSH,INSERT );
		src->name = th->key;
		if (!th->data) {
			* (int *) &th->data = ++NextAllSID;
		}
		src->sid = * (int *) &th->data;
	}

	/* For all decls, translate SIDs and merge "public" things */
	for (int j = -NUM_IDS(cu.fake_uids)+1; j < NUM_IDS(cu.uids); ++j) {
		DECL_t *d;
		if (j > 0) {
			d = CurUIDs[j];
		} else if (j < 0) {
			d = CurFakeUIDs[-j];
		} else {
			d = 0;
		}
		if (!d || d->uid != j) {
			/* Skip unused or duplicate UID slots */
			continue;
		}
		if (d->defsid) {
			d->defsid = CurSIDs[d->defsid]->sid;
		}
		for (CHAIN_t *ic=d->decls.next; ic!=&d->decls; ic=ic->next) {
			ic->sid = CurSIDs[ic->sid]->sid;
		}

		if (d->ispublic) {
			th = Token_Hash_add( AllHash,d->name,d->class,INSERT );
			if (th->data) {
				Decl_Merge( th->data, d );
			} else { /* New public identifier */
				Transfer_UID( d );
				th->data = d;
			}
		}
	}

	/* Process each source file */
	for (int j = 1; j < NUM_IDS(CurSIDs); ++j) {
		src = CurSIDs[j];
		if (!src) {
			continue;
		}
		sid = src->sid;
		if (sid < NUM_IDS(AllSIDs) && AllSIDs[sid]) {
			src->baseline = 0;
			myerMerge_File( AllSIDs[sid], src );
		} else { /* Transfer this source file */
			AddSID( &AllSIDs, src );
			src->baseline = -1;
			Transfer_Spots( src->spots.next );
		}
	}

	for (int j = 1; j < NUM_IDS(CurSIDs); ++j) {
		if (CurSIDs[j] && CurSIDs[j]->baseline) {
			/* This one was transfered to the merged cu, so
			   don't free it when the cu is freed. */
			CurSIDs[j] = 0;
		}
	}
} /* myerMerge */



/* ----------------------------------------------------------------------------
   API for phase 3.
*/
int
myerMerge_From_File( char **fnamep )
{
	COMPILATION_UNIT_t cu;
	char *filename = *fnamep;
	char *extension = strrchr( filename, '.' );

	if (extension && !strcmp( extension, ".myer3" )) {
		assert( !AllHash );
		if (Verbose) {
			fprintf( stderr, "[3] < %s\n", filename );
		}
		cu = myerToken_Read( filename, EOF );
		AllSIDs = cu.sids;
		AllUIDs = cu.uids;
		AllHash = cu.names;
		NextAllSID = NUM_IDS(AllSIDs);
		assert( !cu.fake_uids );
		return 3;
	} else {
		if (!AllHash) { /* First call */
			AllHash = Token_Hash_init();
		}
		cu = myerToken_From_File( fnamep );
		if (!cu.names) {
			/* File has been skipped */
			return 2;
		}
		if (Verbose) {
			fprintf( stderr, "[3] = %s\n", *fnamep );
		}
		myerMerge( cu );
		Common_Free( cu );
		return 2;
	}
} /* myerMerge_From_File */
