/* myercalc.c - Calculate raw coupling & cohesion values -*- coding: latin-1 -*-

   2003 by Jonathan Yavner.  GPL license; see file COPYING for details.
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>

#include "myer.h"


/* ----------------------------------------------------------------------------
   Produce coupling & cohesion values for one file.
*/
void
myerCalc_Process( SRC_t *src )
{
	CHAIN_t *fc  = &src->funcs;
	FUNC_t *func = fc->func;
	INTBLOCK_t *blockp;
	int fdeclrefs, sdeclrefs, uid, target, numinclude = I_MAX;
	int endline = 0x7fffffff, endcol = 0x7fffffff;

	if (Verbose) {
		fprintf( stderr, "[5] = %s\n", src->name );
	}
	for (CHAIN_t *sc = src->spots.next; sc != &src->spots; sc = sc->next) {
		SPOT_t *s = sc->spot;
		if (s->line > endline
		    || (s->line == endline && s->col > endcol)) {
			/* End of function */
			func = src->funcs.func;
			endline = 0x7fffffff;
			endcol  = 0x7fffffff;
		}
		switch (s->kind) {
		case SPOT_Skip:
			s->coupling = COUPLING_IFDEF;
			break;
		case SPOT_IncludeFile:
			if (!src->inclrefs[numinclude]) {
				/* Useless unreferenced include */
				s->coupling = COUPLING_MODLOCAL;
			} else {
				s->coupling = 1.0 / src->inclrefs[numinclude];
			}
			s->cohesion = 1.0 / AllSIDs[s->ref.sid]->numrefs;
			++numinclude;
			break;
		case SPOT_FunDef:
			fc = fc->next;
			func = fc->func;
			assert( func->uid == s->ref.decl->uid );
			endline = s->endline;
			endcol  = s->endcol;
			/* Could do something cute here, such as mark the
			   fundef with a color suggesting how other
			   functions couple to it. */
			break;
		case SPOT_Def:
		case SPOT_Decl:
			s->kind = SPOT_Ref;
			/* Fall through */
		case SPOT_MacroRef:
		case SPOT_MacroDef:
		case SPOT_ConstRef:
		case SPOT_Ref:
			uid = s->ref.decl->uid;
			blockp = src->incldecls[uid>>8];
			if (!blockp || (target = (*blockp)[uid&255]) < 0) {
				/* Local variable */
				s->coupling = COUPLING_FUNLOCAL;
				break;
			}
			/* If not a local, it had better be a global! */
			assert( AllUIDs[uid]->isglobal );
			sdeclrefs = (*src->declrefs[uid>>8])[uid&255];
			fdeclrefs = (*func->declrefs[uid>>8])[uid&255];
			s->cohesion =
				  (ScaleCohesion.func  /fdeclrefs)
				+ (ScaleCohesion.file  /sdeclrefs)
				+ (ScaleCohesion.global/AllUIDs[uid]->numrefs);
			if (target == I_LOCAL && AllUIDs[uid]->numfiles == 1) {
				/* Coupling doesn't apply */
				s->coupling = COUPLING_MODLOCAL;
				break;
			}
			s->coupling =
				  (ScaleCoupling.func  /func->inclrefs[target])
				+ (ScaleCoupling.file  /src->inclrefs[target])
				+ (ScaleCoupling.global/AllSIDs[src->inclsids[target]]->numrefs);
			/* Following line is to give the debugger someplace
			   to land; otherwise the break will be optimized
			   into a continue */
			target = 0;
			break;
		}
	}
} /* myerCalc_Process */


/* ----------------------------------------------------------------------------
   Produce phase-5 intermediate file.
*/
void
myerCalc_To_File()
{
	printf( "%c myer5 * %s  -*- tab-width: 12 -*-\n",
		TOKEN_COMMENT, My_Ctime() );

	for (int sid = 1; sid < NUM_IDS(WantedSIDs); ++sid) {
		SRC_t *src = WantedSIDs[sid];
		if (Verbose) {
			fprintf( stderr, "[5] > %s\n", src->name );
		}
		printf( "\n%c %s\n", TOKEN_NEWFILE, src->name );
		for (CHAIN_t *sc=src->spots.next;sc!=&src->spots;sc=sc->next) {
			SPOT_t *s = sc->spot;
			printf( "%c %d,%d\t%d,%d", s->kind, s->line, s->col,
				s->endline, s->endcol );
			if (s->coupling == 0.0
			    && s->cohesion == 0.0) {
				printf( "\t       0,0" );
			} else if (s->coupling < 0.0) {
				printf( "\t      %d,%.6f",
					(int) s->coupling, s->cohesion );
			} else {
				printf( "\t%.6f,%.6f",
					s->coupling, s->cohesion );
			}
			if (s->kind == SPOT_MacroDef) {
				printf( "\t%s", s->ref.decl->name );
			}
			putchar( '\n' );
		}
	}
} /* myerCalc_To_File */


/* ----------------------------------------------------------------------------
   API for phase 5.
*/
int
myerCalc_From_File( char **fnamep )
{
	char *filename = *fnamep;
	char *extension = strrchr( filename, '.' );

	if (extension && !strcmp( extension, ".myer5" )) {
		assert( !"No input from .myer5 files" );
		return 5;
	} else {
		return myerSum_From_File( fnamep );
	}
} /* myerCalc_From_File */
