// muesli-syx.c -*- C -*-
/* Interface to syx evaluators / template for new language interfaces
   Copyright (C) 2008, 2009, 2010 University of Limerick

   This file is part of Muesli.
   
   Muesli is free software: you can redistribute it and/or modify it under
   the terms of the GNU General Public License as published by the Free
   Software Foundation, either version 3 of the License, or (at your
   option) any later version.
   
   Muesli is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.
   
   You should have received a copy of the GNU General Public License
   along with Muesli.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef _MUESLI_SYX_
#define _MUESLI_SYX_

#include "../muesli-config.h"
#include "muesli.h"
#include "muesli-internal.h"

#ifdef HAVE_LIBSYX

#include <stdio.h>
#include <limits.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <getopt.h>		// todo: do I need this?

#include <syx/syx.h>

static int ambient_transient = 0;

static evaluator_interface *syx_interface = NULL;

static char *option_names_var_name = (char*)"option_names";

static void
syx_set_parameter()
{
#if 0
  if (syx_count_args(cs) < 2) {
    fprintf(stderr, "too few args to syx_set_parameter\n");
    syx_error(cs, (char*)"argcount", (char*)"set_parameter");
  }

  char option_code = muesli_find_option_letter(syx_interface->getopt_options, syx_string_arg(cs, 0));

  if (option_code != -1) {
    if (syx_arg_is_string(cs, 1)) {
      (syx_interface->handle_option)(syx_interface->app_params,
					option_code,
					muesli_malloc_copy_string(syx_string_arg(cs, 1)),
					0.0, 1,
					"syx");
    } else if (syx_arg_is_number(cs, 1)) {
      (syx_interface->handle_option)(syx_interface->app_params,
					option_code, NULL, syx_number_arg(cs, 2), 1,
					"syx");
    } else {
      (syx_interface->handle_option)(syx_interface->app_params,
					option_code, (char*)"true", 0.0, 1,
					"syx");
    }
  }
#endif
}

static void
syx_set_parameters()
{
#if 0
  if (syx_count_args(cs) < 1)  {
    fprintf(stderr, "too few args to syx_set_parameters\n");
    syx_error(cs, (char*)"argcount", (char*)"set_parameters");
  }

#if 0
  // Fill in: use a table iterator from your language
  syx_table table = syx_table_arg(cs, 0);

  syx_table_iteration_start(cs, table);
  while (syx_table_iteration_next(cs, table) != 0) {
    syx_set_parameter(cs,
			 syx_table_iteration_current_key(cs, table),
			 syx_table_iteration_current_value(cs, table));
  }

#endif
#endif
}

static void
syx_get_parameter()
{
#if 0
  if (syx_count_args(cs) < 1)  {
    fprintf(stderr, "too few args to syx_get_parameter\n");
    syx_error(cs, (char*)"argcount", (char*)"get_parameter");
  }

  char option_code = muesli_find_option_letter(syx_interface->getopt_options,
					       syx_string_arg(cs, 0));

  if (option_code != -1) {

muesli_value_t result = (syx_interface->handle_option)(syx_interface->app_params,
							  option_code,	// option
							  NULL, 0.0,	// value
							  0,	// set
							  "syx");

    switch (result.type) {
    case muesli_value_const_string:
      syx_return_string(cs, result.data.as_const_string);
      break;
    case muesli_value_float:
      syx_return_number(cs, result.data.as_float);
      break;
    case muesli_value_integer:
      syx_return_integer(cs, result.data.as_int);
      break;
    case muesli_value_boolean:
      syx_return_boolean(cs, result.data.as_int);
      break;
    default:
      syx_return_boolean(cs, 0);
      break;
    }
  }
#endif
}

static void
syx_get_parameters()
{
#if 0
  syx_table *table = syx_create_table(cs, 48);

  // todo: fix and re-instate -- I have to get long_options across to it somehow
  struct option *option_names = syx_interface->getopt_options;

  while (option_names->name != 0) {
    
    muesli_value_t result = (syx_interface->handle_option)(syx_interface->app_params,
							      (option_names->val), // opt
							      NULL, 0.0, // value
							      0, // set
							      "syx");

    switch (result.type) {
    case muesli_value_const_string:
      syx_set_table_string(cs, table, (char*)(option_names->name), result.data.as_const_string);
      break;
    case muesli_value_float:
      syx_set_table_number(cs, table, (char*)(option_names->name), result.data.as_float);
      break;
    case muesli_value_boolean:
      syx_set_table_boolean(cs, table, (char*)(option_names->name), result.data.as_int);
      break;
    default:
      syx_set_table_boolean(cs, table, (char*)(option_names->name), 0);
      break;
    }
    option_names++;
  }
#endif
}

///////////////////////////////////////
// Call arbitrary evaluators by name //
///////////////////////////////////////

static int
muesli_syx_eval_in_language()
{
#if 0
  if (syx_count_args(cs) < 2) {
    fprintf(stderr, "too few args to syx_eval_in_language\n");
    syx_error(cs, (char*)"argcount", (char*)"eval_in_language");
    return 0;
  }

  if ((syx_arg_is_string(cs, 1)) && (syx_arg_is_string(cs, 2))) {
    const char *language_name = syx_string_arg(cs, 1);
    const char *evaluand = syx_string_arg(cs, 2);
    unsigned int evaluand_length = strlen(evaluand);

    fprintf(stderr, "In syx_eval_in_language(\"%s\", \"%s\")\n", language_name, evaluand);

    muesli_value_t result = muesli_eval_in_language(language_name,
						    evaluand,
						    evaluand_length,
						    ambient_transient);

    switch (result.type) {
    case muesli_value_float:
      syx_return_number(cs, result.data.as_float);
      break;
    case muesli_value_integer:
      syx_return_integer(cs, result.data.as_int);
      break;
    case muesli_value_const_string:
      syx_return_string(cs, result.data.as_const_string);
      break;
    case muesli_value_boolean:
      syx_return_boolean(cs, result.data.as_bool);
      break;
    case muesli_value_none:
    case muesli_value_error:
      syx_return_boolean(cs, 0);
      break;
    }
  } else {
    fprintf(stderr, "wrong type args to syx_eval_in_language\n");
    syx_error(cs, (char*)"argtype", (char*)"eval_in_language");
    return 0;
  }
  return 1;
#endif
}

///////////////////////////////
// Custom built-in functions //
///////////////////////////////

static void
syx_eval_custom_function()
{
#if 0
  if (syx_arg_is_string(cs, 0)) {
    char *string_arg = syx_string_arg(cs, 0);
    muesli_value_t result = (syx_interface->eval_string)(syx_interface,
							    string_arg, strlen(string_arg),
							    ambient_transient);
    switch (result.type) {
    case muesli_value_float:
      syx_return_number(cs, result.data.as_float);
      break;
    case muesli_value_integer:
      syx_return_integer(cs, result.data.as_int);
      break;
    case muesli_value_const_string:
      syx_return_string(cs, result.data.as_const_string);
      break;
    case muesli_value_boolean:
      syx_return_boolean(cs, result.data.as_int);
      break;
    case muesli_value_none:
    case muesli_value_error:
      syx_return_void(cs);
      break;
    }
  } else {
    fprintf(stderr, "syx_eval_function must be given a string\n");
    syx_error(cs, (char*)"argtype", (char*)"syx_eval");
  }
#endif
}

////////////////
// Initialize //
////////////////

void
syx_evaluator_init(evaluator_interface *interface)
{
  syx_interface = interface;

  /* initialize Syx */
  syx_init(interface->app_argc, interface->app_argv, NULL);
  
  /* load the default image */
  syx_memory_load_image(NULL);

#if 0
  // Extend system

  // Fill in: register all these language extensions (as applicable):
  if (muesli_extensions & MUESLI_EXTENSION_SINGLE_PARAMETER) {
    Muesli_Add_Fn_1(interface, (char*)"get_parameter", syx_get_parameter);
    Muesli_Add_Fn_2(interface, (char*)"set_parameter", syx_set_parameter);
  }
  if (muesli_extensions & MUESLI_EXTENSION_PARAMETER_BLOCK) {
    Muesli_Add_Fn_0(interface, (char*)"parameters", syx_get_parameters);
    Muesli_Add_Fn_1(interface, (char*)"modify_parameters", syx_set_parameters);
  }
  if (muesli_extensions & MUESLI_EXTENSION_EVAL_IN_LANGUAGE) {
    Muesli_Add_Fn_1(interface, (char*)"eval_in_language", muesli_syx_eval_in_language);
  }
  if (muesli_extensions & MUESLI_EXTENSION_CUSTOM) {
    Muesli_Add_Fn_1(interface, (char*)"custom", syx_eval_custom_function);
  }
#endif

#if 0
  // Set up option names

  // Fill in: Create a table of option names.  You may well not need to bother.
  syx_table *option_names_table = syx_create_table(our_syx_state, 48);
  struct option *option_names;
  for (option_names = syx_interface->getopt_options;
       (option_names != NULL) && (option_names->name != 0);
       option_names++) {
    syx_set_table_number(our_syx_state, option_names_table,
			 (char*)(option_names->name), (option_names->val));
  }
  syx_set_global_variable_table(our_syx_state,
				syx_make_symbol(our_syx_state,
						option_names_var_name),
				option_names_table);
#endif
}

static void
syx_evaluator_close(evaluator_interface *interface)
{
  syx_quit();
}

static void
syx_load_file(evaluator_interface *interface,
		const char *filename)
{
  int muesli_flags = interface->flags;
  Muesli_Enter_Evaluator(interface, 0);
  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loading %s\n", filename);
  }
  // Fill in: load the functions file given as (char*)(filename)

  /* now file in class and method declarations from our ST file */
  syx_file_in_blocking(filename);

  if (muesli_flags & TRACE_MUESLI_LOAD) {
    fprintf(stderr, "Loaded %s\n", filename);
  }
  Muesli_Leave_Evaluator();
}

static muesli_value_t
syx_eval_string(evaluator_interface *evaluator,
		   const char *syx_c_string,
		   unsigned int string_length,
		   int transient)
{
  muesli_value_t result;
  Muesli_Enter_Evaluator(evaluator, transient);
  ANULL_VALUE(result);

  if (syx_c_string) {
    int old_ambient_transient = ambient_transient;
    // fprintf(stderr, "Syx evaluating string \"%s\"\n", syx_c_string);

    /* based on code from http://code.google.com/p/syx/wiki/ExampleEmbeddingLua */

    SyxOop instance;
    SyxOop context;
    SyxOop process;
    SyxOop result;

    ambient_transient = transient;
  
    /* create a Process */
    process = syx_process_new();

#if 0

    /* todo: parse the string to get an object and a message */

    /* create a function instance */
    instance = syx_object_new(syx_globals_at("Sum"));

    /* create a MethodContext which sends the #with:and: message */
    context = syx_send_message(instance,                    // the receiver
			       "with:and:",                 // the selector
			       2,                           // the number of arguments
			       syx_small_integer_new(41),   // first argument
			       syx_small_integer_new(22));
#endif
    /* enter the context in the created process */
    syx_interp_enter_context(process, context);

    /* execute the process in blocking mode */
    syx_process_execute_blocking(process);

    /* fetch the last returned object (an instance variable of Process) */
    result = SYX_PROCESS_RETURNED_OBJECT(process);

    printf("The result is %d\n", SYX_SMALL_INTEGER(result));

    ambient_transient = old_ambient_transient;
  }

  Muesli_Leave_Evaluator();
  return result;
}

void
syx_setup(evaluator_interface *new_syx_interface)
{
  syx_interface = new_syx_interface;

  syx_interface->eval_string = syx_eval_string;
  syx_interface->load_file = syx_load_file;

  syx_interface->close_evaluator = syx_evaluator_close;

  syx_interface->version = SYX_VERSION;
}

#endif
#endif

