# Pure Ruby implementation of uncompface.
#
# Copyright (c) 2003 Daiki Ueno
#
# Original license notice follows:
#
# Copyright (c) James Ashton - Sydney University - June 1990.
#
# Permission is given to distribute these sources, as long as the
# copyright messages are not removed, and no monies are exchanged.
#
# No responsibility is taken for any errors on inaccuracies inherent
# either to the comments or the code of this program, but if reported
# to me, then an attempt will be made to fix them.

module Compface
  class Bitmap
    def initialize(bytes, length)
      @bytes = bytes
      @length = length
    end
    attr_reader :bytes, :length

    def [](i)
      bytes[i / 8] & (0x80 >> i % 8) == 0 ? 0 : 1
    end

    def []=(i, v)
      if v == 0
        bytes[i / 8] &= ~(0x80 >> i % 8)
      else
        bytes[i / 8] |= 0x80 >> i % 8
      end
    end
  end

  Guess00 = Bitmap.new(<<EOF, 1 << 12)
\x00\x00\x01\x01\x00\x00\xE3\xDF\x05\x17\x05\x0F\x00\x1B\x0F\xDF\x00\x04\x00\
\x00\x0D\x0F\x03\x7F\x00\x00\x00\x01\x00\x1D\x45\x2F\x00\x00\x00\x0D\x00\x0A\
\xFF\xFF\x00\x04\x00\x05\x01\x3F\xCF\xFF\x10\x01\x80\xC9\x0F\x0F\xFF\xFF\x00\
\x00\x00\x00\x1B\x1F\xFF\xFF\x4F\x54\x07\x1F\x57\x47\xD7\x3D\xFF\xFF\x5F\x1F\
\x7F\xFF\x7F\x7F\x05\x0F\x01\x0F\x0F\x5F\x9B\xDF\x7F\xFF\x5F\x1D\x5F\xFF\x0F\
\x1F\x0F\x5F\x03\x1F\x4F\x5F\xF7\x7F\x7F\xFF\x0D\x0F\xFB\xFF\xF7\xBF\x0F\x4F\
\xD7\x3F\x4F\x7F\xFF\xFF\x67\xBF\x56\x25\x1F\x7F\x9F\xFF\x00\x00\x00\x05\x5F\
\x7F\x01\xDF\x14\x00\x05\x0F\x07\xA2\x09\x0F\x00\x00\x00\x00\x0F\x5F\x18\xD7\
\x94\x71\x00\x05\x1F\xB7\x0C\x07\x0F\x0F\x00\x0F\x0F\x1F\x84\x8F\x05\x15\x05\
\x0F\x4F\xFF\x87\xDF\x05\x01\x10\x00\x0F\x0F\x00\x08\x05\x04\x04\x01\x4F\xFF\
\x9F\x8F\x4A\x40\x5F\x5F\xFF\xFE\xDF\xFF\x7F\xF7\xFF\x7F\xFF\xFF\x7B\xFF\x0F\
\xFD\xD7\x5F\x4F\x7F\x7F\xDF\xFF\xFF\xFF\xFF\xFF\x77\xDF\x7F\x4F\xEF\xFF\xFF\
\x77\xFF\xFF\xFF\x6F\xFF\x0F\x4F\xFF\xFF\x9D\xFF\x0F\xEF\xFF\xDF\x6F\xFF\xFF\
\xFF\x4F\xFF\xCD\x0F\x4F\xFF\xFF\xDF\x00\x00\x00\x0B\x05\x02\x02\x0F\x04\x00\
\x00\x0C\x01\x06\x00\x0F\x20\x03\x00\x00\x05\x0F\x40\x08\x00\x00\x00\x01\x00\
\x01\x0C\x0F\x01\x00\x80\x00\x00\x00\x80\x00\x00\x14\x01\x05\x01\x15\xAF\x0F\
\x00\x01\x10\x00\x08\x00\x46\x0C\x20\x00\x88\x00\x0F\x15\xFF\xDF\x02\x00\x00\
\x0F\x7F\x5F\xDB\xFF\x4F\x3E\x05\x0F\x7F\xF7\x95\x4F\x0D\x0F\x01\x0F\x4F\x5F\
\x9F\xDF\x25\x0E\x0D\x0D\x4F\x7F\x8F\x0F\x0F\xFA\x04\x4F\x4F\xFF\xF7\x77\x47\
\xED\x05\x0F\xFF\xFF\xDF\xFF\x4F\x6F\xD8\x5F\x0F\x7F\xDF\x5F\x07\x0F\x94\x0D\
\x1F\xFF\xFF\xFF\x00\x02\x00\x03\x46\x57\x01\x0D\x01\x08\x01\x0F\x47\x6C\x0D\
\x0F\x02\x00\x00\x00\x0B\x4F\x00\x08\x05\x00\x95\x01\x0F\x7F\x0C\x0F\x01\x0E\
\x00\x00\x0F\x41\x00\x00\x04\x24\x0D\x0F\x0F\x7F\xCF\xDF\x00\x00\x00\x00\x04\
\x40\x00\x00\x06\x26\xCF\x05\xCF\x7F\xDF\xDF\x00\x00\x17\x5F\xFF\xFD\xFF\xFF\
\x46\x09\x4F\x5F\x7F\xFD\xDF\xFF\x0A\x88\xA7\x7F\x7F\xFF\xFF\xFF\x0F\x04\xDF\
\x7F\x4F\xFF\x9F\xFF\x0E\xE6\xDF\xFF\x7F\xFF\xFF\xFF\x0F\xEC\x8F\x4F\x7F\xFF\
\xDF\xFF\x0F\xCF\xDF\xFF\x6F\x7F\xFF\xFF\x03\x0C\x9D\x0F\x7F\xFF\xFF\xFF
EOF
  Guess01 = Bitmap.new(<<EOF, 1 << 7)
\x37\x73\x00\x19\x57\x7F\xF5\xFB\x70\x33\xF0\xF9\x7F\xFF\xFF\xFF
EOF
  Guess02 = Bitmap.new("\x50", 1 << 2)
  Guess10 = Bitmap.new(<<EOF, 1 << 9)
\x00\x00\x00\x00\x50\x00\xF3\x5F\x84\x04\x17\x9F\x04\x23\x05\xFF\x00\x00\x00\
\x02\x03\x03\x33\xD7\x05\x03\x5F\x3F\x17\x33\xFF\xFF\x00\x80\x02\x04\x12\x00\
\x11\x57\x05\x25\x05\x03\x35\xBF\x9F\xFF\x07\x6F\x20\x40\x17\x06\xFA\xE8\x01\
\x07\x1F\x9F\x1F\xFF\xFF\xFF
EOF
  Guess20 = Bitmap.new("\x04\x00\x01\x01\x43\x2E\xFF\x3F", 1 << 6)
  Guess30 = Bitmap.new(<<EOF, 1 << 8)
\x11\x11\x11\x11\x51\x11\x13\x11\x11\x11\x13\x11\x11\x11\x33\x11\x13\x11\x13\
\x13\x13\x13\x31\x31\x11\x01\x11\x11\x71\x11\x11\x75
EOF
  Guess40 = Bitmap.new(<<EOF, 1 << 10)
\x00\x0F\x00\x09\x00\x0D\x00\x0D\x00\x0F\x00\x4E\xE4\x0D\x10\x0F\x00\x0F\x44\
\x4F\x00\x1E\x0F\x0F\xAE\xAF\x45\x7F\xEF\xFF\x0F\xFF\x00\x09\x01\x11\x00\x01\
\x1C\xDD\x00\x15\x00\xFF\x00\x10\x00\xFD\x00\x0F\x4F\x5F\x3D\xFF\xFF\xFF\x4F\
\xFF\x1C\xFF\xDF\xFF\x8F\xFF\x00\x0D\x00\x00\x00\x15\x01\x07\x00\x01\x02\x1F\
\x01\x11\x05\x7F\x00\x1F\x41\x57\x1F\xFF\x05\x77\x0D\x5F\x4D\xFF\x4F\xFF\x0F\
\xFF\x00\x00\x02\x05\x00\x11\x05\x7D\x10\x15\x2F\xFF\x40\x50\x0D\xFD\x04\x0F\
\x07\x1F\x07\x7F\x0F\xBF\x0D\x7F\x0F\xFF\x4D\x7D\x0F\xFF
EOF
  Guess11 = Bitmap.new("\x01\x13\x03\x7F", 1 << 5)
  Guess21 = Bitmap.new("\x17", 1 << 3)
  Guess31 = Bitmap.new("\x55\x57\x57\x7F", 1 << 5)
  Guess41 = Bitmap.new("\x01\x01\x01\x1F\x03\x1F\x3F\xFF", 1 << 6)
  Guess12 = Bitmap.new("\x40", 1 << 1)
  Guess22 = Bitmap.new("\x00", 1 << 0)
  Guess32 = Bitmap.new("\x10", 1 << 2)
  Guess42 = Bitmap.new("\x10", 1 << 2)

  Levels = [[[1, 255], [251, 0], [4, 251]],
    [[1, 255], [200, 0], [55, 200]],
    [[33, 223], [159, 0], [64, 159]],
    [[131, 0], [0, 0], [125, 131]]]

  Freqs = [[0, 0], [38, 0], [38, 38], [13, 152],
    [38, 76], [13, 165], [13, 178], [6, 230],
    [38, 114], [13, 191], [13, 204], [6, 236],
    [13, 217], [6, 242], [5, 248], [3, 253]]

  def self.big_pop(p)
    @@b, @@r, = @@b / 256, @@b % 256
    range, offset, i = nil, nil, 0
    p.each {|range, offset|
      break unless @@r < offset || @@r >= range + offset
      i += 1
    }
    @@b *= range
    @@b += @@r - offset
    i
  end

  def self.pop_greys(f, o, w, h)
    if w > 3
      w, h = w / 2, h / 2
      pop_greys(f, o, w, h)
      pop_greys(f, o + w, w, h)
      pop_greys(f, o + h * 48, w, h)
      pop_greys(f, o + w + h * 48, w, h)
    else
      w = big_pop(Freqs)
      f[o] = 1 if w & 1 != 0
      f[o + 1] = 1 if w & 2 != 0
      f[o + 48] = 1 if w & 4 != 0
      f[o + 48 + 1] = 1 if w & 8 != 0
    end
  end

  def self.uncompress(f, o, w, h, l)
    case big_pop(Levels[l])
    when 2			# white
    when 0			# black
      pop_greys(f, o, w, h)
    else
      w, h, l = w / 2, h / 2, l.succ
      uncompress(f, o, w, h, l)
      uncompress(f, o + w, w, h, l)
      uncompress(f, o + h * 48, w, h, l)
      uncompress(f, o + w + h * 48, w, h, l)
    end
  end

  def self.gen(f)
    48.times {|j|
      48.times {|i|
        h = i + j * 48
        k = 0
        (i - 2 .. i + 2).each {|l|
          (j - 2 .. j).each {|m|
            next if l >= i && m == j
            if l > 0 && l <= 48 && m > 0
              k = f[l + m * 48] > 0 ? k * 2 + 1 : k * 2
            end
          }
        }
        case i
        when 1
          case j
          when 1
            f[h] ^= Guess22[k]
          when 2
            f[h] ^= Guess21[k]
          else
            f[h] ^= Guess20[k]
          end
        when 2
          case j
          when 1
            f[h] ^= Guess12[k]
          when 2
            f[h] ^= Guess11[k]
          else
            f[h] ^= Guess10[k]
          end
        when 48 - 1
          case j
          when 1
            f[h] ^= Guess42[k]
          when 2
            f[h] ^= Guess41[k]
          else
            f[h] ^= Guess40[k]
          end
        when 48
          case j
          when 1
            f[h] ^= Guess32[k]
          when 2
            f[h] ^= Guess31[k]
          else
            f[h] ^= Guess30[k]
          end
        else
          case j
          when 1
            f[h] ^= Guess02[k]
          when 2
            f[h] ^= Guess01[k]
          else
            f[h] ^= Guess00[k]
          end
        end
      }
    }
  end

  class Face < Bitmap
    def to_s
      result = ''
      24.times {|i|
        48.times {|j|
          k = i * 2 * 48 + j
          if self[k] == 0
            result << (self[k + 48] == 0 ? ' ' : ',')
          else
            result << (self[k + 48] == 0 ? '`' : ';')
          end
        }
        result << "\n"
      }
      result
    end
  end

  def self.uncompface(s)
    @@b = 0
    s.each_byte {|c|
      next if c < 33 || c > 126
      @@b *= 94; @@b += c - 33
    }
    pixels = 48 * 48
    f = Face.new("\x00" * (pixels / 8), pixels)
    uncompress(f, 0, 16, 16, 0)
    uncompress(f, 16, 16, 16, 0)
    uncompress(f, 32, 16, 16, 0)
    uncompress(f, 48 * 16, 16, 16, 0)
    uncompress(f, 48 * 16 + 16, 16, 16, 0)
    uncompress(f, 48 * 16 + 32, 16, 16, 0)
    uncompress(f, 48 * 32, 16, 16, 0)
    uncompress(f, 48 * 32 + 16, 16, 16, 0)
    uncompress(f, 48 * 32 + 32, 16, 16, 0)
    gen(f)
    f
  end
end
