# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'msgcab/webapp/controller'
require 'msgcab/database'
require 'msgcab/mailtree'
require 'msgcab/webapp/msgcab_view'
require 'enumerator'

class MsgCab::WebApp::MsgCabController < MsgCab::WebApp::Controller
  include MsgCab
  include MsgCab::WebApp
  include Logging

  FolderPat = %r|([a-z0-9-]+)|

  define_rule(:archive, /\A\z/)

  define_rule(:index,
              %r|\A#{FolderPat}/-\z|,
              :folder_name)

  define_rule(:recent_summary,
              %r|\A#{FolderPat}(?:/:(\d+))?\z|,
              :folder_name, :max)

  define_rule(:recent_summary_thread,
              %r|\A#{FolderPat}/thread(?:/:(\d+))?\z|,
              :folder_name, :max)

  define_rule(:number_summary,
              %r|\A#{FolderPat}/(\d+)-(\d+)\z|,
              :folder_name, :min, :max)

  define_rule(:number_summary_mbox,
              %r|\A#{FolderPat}/(\d+)-(\d+)\.txt\z|,
              :folder_name, :min, :max)

  define_rule(:number_summary_thread,
              %r|\A#{FolderPat}/thread/(\d+)-(\d+)\z|,
              :folder_name, :min, :max)

  define_rule(:entity,
              %r|\A#{FolderPat}/(\d+)(/\d+(?:\.\d+)+)?\z|,
              :folder_name, :folder_number, :cid)

  define_rule(:entity_mbox,
              %r|\A#{FolderPat}/(\d+)(/\d+(?:\.\d+)+)?\.txt\z|,
              :folder_name, :folder_number, :cid)

  define_rule(:threads,
              %r|\A#{FolderPat}/(\d+)\.threads\z|,
              :folder_name, :folder_number)

  define_rule(:js,
              %r|\A#{FolderPat}/(\d+)\.js|,
              :folder_name, :folder_number)

  define_rule(:data,
              %r|\A\.data(?:/([a-z]+))?/([^\\/]+\.[^\\/.]+)\z|,
              :suffix, :filename)

  private
  def thread_runs(threads)
    last_depth = -1
    runs = Array.new
    threads.each_with_index do |thread, i|
      run = runs[thread.depth] ||= []
      if last_depth < thread.depth
        runs[thread.depth] << [i, i]
      elsif !run.empty?
        runs[thread.depth][-1][1] = i
      else
        runs[thread.depth] << [-1, i]
      end
      last_depth = thread.depth
    end
    runs
  end

  TreeText = {
    'bar-v' => '|',
    'blank' => HTree::Text.parse_pcdata('&nbsp;'),
    'bar-t' => '+',
    'bar-l' => '`',
    'open0' => '-',
    'open1' => '+',
    'close' => '+',
    'bar-h' => '-'
  }

  def thread_icons(runs, threads, depth, index)
    icons = Array.new
    (1 ... depth).each do |i|
      run = runs[i].detect {|run| run[0] < index && index < run[1]}
      if run
        icons << 'bar-v'
      else
        icons << 'blank'
      end
    end
    if depth > 0
      run = runs[depth].detect {|run| run[0] <= index && index <= run[1]}
      if run[1] == index
        icons << 'bar-l'
      else
        icons << 'bar-t'
      end
    end
    next_thread = threads[index + 1]
    if depth == 0 && (!next_thread || next_thread.depth == 0)
      icons << 'open0'
    elsif next_thread && next_thread.depth > depth
      icons << 'open1'
    else
      icons << 'bar-h'
    end
    icons.collect {|icon| [icon, TreeText[icon]]}
  end

  def ellipsis(s, width)
    if s && s.length > width
      s = s[/.{0,#{width - 4}}/] + ' ...'
    end
    s
  end

  def decode_threads(threads, folder, encoding)
    runs = thread_runs(threads)
    threads.each_with_index do |thread, index|
      thread.xref.sort! {|a, b| a[0] <=> b[0]}
      cur = thread.xref.assoc(folder.name)
      if cur
        thread.xref.delete_at(thread.xref.index(cur))
        thread.xref.unshift(cur)
      end
      thread.nov = thread.nov.decode(encoding)
      thread.icons = thread_icons(runs, threads, thread.depth, index)
      thread.from = ellipsis(thread.nov.canonical_from,
                             Config['webapp', 'from_width'] || 20)
      thread.title = ellipsis(database.property(thread.number, 'title'),
                              Config['webapp', 'title_width'] || 30)
    end
  end

  def database
    @database ||= Database.instance
  end

  def mailtree
    @mailtree ||= MailTree.new
  end

  public
  def do_data(webapp, suffix, filename)
    filename.untaint
    if suffix
      data_path = Config.relative_path(Config['plugin_path'] || './plugin')
      data_path += suffix.untaint
    else
      data_path = Config.relative_path(Config['webapp', 'data_path'] || './data')
    end
    webapp.content_type = 'text/css' if filename =~ /\.css\z/
    webapp.send_resource(data_path + filename)
  end

  def do_archive(webapp)
    view = View.new(webapp)
    folders = database.folders
    folders.sort! {|a, b| a.name <=> b.name}
    view.attributes[:folders] = folders
    view.attributes[:banner] = "#{folders.length} folders"
    view.output_template('archive.html')
  end

  def ranges(folder)
    ranges = Array.new
    index_page_size = Config['webapp', 'index_page_size'] || 100
    (folder.min / index_page_size .. folder.max / index_page_size + 1).each_cons(2) do |cons|
      min = cons[0] * index_page_size + 1
      max = cons[1] * index_page_size
      unless database.numbers(folder.name, min, max).empty?
        ranges << [min, max]
      end
    end
    ranges
  end

  def recent_ranges(folder, limit)
    ranges = Array.new
    index_page_size = Config['webapp', 'index_page_size'] || 100
    max = folder.max
    while ranges.length < limit && folder.min < max
      base = (max - 1) / index_page_size * index_page_size 
      min = base + 1
      unless database.numbers(folder.name, min, max).empty?
        ranges << [min, max]
      end
      max = base
    end
    ranges
  end

  def do_index(webapp, folder_name)
    folder = database.folder(folder_name)
    raise InvalidRequest, "no such folder: #{folder_name}" unless folder

    view = View.new(webapp)
    index_columns = Config['webapp', 'index_columns'] || 5
    index_rows = Config['webapp', 'index_rows'] || 5
    ranges = recent_ranges(folder, index_columns * index_rows)
    rows = Array.new
    ranges.each_slice(index_columns) do |row|
      rows << row
    end

    num_recent = Config['webapp', 'num_recent'] || 100
    max_topics = Config['webapp', 'max_topics'] || 10
    topics = database.recent_topics(folder.name, num_recent)[0 .. max_topics]
    topics.each do |topic|
      topic.canonical_subject = topic.nov.decode(view.encoding).canonical_subject
    end

    view.attributes[:folder] = folder
    view.attributes[:banner] = folder.name
    view.attributes[:topics] = topics
    view.attributes[:rows] = rows
    view.load_template('_ranges.html')
    view.output_template('index.html')
  end

  def do_recent_summary(webapp, folder_name, max)
    folder = database.folder(folder_name)
    raise InvalidRequest, "no such folder: #{folder_name}" unless folder

    view = View.new(webapp)
    num_recent = max ? max.to_i : Config['webapp', 'num_recent'] || 100
    summary = database.recent_summary(folder.name, num_recent)
    summary.each do |message|
      message.nov = message.nov.decode(view.encoding)
    end

    view.attributes[:folder] = folder
    view.attributes[:banner] = "#{folder.name}:recent #{num_recent} messages"
    view.attributes[:summary] = summary
    view.output_template('summary.html')
  end

  def do_recent_summary_thread(webapp, folder_name, max)
    folder = database.folder(folder_name)
    raise InvalidRequest, "no such folder: #{folder_name}" unless folder

    view = View.new(webapp)
    num_recent = max ? max.to_i : Config['webapp', 'num_recent'] || 100
    view.attributes[:folder] = folder
    threads = database.recent_threads(folder.name, num_recent)
    view.attributes[:threads] = decode_threads(threads, folder, view.encoding)
    view.attributes[:banner] = "#{folder.name}:recent #{num_recent} messages (thread)"
    view.load_template('_threads.html')
    view.output_template('summary_thread.html')
  end

  def do_number_summary(webapp, folder_name, min, max)
    folder = database.folder(folder_name)
    raise InvalidRequest, "no such folder: #{folder_name}" unless folder

    view = View.new(webapp)
    min = min.to_i
    max = max.to_i
    summary = database.number_summary(folder.name, min, max)
    summary.each do |message|
      message.nov = message.nov.decode(view.encoding)
    end
    view.attributes[:min] = min
    view.attributes[:max] = max
    view.attributes[:folder] = folder
    view.attributes[:banner] = "#{folder.name}:#{min}-#{max}"
    view.attributes[:summary] = summary
    view.output_template('summary.html')
  end

  def do_number_summary_mbox(webapp, folder_name, min, max)
    folder = database.folder(folder_name)
    raise InvalidRequest, "no such folder: #{folder_name}" unless folder

    min = min.to_i
    max = max.to_i
    database.number_summary(folder.name, min, max).each do |message|
      webapp << Entity.parse(mailtree.fetch(message.number)).mbox
    end
  end

  def do_number_summary_thread(webapp, folder_name, min, max)
    folder = database.folder(folder_name)
    raise InvalidRequest, "no such folder: #{folder_name}" unless folder

    view = View.new(webapp)
    min = min.to_i
    max = max.to_i
    threads = database.number_threads(folder.name, min, max)
    view.attributes[:folder] = folder
    view.attributes[:threads] = decode_threads(threads, folder, view.encoding)
    view.attributes[:min] = min
    view.attributes[:max] = max
    view.attributes[:banner] = "#{folder.name}:#{min}-#{max} (thread)"
    view.load_template('_threads.html')
    view.output_template('summary_thread.html')
  end

  def do_entity(webapp, folder_name, folder_number, cid)
    folder = database.folder(folder_name)
    raise InvalidRequest, "no such folder: #{folder_name}" unless folder

    folder_number = folder_number.to_i
    number = database.from_folder_number(folder.name, folder_number)
    raise InvalidRequest, "no such message: #{folder.name}:#{folder_number}" unless number

    entity = Entity.parse(mailtree.fetch(number))
    entity = entity.find(cid) if cid

    view = EntityView.new(webapp, entity)
    unless view.body_visible?
      webapp.content_type = entity.content_type
      webapp << entity.decode
      return
    end

    references = Array.new
    database.nov(number).references.reverse_each do |reference|
      references.concat(database.numbers_by_msg_id(reference).collect {|number|
                          database.to_folder_number(number)
                        })
    end
    threads = database.one_threads(folder.name, folder_number)
    threads.clear if threads.length == 1
    decode_threads(threads, folder, view.encoding)
    view.attributes[:folder] = folder
    view.attributes[:number] = folder_number
    view.attributes[:references] = references
    view.attributes[:banner] = "%s:%05d" % [folder.name, folder_number]
    view.attributes[:threads] = threads
    view.load_template('_threads.html')
    view.output_template('entity.html')
  end

  def do_entity_mbox(webapp, folder_name, folder_number, cid)
    folder = database.folder(folder_name)
    raise InvalidRequest, "no such folder: #{folder_name}" unless folder

    folder_number = folder_number.to_i
    number = database.from_folder_number(folder.name, folder_number)
    raise InvalidRequest, "no such message: #{folder.name}:#{folder_number}" unless number

    webapp << Entity.parse(mailtree.fetch(number)).mbox
  end

  def do_threads(webapp, folder_name, folder_number)
    folder = database.folder(folder_name)
    raise InvalidRequest, "no such folder: #{folder_name}" unless folder
    folder_number = folder_number.to_i

    view = View.new(webapp)
    view.use_relative_uri = false
    threads = database.one_threads(folder.name, folder_number)
    decode_threads(threads, folder, view.encoding)
    view.attributes[:folder] = folder
    view.attributes[:threads] = threads
    view.load_template('_threads.html')
    view.output_template('threads.html')
  end

  def do_js(webapp, folder_name, folder_number)
    folder = database.folder(folder_name)
    raise InvalidRequest, "no such folder: #{folder_name}" unless folder
    folder_number = folder_number.to_i

    data_path = Config.absolute_path(Config['webapp', 'data_path'] || './data')
    webapp.content_type = 'text/javascript'
    (data_path + 'load_html.js').open do |file|
      webapp << file.read
    end
    div_id = "#{folder_name}:#{folder_number}"
    path_info = "/#{folder_name}/#{folder_number}.threads"
    webapp << <<"End"
document.write('<div id="#{folder_name}:#{folder_number}"></div>')
loadHTML("#{div_id}", "#{webapp.make_absolute_uri({:path_info => path_info})}")
End
  end
end
