# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

module MsgCab
  class Chunk
    def initialize(desc, data)
      @desc = desc
      @data = data
    end
    attr_reader :desc, :data
  end

  class Source
    include Enumerable

    def initialize(path)
      @path = path
    end
    attr_reader :path

    def self.open(path)
      if path.directory?
        DirSource.new(path)
      else
	path.open do |file|
	  line = file.gets
	  if line && line =~ /\AFrom /	# mbox file
	    MboxSource.new(path)
	  else
	    file.rewind
	    FileSource.new(path)
	  end
	end
      end
    end
  end

  class IOSource < Source
    def initialize(io)
      @io = io
    end

    def each
      yield Chunk.new("io:#{@io.fileno}", @io.read)
    end

    def size
      1
    end
  end

  class FileSource < Source
    def each
      yield Chunk.new("file:#{path.expand_path}", path.read)
    end

    def size
      1
    end
  end

  class MboxSource < Source
    def each
      path.open do |file|
        buf = ''
        count = 0
        loop do
          line = file.gets
          if !line
            strip_message(buf)
            yield Chunk.new("mbox:#{path.expand_path}:#{count}", buf)
            break
          elsif line =~ /\AFrom / && !buf.empty?
            strip_message(buf)
            yield Chunk.new("mbox:#{path.expand_path}:#{count}", buf)
            count += 1
            buf = ''
          else
            buf << line
          end
        end
      end
    end

    def size
      count = 0
      path.open do |file|
        loop do
          line = file.gets
          break unless line
          count += 1 if line =~ /\AFrom /
        end
      end
      count
    end

    private
    def strip_message(s)
      s.sub!(/(\r?\n)+\z/, '\1')
      s.gsub!(/^>+From/, 'From')
    end
  end

  class DirSource < Source
    def initialize(path, safe = true)
      super(path)
      @cur_path = (path + 'cur').expand_path
      @new_path = (path + 'new').expand_path
      @safe = safe
    end

    def each(&block)
      if @cur_path.directory? && @new_path.directory?
        each_cur(&block)
        each_new(&block)
      else
        each_num(&block)
      end
    end

    def size
      if @cur_path.directory? && @new_path.directory?
        @cur_path.entries.reject {|name| name.to_s =~ /\A\./}.length +
          @new_path.entries.reject {|name| name.to_s =~ /\A\./}.length
      else
        path.entries.reject {|name| name.to_s !~ /\A\d+\z/}.length
      end
    end

    private
    def each_num
      path.opendir do |dir|
        dir.each do |name|
          next unless name =~ /\A\d+\z/
          yield Chunk.new("mh:#{path.expand_path}", (path + name).read)
        end
      end
    end

    def each_cur
      @cur_path.opendir do |dir|
        dir.each do |name|
          next if name =~ /\A\./
          yield Chunk.new("maildir:cur:#{path.expand_path}",
                          (@cur_path + name).read)
        end
      end
    end

    def each_new
      @new_path.opendir do |dir|
        dir.each do |name|
          next if name == '.' || name == '..'
          new_name = @new_path + name
          if @safe
            yield Chunk.new("maildir:new:#{path.expand_path}", new_name.read)
          else
            cur_name = @cur_path + "#{name}:2,"
            begin
              cur_name.make_link(new_name)
              new_name.unlink
            rescue Exception
              cur_name.unlink
            end
            yield Chunk.new("maildir:cur:#{path.expand_path}", cur_name.read)
          end
        end
      end
    end
  end
end
