# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'iconv'
require 'msgcab/logger'

module MsgCab
  class EWord
    def initialize(charset, decoded)
      @charset = charset
      @decoded = decoded
    end
    attr_reader :charset, :decoded
  end

  class EWord
    class DecodeError < StandardError; end

    class Decoder
      Token = /[^\x0-\x20\]\[()<>@,;:\"\/?.=]+/

      def decode(input)
        input = input.gsub(/\r?\n/, '')
        ewords = []
        while input =~ /=\?(#{Token})\?(#{Token})\?([!->@-~]+)\?=/io
          charset, encoding, encoded_text = $1.upcase, $2.upcase, $3
          preceding, input = $`, $'		# '`
          if preceding =~ /\S/
            ewords << EWord.new('US-ASCII', preceding.sub(/\s+\z/, ''))
          end
          case encoding
          when 'Q'
            ewords << EWord.new(charset, encoded_text.unpack('M')[0])
          when 'B'
            ewords << EWord.new(charset, encoded_text.unpack('m')[0])
          else
            raise DecodeError, "Unknown encoding #{encoding}"
          end
        end
        unless input.empty?
          ewords << EWord.new('US-ASCII', input)
        end
        ewords
      end

      class << self
        include Logging
      end

      def self.decode(input, output_encoding = nil)
        ewords = Decoder.new.decode(input)
        if output_encoding
          buf = ''
          ewords.each do |eword|
            string = eword.decoded
            begin
              buf << Iconv.conv(output_encoding, eword.charset, string)
            rescue Iconv::Failure => e
              buf << e.success
              log(3, "Failed to decode: #{e.failed}")
            rescue Exception
              log(3, 'Failed to decode')
            end
          end
          buf
        else
          words.collect {|word| word.decoded}.join
        end
      end
    end
  end
end
