# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'rmail'
require 'iconv'
require 'msgcab/encode'
require 'msgcab/logger'

module MsgCab
  class Field
    private
    def initialize(name, body)
      @name, @body = name, body
    end

    public
    attr_reader :name, :body
  end

  class Entity
    include Logging

    private
    def initialize(message, cid)
      @message = message
      @cid = cid

      content_type = @message.header.content_type

      # No Content-Type: header.
      unless content_type
        @content_type = 'text/plain'
      else
        @content_type = content_type.downcase
      end
    end

    public
    attr_reader :content_type, :cid

    def self.parse(input)
      Entity.new(RMail::Parser.read(input), Array.new)
    end

    def [](key)
      message = @message.part(key)
      Entity.new(message, @cid + [key]) if message
    end

    def each
      index = 0
      @message.each_part do |part|
        yield Entity.new(part, @cid + [index])
        index += 1
      end
    end

    def find_part(message, cid)
      if cid.empty?
        message
      else
        find_part(message.part(cid.shift), cid)
      end
    end

    def find(cid)
      if cid.kind_of?(String)
        cid = cid.split('.').collect {|i| i.to_i - 1}
        cid.shift
      end
      Entity.new(find_part(@message, cid.dup), cid)
    end

    def size
      result = 0
      @message.each_part do
        result += 1
      end
      result
    end

    def header_fields(*names)
      @message.header.select(*names).collect {
        |name, body|
        Field.new(name, body)
      }
    end

    def multipart?
      @message.multipart?
    end

    def header_param(field_name, param_name, default = nil)
      @message.header.param(field_name, param_name, default)
    end

    def decode
      @message.decode
    end

    def charset
      @message.header.param('content-type', 'charset')
    end

    def to_s
      @message.to_s
    end

    def mbox
      mbox_from = @message.header.mbox_from ||
        "From bogus@example.org #{Time.now.rfc822}"
      result = String.new
      result << mbox_from << "\r\n" << @message.header.to_string(false) <<
        "\r\n" << @message.body.gsub(/^From /, '>From ') << "\r\n"
      result
    end

    def decode_body(to_charset = MsgCab.kcode_charset, from_charset = nil)
      from_charset = charset || Config['default_charset']
      result = String.new
      if to_charset && from_charset
        begin
          result << Iconv.conv(to_charset, from_charset, decode)
        rescue Iconv::Failure => e
          result << e.success if e.success
          log(3, "Failed to decode: #{e.failed}")
        rescue Exception
          log(3, 'Failed to decode')
        end
      else
        result = decode
      end
      result
    end
  end
end
