# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'test/unit'
require 'pathname'
require 'fileutils'
require 'msgcab/config'

include MsgCab

class TestConfig < Test::Unit::TestCase
  def test_initialize
    path = Pathname.new(__FILE__).dirname + 'test-config0.yml'
    assert_nothing_raised do
      Config.new(path)
    end
  end

  def test_s_instance
    path = Pathname.new(__FILE__).dirname + 'test-config0.yml'
    instance = Config.load(path)
    assert_equal(instance, Config.load(path))
  end

  def test_aref
    path = Pathname.new(__FILE__).dirname + 'test-config0.yml'
    config = Config.new(path)
    assert_instance_of(Hash, config['database'])
    assert_equal('DBI', config['database', 'adapter'])

    path = Pathname.new(__FILE__).dirname + 'test-config-nonexistent.yml'
    assert_nothing_raised do
      config = Config.new(path)
      assert_nil(config['a'])
    end
  end

  def test_aref_modified
    path = Pathname.new(__FILE__).dirname + 'test-config2.yml'
    begin
      config = Config.new(path)
      orig_path = Pathname.new(__FILE__).dirname + 'test-config0.yml'
      FileUtils.cp(orig_path.to_s, path.to_s)
      mtime = path.mtime
      path.utime(path.atime, mtime - 1)
      assert_equal('DBI', config['database', 'adapter'])
      orig_path = Pathname.new(__FILE__).dirname + 'test-config1.yml'
      FileUtils.cp(orig_path.to_s, path.to_s)
      assert_equal('SQLite', config['database', 'adapter'])
    ensure
      path.unlink
    end
  end

  def test_aref_not_modified
    path = Pathname.new(__FILE__).dirname + 'test-config2.yml'
    begin
      config = Config.new(path)
      orig_path = Pathname.new(__FILE__).dirname + 'test-config0.yml'
      FileUtils.cp(orig_path.to_s, path.to_s)
      mtime = path.mtime
      assert_equal('DBI', config['database', 'adapter'])
      orig_path = Pathname.new(__FILE__).dirname + 'test-config1.yml'
      FileUtils.cp(orig_path.to_s, path.to_s)
      path.utime(path.atime, mtime)
      assert_equal('DBI', config['database', 'adapter'])
    ensure
      path.unlink
    end
  end

  def test_relative_path
    path = Pathname.new(__FILE__).dirname + 'test-config0.yml'
    config = Config.new(path)
    assert_equal(Pathname.new('/').relative_path_from(path.dirname.expand_path),
                 config.relative_path('/'))
    assert_equal((path.dirname + Pathname.new('./a')).relative_path_from(path.dirname),
                 config.relative_path('./a'))
  end

  def test_absolute_path
    path = Pathname.new(__FILE__).dirname + 'test-config0.yml'
    config = Config.new(path)
    assert_equal(Pathname.new('/'),
                 config.absolute_path('/'))
    assert_equal((path.dirname + Pathname.new('./a')).expand_path,
                 config.absolute_path('./a'))
  end
end
