# Copyright (C) 2004, 2005  National Institute of Advanced Industrial Science and Technology
#
# This file is part of msgcab.
#
# msgcab is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# msgcab is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with msgcab; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'yaml'
require 'pathname'
require 'msgcab/logger'

module MsgCab
  class Plugin
    def initialize(attributes)
      @attributes = attributes
    end
    attr_reader :attributes
  end

  class PluginLoader
    include Logging

    def initialize
      @plugin_path = Config.absolute_path(Config['plugin_path'] || './plugin')
      @plugins = Hash.new
      @plugin_status = Hash.new
      @attributes = Hash.new
    end
    attr_reader :attributes

    def loaded?(name)
      @plugin_status[name] == :LOADED || running?(name)
    end

    def running?(name)
      @plugin_status[name] == :RUNNING
    end

    def start_plugins
      plugin_names = Config['plugins'] || Array.new
      (@plugins.keys - plugin_names).each do |plugin_name|
        stop_plugin(plugin_name)
      end
      plugin_names.each do |plugin_name|
        plugin_name.untaint
        start_plugin(plugin_name)
      end
    end

    # Start plugin.
    def start_plugin(plugin_name)
      if @plugin_status[plugin_name] == :RUNNING
        log(3, "#{plugin_name} is already running")
        return
      end
      unless @plugin_status[plugin_name] == :LOADED
        load_plugin(plugin_name)
      end
      return unless @plugin_status[plugin_name] == :LOADED
      plugin = @plugins[plugin_name]
      if plugin.respond_to? 'start_plugin'
        plugin.start_plugin
      end
      @plugin_status[plugin_name] = :RUNNING
    end

    # Stop plugin.
    def stop_plugin(plugin_name)
      unless @plugin_status[plugin_name] == :RUNNING
        log(3, "#{plugin_name} is not running")
        return
      end
      plugin = @plugins[plugin_name]
      if plugin.respond_to? 'stop_plugin'
        plugin.stop_plugin
        @plugin_status[plugin_name] = :LOADED
      else
        log(3, "#{plugin_name} doesn't support stop_plugin")
      end
    end

    private
    def plugin_filename(basename)
      if defined? MsgCab::FLAVOR and MsgCab::FLAVOR
        "#{basename}.#{MsgCab::FLAVOR}.rb"
      else
        "#{basename}.rb"
      end
    end

    def load_plugin(plugin_name)
      path = @plugin_path + plugin_name
      if path.directory?
        load_plugin_dir(path, plugin_name)
      else
        load_plugin_dir(@plugin_path, plugin_name)
      end
    end

    def load_plugin_dir(path, plugin_name)
      return if @plugin_status[plugin_name] == :LOADED ||
        @plugin_status[plugin_name] == :RUNNING
      path += plugin_filename(plugin_name)
      return unless path.exist?
      begin
        $LOAD_PATH.unshift(path.to_s) unless $LOAD_PATH.include?(path.to_s)
        load(path.to_s)
        plugin_class = Plugin.const_get("#{plugin_name.split(/[-_]+/).collect {|word| word.capitalize}.join}Plugin")
        @plugins[plugin_name] = plugin_class.new(@attributes)
        @plugin_status[plugin_name] = :LOADED
      rescue Exception => e
        log(3, "Failed to load #{plugin_name}")
      end
    end
  end
end
