/* This file is part of the MPFRCPP Library.

  Copyright (c) 2006 -- 2007 Alexey V. Beshenov <bav.272304@gmail.com>.

  The MPFRCPP Library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  The MPFRCPP Library is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with the MPFRCPP Library; see the file COPYING.LIB. If
  not, write to the Free Software Foundation, Inc., 51 Franklin Street,
  Fifth Floor, Boston, MA 02110-1301, USA. */

/**
 * @file real.hpp
 * @date 2007-03-30
 * mpfrcpp::Real definition.
 * INTERNAL HEADER, NOT TO BE USED DIRECTLY.
 */

#ifndef INCLUDED_BY_MPFRCPP
    #error THIS IS AN INTERNAL HEADER ONLY, NOT TO BE USED DIRECTLY
#endif

#ifndef MPFRCPP_REAL_REAL
#define MPFRCPP_REAL_REAL

namespace mpfrcpp {

    //------------------------------------------------------------

    class Real {

            //------------------------------------------------------------

            mpfr_t x_;    // Number
            static GlobalParameters& params_;

            //------------------------------------------------------------

        public:

            //------------------------------------------------------------

            /**
             * Constructors
             */

            Real ( const Precision& = getParameters().getDefaultPrecision() ) throw();
            Real ( const Real& ) throw(); // With the same precision
            Real ( const Real&, const Precision&, const RoundMode& ) throw();
            // With the specified precision and round mode
            Real ( const Real&, const Precision& ) throw();
            // With the specified precision and default round mode
            Real ( const Real&, const RoundMode& ) throw();
            // With the specified round mode and default precision

            Real ( const mpfr_t&,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const mpfr_t&, const RoundMode& ) throw();

            Real ( const unsigned long int,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const unsigned long int, const RoundMode& ) throw();

            Real ( const unsigned int,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const unsigned int, const RoundMode& ) throw();

            Real ( const unsigned short int,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const unsigned short int, const RoundMode& ) throw();

            Real ( const long int,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const long int, const RoundMode& ) throw();

            Real ( const int,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const int, const RoundMode& ) throw();

            Real ( const short int,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const short int, const RoundMode& ) throw();

            Real ( const double&,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const double&, const RoundMode& ) throw();

            Real ( const long double&,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const long double&, const RoundMode& ) throw();

            Real ( const mpz_t&,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const mpz_t&, const RoundMode& ) throw();

            Real ( const mpq_t&,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const mpq_t&, const RoundMode& ) throw();

            Real ( const mpf_t&,
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();

            Real ( const std::string&, const Base&,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() )
            throw( InvalidNumberStringError );
            Real ( const std::string&, const Base&, const RoundMode& )
            throw( InvalidNumberStringError );
            Real ( const std::string&,
                   const Precision& = getParameters().getDefaultPrecision() )
            throw( InvalidNumberStringError );
            Real ( const std::string&, const RoundMode& )
            throw( InvalidNumberStringError );

#ifdef GMP_CPP_INTERFACE

            Real ( const mpz_class&,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const mpz_class&, const RoundMode& ) throw();

            Real ( const mpq_class&,
                   const Precision& = getParameters().getDefaultPrecision(),
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
            Real ( const mpq_class&, const RoundMode& ) throw();

            Real ( const mpf_class&,
                   const RoundMode& = getParameters().getDefaultRoundMode() ) throw();
#endif    // GMP_CPP_INTERFACE

            ~Real() throw();

            //------------------------------------------------------------

            /**
             * Static methods
             */

            static GlobalParameters& getParameters() throw();
            static void setParameters ( GlobalParameters& ) throw();
            static Real epsilon ( const Precision& =
                                      getParameters().getDefaultPrecision() ) throw();

            //------------------------------------------------------------

            /**
             * Non-static methods
             */

            int checkRange () throw();
            // BUG: NON-CONST, mpfr_check_range() argument isn't const
            Exponent getExponent () const throw();

            mpfr_t& getMpfrT () throw();
            const mpfr_t& getMpfrT () const throw();
            Precision getPrecision() const throw();

            template <typename T> bool isFits () const throw();

            bool isInfinity () const throw();
            bool isInteger () const throw();
            bool isNaN () const throw();
            bool isNumber () const throw();
            bool isZero () const throw();

            void round ( const Precision&, const RoundMode& ) throw();
            void round ( const Precision& ) throw(); // with default round mode
            void setExponent ( const Exponent& ) throw();
            void setPrecision( const Precision& ) throw();
            void setPrecision( mpfr_prec_t ) throw();
            void setToInfinity ( const int sign ) throw();
            void setToNaN () throw();
            int sign () const throw( ComparisonWithNaNError );
            void subnormalize ( const Precision&,
                                const RoundMode& =
                                    getParameters().getDefaultRoundMode() ) throw();

            /**
             * toString
             *     p is precision FOR GIVEN BASE.
             *     p = -1 means real precision.
             *     now p = 0 is equivalent to p = -1, but reserved to denote precision 0
             *     in further versions.
             */

            std::string toString () const throw( StringOutputError );

            std::string toString ( int p, const RoundMode&, const Base& )
            const throw( StringOutputError );

            std::string toString ( int p ) const throw( StringOutputError );

            std::string toString ( const RoundMode& ) const throw( StringOutputError );

            std::string toString ( const Base& ) const throw( StringOutputError );

            std::string toString ( int p, const RoundMode& )
            const throw( StringOutputError );

            std::string toString ( int p, const Base& )
            const throw( StringOutputError );

            std::string toString ( const RoundMode&, const Base& )
            const throw( StringOutputError );

            //------------------------------------------------------------

            /**
             * Assignment
             */

            Real& operator= ( const Real& ) throw();
            Real& operator= ( const mpfr_t& ) throw();
            Real& operator= ( const unsigned long int ) throw();
            Real& operator= ( const unsigned int ) throw();
            Real& operator= ( const unsigned short int ) throw();
            Real& operator= ( const long int ) throw();
            Real& operator= ( const int ) throw();
            Real& operator= ( const short int ) throw();
            Real& operator= ( const double& ) throw();
            Real& operator= ( const long double& ) throw();
            Real& operator= ( const mpz_t& ) throw();
            Real& operator= ( const mpq_t& ) throw();
            Real& operator= ( const mpf_t& ) throw();
#ifdef GMP_CPP_INTERFACE

            Real& operator= ( const mpz_class& ) throw();
            Real& operator= ( const mpq_class& ) throw();
            Real& operator= ( const mpf_class& ) throw();
#endif    // GMP_CPP_INTERFACE

            Real& operator= ( const std::string& ) throw();
            Real& operator= ( const bool ) throw();

            //------------------------------------------------------------

            /**
             * Conversion
             */

            operator double () const throw( ConversionDoesNotFitsError );
            operator long int () const throw( ConversionDoesNotFitsError );
            operator int () const throw( ConversionDoesNotFitsError );
            operator short int () const throw( ConversionDoesNotFitsError );
            operator long double () const throw( ConversionDoesNotFitsError );
#ifdef GMP_CPP_INTERFACE

            operator mpz_class () const throw();
            operator mpf_class () const throw();
#endif

            operator std::string() const throw( StringOutputError );
            // equivalent to toString()
            operator unsigned long int() const throw( ConversionDoesNotFitsError );
            operator unsigned int() const throw( ConversionDoesNotFitsError );
            operator unsigned short int() const throw( ConversionDoesNotFitsError );

            //------------------------------------------------------------

            /**
             * Arithmetic operators
             */

            Real operator- () const throw();
            Real operator+ () const throw() {
                return * this;
            }

            Real operator+ ( const Real& ) const throw();
            Real operator+ ( const unsigned long int& ) const throw();
            Real operator+ ( const unsigned int& ) const throw();
            Real operator+ ( const unsigned short int& ) const throw();
            Real operator+ ( const long int& ) const throw();
            Real operator+ ( const int& ) const throw();
            Real operator+ ( const short int& ) const throw();
            Real operator+ ( const mpz_t& ) const throw();
            Real operator+ ( const mpq_t& ) const throw();

            Real operator- ( const Real& ) const throw();
            Real operator- ( const unsigned long int& ) const throw();
            Real operator- ( const unsigned int& ) const throw();
            Real operator- ( const unsigned short int& ) const throw();
            Real operator- ( const long int& ) const throw();
            Real operator- ( const int& ) const throw();
            Real operator- ( const short int& ) const throw();
            Real operator- ( const mpz_t& ) const throw();
            Real operator- ( const mpq_t& ) const throw();

            Real operator* ( const Real& ) const throw();
            Real operator* ( const unsigned long int& ) const throw();
            Real operator* ( const unsigned int& ) const throw();
            Real operator* ( const unsigned short int& ) const throw();
            Real operator* ( const long int& ) const throw();
            Real operator* ( const int& ) const throw();
            Real operator* ( const short int& ) const throw();
            Real operator* ( const mpz_t& ) const throw();
            Real operator* ( const mpq_t& ) const throw();

            Real operator/ ( const Real& ) const throw();
            Real operator/ ( const unsigned long int& ) const throw();
            Real operator/ ( const unsigned int& ) const throw();
            Real operator/ ( const unsigned short int& ) const throw();
            Real operator/ ( const long int& ) const throw();
            Real operator/ ( const int& ) const throw();
            Real operator/ ( const short int& ) const throw();
            Real operator/ ( const mpz_t& ) const throw();
            Real operator/ ( const mpq_t& ) const throw();

            Real operator^ ( const Real& ) const throw();
            Real operator^ ( const unsigned long int& ) const throw();
            Real operator^ ( const unsigned int& ) const throw();
            Real operator^ ( const unsigned short int& ) const throw();
            Real operator^ ( const long int& ) const throw();
            Real operator^ ( const int& ) const throw();
            Real operator^ ( const short int& ) const throw();
            Real operator^ ( const mpz_t& ) const throw();
#ifdef GMP_CPP_INTERFACE
            Real operator^ ( const mpz_class& ) const throw();
#endif    // GMP_CPP_INTERFACE

            Real operator-- ( int ) throw();
            Real operator++ ( int ) throw();
            Real operator-- () throw();
            Real operator++ () throw();

            //------------------------------------------------------------

            /**
             * Comparison
             */

            bool operator> ( const Real& ) const throw( ComparisonWithNaNError );
            bool operator>= ( const Real& ) const throw( ComparisonWithNaNError );
            bool operator< ( const Real& ) const throw( ComparisonWithNaNError );
            bool operator<= ( const Real& ) const throw( ComparisonWithNaNError );
            bool operator== ( const Real& ) const throw( ComparisonWithNaNError );
            bool operator!= ( const Real& ) const throw( ComparisonWithNaNError );
    };    // class Real

    //------------------------------------------------------------

}    // namespace mpfrcpp

#endif    // MPFRCPP_REAL_REAL
