/* This file is part of the MPFRCPP Library.

  Copyright (c) 2006 -- 2007 Alexey V. Beshenov <bav.272304@gmail.com>.

  The MPFRCPP Library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  The MPFRCPP Library is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with the MPFRCPP Library; see the file COPYING.LIB. If
  not, write to the Free Software Foundation, Inc., 51 Franklin Street,
  Fifth Floor, Boston, MA 02110-1301, USA. */

/**
 * @file arithmetic_operators.hpp
 * @date 2007-03-30
 * Arithmetic operator overloads for mpfrcpp::Real.
 * INTERNAL HEADER, NOT TO BE USED DIRECTLY.
 */

#ifndef INCLUDED_BY_MPFRCPP
    #error THIS IS AN INTERNAL HEADER ONLY, NOT TO BE USED DIRECTLY
#endif

#ifndef MPFRCPP_REAL_ARITHMETIC_OPERATORS
#define MPFRCPP_REAL_ARITHMETIC_OPERATORS

namespace mpfrcpp {

//------------------------------------------------------------

/**
 * Unary negative sign
 */

Real Real::operator- () const throw() {
    Real y ( getPrecision() );
    mpfr_neg ( y.getMpfrT(), getMpfrT(),
               getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

//------------------------------------------------------------

/**
 * Addition
 */

Real Real::operator+ ( const Real &x ) const throw() {
    Real y ( std::max( getPrecision(), x.getPrecision() ) );
    mpfr_add ( y.getMpfrT(), getMpfrT(), x.getMpfrT(),
               getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator+ ( const unsigned long int &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_add_ui ( y.getMpfrT(), getMpfrT(), x,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
inline Real Real::operator+ ( const unsigned int &x ) const throw() {
    return * this + static_cast<unsigned long int>( x );
}
inline Real Real::operator+ ( const unsigned short int &x ) const throw() {
    return * this + static_cast<unsigned long int>( x );
}

Real Real::operator+ ( const long int &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_add_si ( y.getMpfrT(), getMpfrT(), x,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
inline Real Real::operator+ ( const int &x ) const throw() {
    return * this + static_cast<long int>( x );
}
inline Real Real::operator+ ( const short int &x ) const throw() {
    return * this + static_cast<long int>( x );
}

Real Real::operator+ ( const mpz_t &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_add_z ( y.getMpfrT(), getMpfrT(), x,
                 getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator+ ( const mpq_t &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_add_q ( y.getMpfrT(), getMpfrT(), x,
                 getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

inline Real operator+ ( const unsigned long int &x, const Real &y ) throw() {
    return y + x;
}
inline Real operator+ ( const unsigned int &x, const Real &y ) throw() {
    return y + x;
}
inline Real operator+ ( const unsigned short int &x, const Real &y ) throw() {
    return y + x;
}

inline Real operator+ ( const long int &x, const Real &y ) throw() {
    return y + x;
}
inline Real operator+ ( const int &x, const Real &y ) throw() {
    return y + x;
}
inline Real operator+ ( const short int &x, const Real &y ) throw() {
    return y + x;
}

inline Real operator+ ( const mpz_t &x, const Real &y ) throw() {
    return y + x;
}
inline Real operator+ ( const mpq_t &x, const Real &y ) throw() {
    return y + x;
}

//------------------------------------------------------------

/**
 * Subtraction
 */

Real Real::operator- ( const Real &x ) const throw() {
    Real y ( std::max( getPrecision(), x.getPrecision() ) );
    mpfr_sub ( y.getMpfrT(), getMpfrT(), x.getMpfrT(),
               getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator- ( const unsigned long int &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_sub_ui ( y.getMpfrT(), getMpfrT(), x,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
inline Real Real::operator- ( const unsigned int &x ) const throw() {
    return * this - static_cast<unsigned long int>( x );
}
inline Real Real::operator- ( const unsigned short int &x ) const throw() {
    return * this - static_cast<unsigned long int>( x );
}

Real Real::operator- ( const long int &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_sub_si ( y.getMpfrT(), getMpfrT(), x,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
inline Real Real::operator- ( const int &x ) const throw() {
    return * this - static_cast<long int>( x );
}
inline Real Real::operator- ( const short int &x ) const throw() {
    return * this - static_cast<long int>( x );
}

Real Real::operator- ( const mpz_t &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_sub_z ( y.getMpfrT(), getMpfrT(), x,
                 getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator- ( const mpq_t &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_sub_q ( y.getMpfrT(), getMpfrT(), x,
                 getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real operator- ( const unsigned long int &x, const Real & y ) throw() {
    Real z ( y.getPrecision() );
    mpfr_ui_sub ( z.getMpfrT(), x, y.getMpfrT(),
                  y.getParameters().getDefaultRoundMode().getMpfrRndT() );
    return z;
}
inline Real operator- ( const unsigned int &x, const Real & y ) throw() {
    return static_cast<unsigned long int>( x ) - y;
}
inline Real operator- ( const unsigned short int &x, const Real & y ) throw() {
    return static_cast<unsigned long int>( x ) - y;
}

Real operator- ( const long int &x, const Real &y ) throw() {
    Real z ( y.getPrecision() );
    mpfr_si_sub ( z.getMpfrT(), x, y.getMpfrT(),
                  y.getParameters().getDefaultRoundMode().getMpfrRndT() );
    return z;
}
inline Real operator- ( const int &x, const Real & y ) throw() {
    return static_cast<long int>( x ) - y;
}
inline Real operator- ( const short int &x, const Real & y ) throw() {
    return static_cast<long int>( x ) - y;
}

//------------------------------------------------------------

/**
 * Multiplication
 */

Real Real::operator* ( const Real &x ) const throw() {
    Real y ( std::max( getPrecision(), x.getPrecision() ) );
    mpfr_mul ( y.getMpfrT(), getMpfrT(), x.getMpfrT(),
               getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator* ( const unsigned long int &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_mul_ui ( y.getMpfrT(), getMpfrT(), x,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
inline Real Real::operator* ( const unsigned int &x ) const throw() {
    return * this * static_cast<unsigned long int>( x );
}
inline Real Real::operator* ( const unsigned short int &x ) const throw() {
    return * this * static_cast<unsigned long int>( x );
}

Real Real::operator* ( const long int &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_mul_si ( y.getMpfrT(), getMpfrT(), x,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
inline Real Real::operator* ( const int &x ) const throw() {
    return * this * static_cast<long int>( x );
}
inline Real Real::operator* ( const short int &x ) const throw() {
    return * this * static_cast<long int>( x );
}

Real Real::operator* ( const mpz_t &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_mul_z ( y.getMpfrT(), getMpfrT(), x,
                 getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator* ( const mpq_t &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_mul_q ( y.getMpfrT(), getMpfrT(), x,
                 getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

inline Real operator* ( const unsigned long int &x, const Real &y ) throw() {
    return y * x;
}
inline Real operator* ( const unsigned int &x, const Real &y ) throw() {
    return y * x;
}
inline Real operator* ( const unsigned short int &x, const Real &y ) throw() {
    return y * x;
}

inline Real operator* ( const long int &x, const Real &y ) throw() {
    return y * x;
}
inline Real operator* ( const int &x, const Real &y ) throw() {
    return y * x;
}
inline Real operator* ( const short int &x, const Real &y ) throw() {
    return y * x;
}

inline Real operator* ( const mpz_t &y, const Real &x ) throw() {
    return y * x;
}
inline Real operator* ( const mpq_t &x, const Real &y ) throw() {
    return y * x;
}

//------------------------------------------------------------

/**
 * Division
 */

Real Real::operator/ ( const Real &x ) const throw() {
    Real y ( std::max( getPrecision(), x.getPrecision() ) );
    mpfr_div ( y.getMpfrT(), getMpfrT(), x.getMpfrT(),
               getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator/ ( const unsigned long int &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_div_ui ( y.getMpfrT(), getMpfrT(), x,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
inline Real Real::operator/ ( const unsigned int &x ) const throw() {
    return * this / static_cast<unsigned long int>( x );
}
inline Real Real::operator/ ( const unsigned short int &x ) const throw() {
    return * this / static_cast<unsigned long int>( x );
}

Real Real::operator/ ( const long int &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_div_si ( y.getMpfrT(), getMpfrT(), x,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
inline Real Real::operator/ ( const int &x ) const throw() {
    return * this / static_cast<long int>( x );
}
inline Real Real::operator/ ( const short int &x ) const throw() {
    return * this / static_cast<long int>( x );
}

Real Real::operator/ ( const mpz_t &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_div_z ( y.getMpfrT(), getMpfrT(), x,
                 getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator/ ( const mpq_t &x ) const throw() {
    Real y ( getPrecision() );
    mpfr_div_q ( y.getMpfrT(), getMpfrT(), x,
                 getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real operator/ ( const unsigned long int &x, const Real &y ) throw() {
    Real z ( y.getPrecision() );
    mpfr_ui_div ( z.getMpfrT(), x, y.getMpfrT(),
                  y.getParameters().getDefaultRoundMode().getMpfrRndT() );
    return z;
}
inline Real operator/ ( const unsigned int &x, const Real &y ) throw() {
    return static_cast<unsigned long int>( x ) / y;
}
inline Real operator/ ( const unsigned short int &x, const Real &y ) throw() {
    return static_cast<unsigned long int>( x ) / y;
}

Real operator/ ( const long int &x, const Real &y ) throw() {
    Real z ( y.getPrecision() );
    mpfr_si_div ( z.getMpfrT(), x, y.getMpfrT(),
                  y.getParameters().getDefaultRoundMode().getMpfrRndT() );
    return z;
}
inline Real operator/ ( const int &x, const Real &y ) throw() {
    return static_cast<long int>( x ) / y;
}
inline Real operator/ ( const short int &x, const Real &y ) throw() {
    return static_cast<long int>( x ) / y;
}

//------------------------------------------------------------

/**
 * Power
 */

Real Real::operator^ ( const Real& x ) const throw() {
    Real y ( getPrecision() );
    mpfr_pow ( y.getMpfrT(), getMpfrT(), x.getMpfrT(),
               getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator^ ( const unsigned long int& x ) const throw() {
    Real y ( getPrecision() );
    mpfr_pow_ui ( y.getMpfrT(), getMpfrT(), x,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
inline Real Real::operator^ ( const unsigned int& x ) const throw() {
    (*this)^static_cast<unsigned long int>(x);
}
inline Real Real::operator^ ( const unsigned short int& x ) const throw() {
    (*this)^static_cast<unsigned long int>(x);
}

Real operator^ ( const unsigned long int& x, const Real& y ) throw() {
    Real z ( y.getPrecision() );
    mpfr_ui_pow ( z.getMpfrT(), x, y.getMpfrT(),
                  Real::getParameters().getDefaultRoundMode().getMpfrRndT() );
    return z;
}
inline Real operator^ ( const unsigned int& x, const Real& y ) throw() {
    return static_cast<unsigned long int>(x)^y;
}
inline Real operator^ ( const unsigned short int& x, const Real& y ) throw() {
    return static_cast<unsigned long int>( x )^y;
}

Real Real::operator^ ( const long int& x ) const throw() {
    Real y ( getPrecision() );
    mpfr_pow_si ( y.getMpfrT(), getMpfrT(), x,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
inline Real Real::operator^ ( const int& x ) const throw() {
    return ( *this ) ^ static_cast<long int>( x );
}
inline Real Real::operator^ ( const short int& x ) const throw() {
    return ( *this ) ^ static_cast<long int>( x );
}

Real Real::operator^ ( const mpz_t& x ) const throw() {
    Real y ( getPrecision() );
    mpfr_pow_z ( y.getMpfrT(), getMpfrT(), x,
                 getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

#ifdef GMP_CPP_INTERFACE
Real Real::operator^ ( const mpz_class& x ) const throw() {
    Real y ( getPrecision() );
    mpfr_pow_z ( y.getMpfrT(), getMpfrT(), x.get_mpz_t(),
                 getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}
#endif    // GMP_CPP_INTERFACE

//------------------------------------------------------------

/**
 * Incrementation / Decrementation
 */

Real Real::operator-- ( int ) throw() {
    Real y = *this;
    mpfr_sub_ui ( getMpfrT(), getMpfrT(), 1,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator++ ( int ) throw() {
    Real y = *this;
    mpfr_add_ui ( getMpfrT(), getMpfrT(), 1,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return y;
}

Real Real::operator-- () throw() {
    mpfr_sub_ui ( getMpfrT(), getMpfrT(), 1,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return *this;
}

Real Real::operator++ () throw() {
    mpfr_add_ui ( getMpfrT(), getMpfrT(), 1,
                  getParameters().getDefaultRoundMode().getMpfrRndT() );
    return *this;
}

//------------------------------------------------------------

/**
 * Operations with assignment
 */

template <typename T>
inline Real& operator+= ( Real &x, const T &y ) throw() {
    return x = x + y;
}
template <typename T>
inline Real& operator-= ( Real &x, const T &y ) throw() {
    return x = x - y;
}
template <typename T>
inline Real& operator*= ( Real &x, const T &y ) throw() {
    return x = x * y;
}
template <typename T>
inline Real& operator/= ( Real &x, const T &y ) throw() {
    return x = x / y;
}
template <typename T>
inline Real& operator^= ( Real &x, const T &y ) throw() {
    return x = x ^ y;
}

//------------------------------------------------------------

} // namespace mpfrcpp

#endif    // MPFRCPP_REAL_ARITHMETIC_OPERATORS
