/* This file is part of the MPFRCPP Library.

  Copyright (c) 2006 -- 2007 Alexey V. Beshenov <bav.272304@gmail.com>.

  The MPFRCPP Library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  The MPFRCPP Library is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with the MPFRCPP Library; see the file COPYING.LIB. If
  not, write to the Free Software Foundation, Inc., 51 Franklin Street,
  Fifth Floor, Boston, MA 02110-1301, USA. */

/**
 * @file global_parameters.hpp
 * @date 2007-03-23
 * MPFRCPP global parameters.
 * INTERNAL HEADER, NOT TO BE USED DIRECTLY.
 */

#ifndef INCLUDED_BY_MPFRCPP
    #error THIS IS AN INTERNAL HEADER ONLY, NOT TO BE USED DIRECTLY
#endif

#ifndef MPFRCPP_LIBRARY_GLOBAL_PARAMETERS
#define MPFRCPP_LIBRARY_GLOBAL_PARAMETERS

namespace mpfrcpp {

    //------------------------------------------------------------

    /**
     * MPFRCPP global parameters
     */

    class GlobalParameters {

            Precision defaultPrecision_;
            Base defaultBase_;
            RoundMode defaultRoundMode_;
            bool initializeByZero_; // initializeByNaN otherwise

        public:

            GlobalParameters ( const Precision& = Precision(),
                               const Base& = Base(),
                               const RoundMode& = RoundMode(),
                               bool initializeByZero = true ) throw();

            void setDefaultBase( const Base& ) throw();
            Base getDefaultBase() const throw();

            void setDefaultPrecision( const Precision& ) throw();
            Precision getDefaultPrecision() const throw();

            void setDefaultRoundMode( const RoundMode& ) throw();
            RoundMode getDefaultRoundMode() const throw();

            bool initializeByZero() throw();
            bool initializeByNaN() throw();
            bool isInitializedByZero() const throw();
            bool isInitializedByNaN() const throw();

            // Methods implemented in MPFR:

            static int setExponentMin ( const Exponent& e ) throw() {
                return mpfr_set_emin( e.getMpExpT() );
            }
            static Exponent getExponentMin () throw() {
                return Exponent( mpfr_get_emin() );
            }
            static int setExponentMax ( const Exponent& e ) throw() {
                return mpfr_set_emax( e.getMpExpT() );
            }
            static Exponent getExponentMax () throw() {
                return Exponent( mpfr_get_emax() );
            }

            static inline Exponent getExponentMinMin() throw() {
                return Exponent( mpfr_get_emin_min() );
            }
            static inline Exponent getExponentMinMax() throw() {
                return Exponent( mpfr_get_emin_max() );
            }
            static inline Exponent getExponentMaxMin() throw() {
                return Exponent( mpfr_get_emax_min() );
            }
            static inline Exponent getExponentMaxMax() throw() {
                return Exponent( mpfr_get_emax_max() );
            }
    };

    //------------------------------------------------------------

    GlobalParameters::GlobalParameters ( const Precision& pr, const Base& b,
                                         const RoundMode& rm,
                                         bool initializeByZero ) throw() :
            defaultPrecision_ ( pr ), defaultBase_ ( b ),
    defaultRoundMode_ ( rm ), initializeByZero_( initializeByZero ) {}

    void GlobalParameters::setDefaultBase( const Base& b ) throw() {
        defaultBase_ = b;
    }

    Base GlobalParameters::getDefaultBase() const throw() {
        return defaultBase_;
    }

    void GlobalParameters::setDefaultPrecision( const Precision& p ) throw() {
        defaultPrecision_ = p;
    }

    Precision GlobalParameters::getDefaultPrecision() const throw() {
        return defaultPrecision_;
    }

    void GlobalParameters::setDefaultRoundMode( const RoundMode& r ) throw() {
        defaultRoundMode_ = r;
    }

    RoundMode GlobalParameters::getDefaultRoundMode() const throw() {
        return defaultRoundMode_;
    }

    bool GlobalParameters::initializeByZero() throw() {
        bool z = initializeByZero_;
        initializeByZero_ = true;
        return z;
    }

    bool GlobalParameters::initializeByNaN() throw() {
        bool z = !initializeByZero_;
        initializeByZero_ = false;
        return z;
    }

    bool GlobalParameters::isInitializedByZero() const throw() {
        return initializeByZero_;
    }

    bool GlobalParameters::isInitializedByNaN() const throw() {
        return !initializeByZero_;
    }

    //------------------------------------------------------------

} // namespace mpfrcpp

#endif    // MPFRCPP_LIBRARY_GLOBAL_PARAMETERS
