// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/build/stage.cc,v 1.3 2004/07/01 22:45:37 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/build/dependency_calc.hh>
#include <mpak/build/environment.hh>
#include <mpak/util/dependency.hh>
#include <mpak/util/node_path.hh>
#include <mpak/util/node_path_list.hh>
#include <mpak/spec/node.hh>
#include <mpak/builtins/version_node.hh>
#include <mpak/builtins/package_node.hh>
#include <mpak/builtins/category_node.hh>
#include <mpak/builtins/dependency_node_data.hh>
#include <mpak/util/cow_vector.hh>
#include <mpak/util/cow_set.hh>
#include <mpak/util/version_spec.hh>

#include <boost/shared_ptr.hpp>
#include <boost/utility.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/topological_sort.hpp>
#include <boost/graph/depth_first_search.hpp>
#include <boost/graph/visitors.hpp>

#include <map>
#include <deque>
#include <algorithm>
#include <utility>
#include <list>
#include <iterator>

#ifdef DEP_CALC_DEBUG
#include <iostream>
#ifndef PRINT_LOCATION
#define PRINT_LOCATION std::cerr << __FILE__ << ':' << __LINE__ << ':' << __PRETTY_FUNCTION__ << std::endl
#endif
#endif

// A few notes about dependency resolution:
// 
// Dependencies that specify slots are given priority over those that
// don't. That is, if a dependency doesn't specify a slot, other
// dependencies that do specify slots will be fulfilled, and if the
// non-slot dependencies happen to work, then they will use the slot
// dependency's dependee.

// The reason we have slots is to allow multiple versions of a package
// to be installed simultaneously. Take gtk+, for example. There are 2
// versions of the gtk+ api (ignoring the older 1.0 series):
//
//    slot 1.2: the gtk+-1.2.x series
//    slot 2.0: the gtk+-2.0.x and gtk+-2.2.x series
//
// We want to be able to have a gtk+-1.2 package installed, as well as
// a gtk+-2.0 package installed, because while most packages now use
// the new api, some older packages still use gtk+-1.2.
// 
// Now lets take a look at gcc. We want to be able to install multiple
// versions of gcc, because while newer versions of gcc provide more
// features, older versions may create more stable code (or maybe
// not).  But regardless of the reason, it would be nice to be able to
// install gcc-3.0, gcc-3.1, and gcc-3.2 simultaneously. But while
// every package using gtk+ will need to specify either slot 1.2 or
// slot 2.0, most packages won't care which version of gcc is
// installed, except maybe that we have a new enough version. (This is
// especially true for C++ packages). So these packages won't specify
// which slot to use in their dependencies.
// 
// Take the following for example:
//
// package gcc:
//    slot 3.1: versions 3.1.0, 3.1.1
//    slot 3.2: versions 3.2.0, 3.2.1, 3.2.2
// 
// package1 has dependency:
//    gcc:3.2
// package2 has dependency:
//    gcc >= 3.1.0
// package3 has dependency:
//    gcc <= 3.2.1
// 
// So starting with package1, we will select gcc-3.2.2. Adding
// package2, which doesn't specify a slot, will also select gcc-3.2.2.
// But now adding package3, we can't choose 3.2.1, and since package1
// actually specifies a slot, we give it priority. So we select
// gcc-3.1.1, which can be installed alongside gcc-3.2.2.

// And now, we begin...

// Whoa, this is really a mess...

namespace mpak
{
    namespace build
    {
        namespace
        {
            struct vertex_info
            {
                boost::optional<util::node_path> version_path;
                boost::optional<std::string> slot;
                bool installed;
                
                vertex_info (void)
                    : version_path (),
                      slot (),
                      installed (false)
                {
                }
                
                vertex_info (const boost::optional<util::node_path> &version_path,
                             const boost::optional<std::string> &slot,
                             bool installed)
                    : version_path (version_path),
                      slot (slot),
                      installed (installed)
                {
                }
                
                vertex_info (const vertex_info &that)
                    : version_path (that.version_path),
                      slot (that.slot),
                      installed (that.installed)
                {
                }
            };
            
            struct edge_info
            {
                util::dependency dependency;
                
                edge_info (void)
                    : dependency ()
                {
                }
                
                edge_info (const util::dependency &dependency)
                    : dependency (dependency)
                {
                }
                
                edge_info (const edge_info &that)
                    : dependency (that.dependency)
                {
                }
            };
            
            typedef boost::property<boost::vertex_name_t, vertex_info,
                                    boost::property<boost::vertex_color_t, boost::default_color_type,
                                                    boost::property<boost::vertex_index_t, unsigned> > > dep_graph_vertex_property_type;
            typedef boost::property<boost::edge_name_t, edge_info> dep_graph_edge_property_type;
            typedef boost::adjacency_list<boost::listS, boost::listS, boost::bidirectionalS,
                                          dep_graph_vertex_property_type, dep_graph_edge_property_type> dep_graph_type;
            typedef boost::graph_traits<dep_graph_type>::vertex_descriptor dep_graph_vertex_descriptor_type;
            typedef boost::graph_traits<dep_graph_type>::edge_descriptor dep_graph_edge_descriptor_type;
            typedef std::pair<boost::optional<util::node_path>, boost::optional<std::string> > node_path_slot_pair_type;
            typedef std::map<node_path_slot_pair_type, dep_graph_vertex_descriptor_type> vertices_by_package_path_slot_map_type;
            typedef std::deque<dep_graph_vertex_descriptor_type> to_visit_queue_type;
            
            struct dep_graph_info
                : boost::noncopyable
            {
                vertices_by_package_path_slot_map_type vertices_by_package_path_slot_map;
                
                dep_graph_info (void)
                    : vertices_by_package_path_slot_map ()
                {
                }
            };
            
            struct version_node_path_compare
            {
                bool
                operator () (const util::node_path &version_node_path1, const util::node_path &version_node_path2)
                {
                    util::version_spec spec1 (version_node_path1.leaf ());
                    util::version_spec spec2 (version_node_path2.leaf ());
                    
                    return spec1 < spec2;
                }
            };
            
            void
            check_dependency (const util::dependency &dependency,
                              const environment &env)
            {
                assert (dependency.valid ());
                
                std::cerr << "checking dependency " << dependency.get_string () << std::endl;
                
                if (dependency.empty ())
                    return;
                
                boost::shared_ptr<const spec::node> node (dependency.get_path ().match (env.get_package_tree_root (), "mpak:package"));
                if (!node) {
                    throw dependency_calc::failure ("no match for dependency " + dependency.get_string ());
                }
                
                boost::shared_ptr<const builtins::package_node> package_node (boost::dynamic_pointer_cast<const builtins::package_node> (node));
                assert (package_node);
                
                if (package_node->has_slots ()) {
                    for (spec::node::child_const_iterator child_i (package_node->begin_children ()); child_i != package_node->end_children ();
                         ++child_i) {
                        if ((*child_i)->get_type () == "mpak:version") {
                            boost::shared_ptr<const builtins::version_node> version_node (boost::dynamic_pointer_cast<const builtins::version_node> (*child_i));
                            if (!version_node->get_slot ()) {
                                throw dependency_calc::failure ("version " + version_node->get_name () +
                                                                " of package " + dependency.get_path ().get_string () +" must specify a slot");
                            } else if (!package_node->has_slot (*version_node->get_slot ())) {
                                throw dependency_calc::failure ("version " + version_node->get_name () +
                                                                " of package " + dependency.get_path ().get_string () +" specifies an unavailable slot");
                            }
                        }
                    }
                } else {
                    for (spec::node::child_const_iterator child_i (package_node->begin_children ()); child_i != package_node->end_children ();
                         ++child_i) {
                        if ((*child_i)->get_type () == "mpak:version") {
                            boost::shared_ptr<const builtins::version_node> version_node (boost::dynamic_pointer_cast<const builtins::version_node> (*child_i));
                            if (version_node->get_slot ()) {
                                throw dependency_calc::failure ("version " + version_node->get_name () +
                                                                " of package " + dependency.get_path ().get_string () +" must not specify a slot");
                            }
                        }
                    }
                }
            }
            
            util::node_path_list
            find_new_dependees (dep_graph_type &dep_graph,
                                const util::dependency &dependency,
                                dep_graph_vertex_descriptor_type dependee_vertex,
                                dep_graph_info &dep_graph_info,
                                const boost::shared_ptr<const builtins::config_node> &config_root,
                                const boost::shared_ptr<const builtins::category_node> &root_node,
                                bool use_slots,
                                bool throw_on_fail)
            {
                util::node_path_list cow_matches (dependency.match (config_root,
                                                                    root_node,
                                                                    util::dependency::match_check_options |
                                                                    util::dependency::match_collect_options));
                if (cow_matches.empty ()) {
                    if (throw_on_fail) {
                        throw dependency_calc::failure ("no match for dependency " + dependency.get_string ());
                    } else {
                        return false;
                    }
                }
                
                boost::property_map<dep_graph_type, boost::edge_name_t>::type edge_name_map (get (boost::edge_name, dep_graph));
                
                std::list<util::node_path> matches (cow_matches.begin (), cow_matches.end ());
                boost::graph_traits<dep_graph_type>::in_edge_iterator edge_i, end_i;
                for (boost::tie (edge_i, end_i) = in_edges (dependee_vertex, dep_graph); edge_i != end_i; ++edge_i) {
                    
                    edge_info edge_info (edge_name_map[*edge_i]);
                    const util::dependency &dependency (edge_info.dependency);
                    
                    if (!use_slots || dependency.get_slot ()) {
                        std::list<util::node_path>::iterator match_i (matches.begin ());
                        while (match_i != matches.end ()) {
                            if (!dependency.check (config_root,
                                                   root_node,
                                                   *match_i,
                                                   util::dependency::match_check_options | util::dependency::match_collect_options)) {
                                match_i = matches.erase (match_i);
                            } else {
                                ++match_i;
                            }
                        }
                        if (matches.empty ())
                            break;
                    }
                }
                
                if (matches.empty () && throw_on_fail) {
                    boost::property_map<dep_graph_type, boost::vertex_name_t>::type vertex_name_map (get (boost::vertex_name, dep_graph));
                    std::string what ("conflicting dependencies:\n");
                    for (boost::tie (edge_i, end_i) = in_edges (dependee_vertex, dep_graph); edge_i != end_i; ++edge_i) {
                        dep_graph_vertex_descriptor_type dependor_vertex (source (*edge_i, dep_graph));
                        vertex_info vertex_info (vertex_name_map[dependor_vertex]);
                        edge_info edge_info (edge_name_map[*edge_i]);
                        
                        if (!use_slots || edge_info.dependency.get_slot ()) {
                            what.push_back ('\t');
                            what.append (edge_info.dependency.get_string ());
                            what.append (" (from ");
                            if (vertex_info.version_path) {
                                what.append (vertex_info.version_path->get_string ());
                            } else {
                                what.append ("config");
                            }
                            what.append (")\n");
                        }
                    }
                    throw dependency_calc::failure (what);
                }
                
                return util::node_path_list (matches.begin (), matches.end ());
            }
            
            util::node_path
            find_new_dependee (dep_graph_type &dep_graph,
                               const util::dependency &dependency,
                               dep_graph_vertex_descriptor_type dependee_vertex,
                               dep_graph_info &dep_graph_info,
                               const environment &env,
                               bool use_slots)
            {
                util::node_path_list cow_matches (dependency.match (env.get_config_root (),
                                                                    env.get_package_tree_root (),
                                                                    util::dependency::match_check_options |
                                                                    util::dependency::match_collect_options));
                if (cow_matches.empty ()) {
                    throw dependency_calc::failure ("no match for dependency " + dependency.get_string ());
                }
                
                boost::property_map<dep_graph_type, boost::edge_name_t>::type edge_name_map (get (boost::edge_name, dep_graph));
                
                std::list<util::node_path> matches (cow_matches.begin (), cow_matches.end ());
                boost::graph_traits<dep_graph_type>::in_edge_iterator edge_i, end_i;
                for (boost::tie (edge_i, end_i) = in_edges (dependee_vertex, dep_graph); edge_i != end_i; ++edge_i) {
                    
                    edge_info edge_info (edge_name_map[*edge_i]);
                    const util::dependency &dependency (edge_info.dependency);
                    
                    if (!use_slots || dependency.get_slot ()) {
                        std::list<util::node_path>::iterator match_i (matches.begin ());
                        while (match_i != matches.end ()) {
                            if (!dependency.check (env.get_config_root (),
                                                   env.get_package_tree_root (),
                                                   *match_i,
                                                   util::dependency::match_check_options | util::dependency::match_collect_options)) {
                                match_i = matches.erase (match_i);
                            } else {
                                ++match_i;
                            }
                        }
                        if (matches.empty ())
                            break;
                    }
                }
                
                if (matches.empty ()) {
                    boost::property_map<dep_graph_type, boost::vertex_name_t>::type vertex_name_map (get (boost::vertex_name, dep_graph));
                    std::string what ("conflicting dependencies:\n");
                    for (boost::tie (edge_i, end_i) = in_edges (dependee_vertex, dep_graph); edge_i != end_i; ++edge_i) {
                        dep_graph_vertex_descriptor_type dependor_vertex (source (*edge_i, dep_graph));
                        vertex_info vertex_info (vertex_name_map[dependor_vertex]);
                        edge_info edge_info (edge_name_map[*edge_i]);
                        
                        if (!use_slots || edge_info.dependency.get_slot ()) {
                            what.push_back ('\t');
                            what.append (edge_info.dependency.get_string ());
                            what.append (" (from ");
                            if (vertex_info.version_path) {
                                what.append (vertex_info.version_path->get_string ());
                            } else {
                                what.append ("config");
                            }
                            what.append (")\n");
                        }
                    }
                    throw dependency_calc::failure (what);
                }
                
                return *max_element (matches.begin (), matches.end (), version_node_path_compare ());
            }
            
            dep_graph_vertex_descriptor_type
            add_version_path (dep_graph_type &dep_graph,
                              const boost::optional<util::node_path> &version_path,
                              const boost::optional<std::string> &slot,
                              bool installed,
                              dep_graph_info &dep_graph_info)
            {
                assert (!version_path ? !slot : true);
                
                vertices_by_package_path_slot_map_type::iterator vertex_i (dep_graph_info.vertices_by_package_path_slot_map.find
                                                                           (std::make_pair (version_path, slot)));
                assert (vertex_i == dep_graph_info.vertices_by_package_path_slot_map.end ());
                
                const boost::optional<util::node_path> package_path (version_path ? version_path->parent () :
                                                                     boost::optional<util::node_path> ());
                vertex_info vertex_info (version_path, slot, installed);
                
                dep_graph_vertex_descriptor_type vertex (add_vertex (vertex_info, dep_graph));
                
#ifdef DEP_CALC_DEBUG
                std::cerr << "added vertex:" << std::endl;
                std::cerr << "    version_path: " << (vertex_info.version_path ? vertex_info.version_path->get_string () : "(not set)") << std::endl;
                std::cerr << "    slot:         " << (vertex_info.slot ? *vertex_info.slot : "(not set)") << std::endl;
                std::cerr << "    installed:    " << vertex_info.installed << std::endl;
                std::cerr << std::endl;
#endif
                
                bool success (dep_graph_info.vertices_by_package_path_slot_map.insert (std::make_pair (std::make_pair (package_path, slot),
                                                                                                       vertex)).second);
                assert (success);
                
                return vertex;
            }
            
            void
            remove_dependees (dep_graph_type &dep_graph,
                              dep_graph_vertex_descriptor_type dependor_vertex,
                              dep_graph_info &dep_graph_info)
            {
                boost::property_map<dep_graph_type, boost::vertex_name_t>::type vertex_name_map (get (boost::vertex_name, dep_graph));
                
                boost::graph_traits<dep_graph_type>::out_edge_iterator edge_i, end_i, next_edge_i;
                boost::tie (edge_i, end_i) = out_edges (dependor_vertex, dep_graph);
                for (next_edge_i = edge_i; edge_i != end_i; edge_i = next_edge_i) {
                    ++next_edge_i;
                    dep_graph_vertex_descriptor_type dependee_vertex = target (*edge_i, dep_graph);
                    
#ifdef DEP_CALC_DEBUG                    
                    boost::property_map<dep_graph_type, boost::edge_name_t>::type edge_name_map (get (boost::edge_name, dep_graph));
                    std::string source_vertex_name (vertex_name_map[source (*edge_i, dep_graph)].version_path ?
                                                    (vertex_name_map[source (*edge_i, dep_graph)].version_path->get_string () +
                                                     (vertex_name_map[source (*edge_i, dep_graph)].slot ? ("::" + *vertex_name_map[source (*edge_i, dep_graph)].slot) : "")) :
                                                    "(not set)");
                    std::string target_vertex_name (vertex_name_map[dependee_vertex].version_path ?
                                                    (vertex_name_map[dependee_vertex].version_path->get_string () +
                                                     (vertex_name_map[dependee_vertex].slot ? ("::" + *vertex_name_map[dependee_vertex].slot) : "")) :
                                                    "(not set)");
                    
                    std::cerr << "removing edge:" << std::endl;
                    std::cerr << "    source vertex: " << source_vertex_name << std::endl;
                    std::cerr << "    target vertex: " << target_vertex_name << std::endl;
                    std::cerr << "    dependency:    " << edge_name_map[*edge_i].dependency.get_string () << std::endl;
                    std::cerr << std::endl;
#endif
                    
                    remove_edge (*edge_i, dep_graph);
                    if (out_degree (dependee_vertex, dep_graph) == 0) {
                        // if nothing else depends on this version, remove it entirely.
                        vertex_info vertex_info (vertex_name_map[dependee_vertex]);
                        
                        boost::optional<util::node_path> package_path (vertex_info.version_path ? vertex_info.version_path->parent () :
                                                                       boost::optional<util::node_path> ());
                        vertices_by_package_path_slot_map_type::size_type n_vertices_erased
                            (dep_graph_info.vertices_by_package_path_slot_map.erase (std::make_pair (package_path,
                                                                                                     vertex_info.slot)));
                        assert (n_vertices_erased == 1);
                        
                        remove_dependees (dep_graph, dependee_vertex, dep_graph_info);
                        remove_vertex (dependee_vertex, dep_graph);
                    }
                }
            }
            
            void
            add_dep_graph_dependency (dep_graph_type &dep_graph,
                                      dep_graph_vertex_descriptor_type dependor_vertex,
                                      const util::dependency &dependency,
                                      dep_graph_info &dep_graph_info,
                                      const environment &env,
                                      to_visit_queue_type &to_visit_queue);
            
            void
            redo_no_slot_dependencies (dep_graph_type &dep_graph,
                                       dep_graph_vertex_descriptor_type dependee_vertex,
                                       dep_graph_info &dep_graph_info,
                                       const environment &env,
                                       to_visit_queue_type &to_visit_queue)
            {
                boost::property_map<dep_graph_type, boost::vertex_name_t>::type vertex_name_map
                    (get (boost::vertex_name, dep_graph));
                boost::property_map<dep_graph_type, boost::edge_name_t>::type edge_name_map
                    (get (boost::edge_name, dep_graph));
                vertex_info dependee_vertex_info (vertex_name_map[dependee_vertex]);
                assert (dependee_vertex_info.version_path);
                
                // for each node that dependee_vertex is depended on by
                boost::graph_traits<dep_graph_type>::in_edge_iterator edge_i, end_edge_i, next_edge_i;
                boost::tie (edge_i, end_edge_i) = in_edges (dependee_vertex, dep_graph);
                for (next_edge_i = edge_i; edge_i != end_edge_i; edge_i = next_edge_i) {
                    ++next_edge_i;
                    edge_info edge_info (edge_name_map[*edge_i]);
                    if (!edge_info.dependency.get_slot () &&
                        !edge_info.dependency.check (env.get_config_root (),
                                                     dependee_vertex_info.installed ?
                                                     boost::shared_ptr<const builtins::category_node> (env.get_installed_tree_root ()) :
                                                     env.get_package_tree_root (),
                                                     *dependee_vertex_info.version_path,
                                                     util::dependency::match_check_options | util::dependency::match_collect_options)) {
                        dep_graph_vertex_descriptor_type dependor_vertex (source (*edge_i, dep_graph));
#ifdef DEP_CALC_DEBUG                    
                        std::string source_vertex_name (vertex_name_map[source (*edge_i, dep_graph)].version_path ?
                                                        (vertex_name_map[source (*edge_i, dep_graph)].version_path->get_string () +
                                                         (vertex_name_map[source (*edge_i, dep_graph)].slot ? ("::" + *vertex_name_map[source (*edge_i, dep_graph)].slot) : "")) :
                                                        "(not set)");
                        std::string target_vertex_name (vertex_name_map[dependee_vertex].version_path ?
                                                        (vertex_name_map[dependee_vertex].version_path->get_string () +
                                                         (vertex_name_map[dependee_vertex].slot ? ("::" + *vertex_name_map[dependee_vertex].slot) : "")) :
                                                        "(not set)");
                        
                        std::cerr << "removing edge:" << std::endl;
                        std::cerr << "    source vertex: " << source_vertex_name << std::endl;
                        std::cerr << "    target vertex: " << target_vertex_name << std::endl;
                        std::cerr << "    dependency:    " << edge_name_map[*edge_i].dependency.get_string () << std::endl;
                        std::cerr << std::endl;
#endif
                        remove_edge (*edge_i, dep_graph);
                        add_dep_graph_dependency (dep_graph, dependor_vertex,
                                                  edge_info.dependency,
                                                  dep_graph_info,
                                                  env,
                                                  to_visit_queue);
                    }
                }
            }
            
            bool
            find_no_slot_dependency_for_slotted_package (dep_graph_type &dep_graph,
                                                         dep_graph_vertex_descriptor_type dependor_vertex,
                                                         const util::dependency &dependency,
                                                         dep_graph_info &dep_graph_info,
                                                         const boost::shared_ptr<const builtins::config_node> config_root,
                                                         const boost::shared_ptr<const builtins::category_node> root_node,
                                                         bool installed,
                                                         bool throw_on_fail,
                                                         to_visit_queue_type &to_visit_queue)
            {
                util::node_path_list matches (dependency.match (config_root, root_node,
                                                                util::dependency::match_check_options |
                                                                util::dependency::match_collect_options));
                if (matches.empty ()) {
                    if (throw_on_fail) {
                        boost::property_map<dep_graph_type, boost::vertex_name_t>::type vertex_name_map (get (boost::vertex_name, dep_graph));
                        vertex_info vertex_info (vertex_name_map[dependor_vertex]);
                        throw dependency_calc::failure ("no match for dependency " + dependency.get_string () + " in " +
                                                        (vertex_info.version_path ? vertex_info.version_path->get_string () : "config"));
                    } else {
                        return false;
                    }
                }
                
                matches.make_unique ();
                std::sort (matches.begin (), matches.end (), version_node_path_compare ());
                
                // highest version match is at end, lowest at beginning
                util::node_path_list::const_reverse_iterator match_i (matches.rbegin ());
                util::node_path_list::const_reverse_iterator end_match_i (matches.rend ());
                for (; match_i != end_match_i; ++match_i) {
                    const boost::shared_ptr<const spec::node> version_spec_node (match_i->match (root_node,
                                                                                                 "mpak:version"));
                    assert (version_spec_node);
                    const boost::shared_ptr<const builtins::version_node> version_node
                        (boost::dynamic_pointer_cast<const builtins::version_node>
                         (version_spec_node));
                    assert (version_node);
                    assert (version_node->get_slot ());
                    
                    vertices_by_package_path_slot_map_type::iterator dependee_vertex_i
                        (dep_graph_info.vertices_by_package_path_slot_map.find (std::make_pair (dependency.get_path (),
                                                                                                version_node->get_slot ())));
                    if (dependee_vertex_i == dep_graph_info.vertices_by_package_path_slot_map.end ()) {
                        dep_graph_vertex_descriptor_type dependee_vertex (add_version_path (dep_graph,
                                                                                            *match_i,
                                                                                            version_node->get_slot (),
                                                                                            installed, dep_graph_info));
                        to_visit_queue.push_back (dependee_vertex);
                        
                        dep_graph_edge_descriptor_type edge (add_edge (dependee_vertex,
                                                                       dependor_vertex,
                                                                       edge_info (dependency),
                                                                       dep_graph).first);
                        
#ifdef DEP_CALC_DEBUG
                        boost::property_map<dep_graph_type, boost::vertex_name_t>::type vertex_name_map (get (boost::vertex_name, dep_graph));
                        
                        std::string source_vertex_name (vertex_name_map[dependor_vertex].version_path ?
                                                        (vertex_name_map[dependor_vertex].version_path->get_string () +
                                                         (vertex_name_map[dependor_vertex].slot ? ("::" + *vertex_name_map[dependor_vertex].slot) : "")) :
                                                        "(not set)");
                        std::string target_vertex_name (vertex_name_map[dependee_vertex].version_path ?
                                                        (vertex_name_map[dependee_vertex].version_path->get_string () +
                                                         (vertex_name_map[dependee_vertex].slot ? ("::" + *vertex_name_map[dependee_vertex].slot) : "")) :
                                                        "(not set)");
                        std::cerr << "added edge:" << std::endl;
                        std::cerr << "    source vertex: " << source_vertex_name << std::endl;
                        std::cerr << "    target vertex: " << target_vertex_name << std::endl;
                        std::cerr << "    dependency:    " << dependency.get_string () << std::endl;
                        std::cerr << std::endl;
#endif
                        break;
                    } else {
                        boost::property_map<dep_graph_type, boost::vertex_name_t>::type vertex_name_map (get (boost::vertex_name,
                                                                                                              dep_graph));
                        dep_graph_vertex_descriptor_type &dependee_vertex (dependee_vertex_i->second);
                        vertex_info vertex_info (vertex_name_map[dependee_vertex]);
                        assert (vertex_info.version_path);
                        
                        if ((installed == vertex_info.installed) && (*match_i == *vertex_info.version_path)) {
                            dep_graph_edge_descriptor_type edge (add_edge (dependee_vertex,
                                                                           dependor_vertex,
                                                                           edge_info (dependency),
                                                                           dep_graph).first);
#ifdef DEP_CALC_DEBUG
                            std::string source_vertex_name (vertex_name_map[dependor_vertex].version_path ?
                                                            (vertex_name_map[dependor_vertex].version_path->get_string () +
                                                             (vertex_name_map[dependor_vertex].slot ? ("::" + *vertex_name_map[dependor_vertex].slot) : "")) :
                                                            "(not set)");
                            std::string target_vertex_name (vertex_name_map[dependee_vertex].version_path ?
                                                            (vertex_name_map[dependee_vertex].version_path->get_string () +
                                                             (vertex_name_map[dependee_vertex].slot ? ("::" + *vertex_name_map[dependee_vertex].slot) : "")) :
                                                            "(not set)");
                            std::cerr << "added edge:" << std::endl;
                            std::cerr << "    source vertex: " << source_vertex_name << std::endl;
                            std::cerr << "    target vertex: " << target_vertex_name << std::endl;
                            std::cerr << "    dependency:    " << dependency.get_string () << std::endl;
                            std::cerr << std::endl;
#endif
                            break;
                        }
                    }
                }
                
                return match_i != end_match_i;
            }
            
            void
            add_dep_graph_dependency (dep_graph_type &dep_graph,
                                      dep_graph_vertex_descriptor_type dependor_vertex,
                                      const util::dependency &dependency,
                                      dep_graph_info &dep_graph_info,
                                      const environment &env,
                                      to_visit_queue_type &to_visit_queue)
            {
                check_dependency (dependency, env);
                
                if (dependency.empty ())
                    return;
                
                dep_graph_vertex_descriptor_type dependee_vertex;
                
                const boost::shared_ptr<const spec::node> package_spec_node (dependency.get_path ().match (env.get_package_tree_root (),
                                                                                                     "mpak:package"));
                assert (package_spec_node);
                const boost::shared_ptr<const builtins::package_node> package_node (boost::dynamic_pointer_cast<const builtins::package_node>
                                                                                    (package_spec_node));
                assert (package_node);
                
                boost::property_map<dep_graph_type, boost::vertex_name_t>::type vertex_name_map (get (boost::vertex_name, dep_graph));
                
                // make sure dependency doesn't disqualify the dependor
                const vertex_info &dependor_vertex_info (vertex_name_map[dependor_vertex]);
                if (dependor_vertex_info.version_path &&
                    (dependency.get_path () == dependor_vertex_info.version_path->parent ()) &&
                    (!dependency.get_slot () || (*dependency.get_slot () == *dependor_vertex_info.slot)) &&
                    !dependency.check (env.get_config_root (),
                                       dependor_vertex_info.installed ?
                                       boost::shared_ptr<const builtins::category_node> (env.get_installed_tree_root ()) :
                                       env.get_package_tree_root (),
                                       *dependor_vertex_info.version_path,
                                       util::dependency::match_check_options | util::dependency::match_collect_options)) {
                    throw dependency_calc::failure (dependor_vertex_info.version_path->get_string () +
                                                    " is disqualified by its own dependency (" + dependency.get_string () + ")");
                }
                
                if (package_node->has_slots ()) {
                    
                    if (dependency.get_slot ()) {
                        
                        // package has slots, dependency specifies slot
                        vertices_by_package_path_slot_map_type::iterator dependee_vertex_i
                            (dep_graph_info.vertices_by_package_path_slot_map.find (std::make_pair (dependency.get_path (),
                                                                                                    dependency.get_slot ())));
                        
                        if (dependee_vertex_i == dep_graph_info.vertices_by_package_path_slot_map.end ()) {
                            // we have never seen this package/slot combination.
                            // just create a new vertex for it and add an edge.
                            bool installed (true);
                            util::node_path_list matches (dependency.match (env.get_config_root (), env.get_installed_tree_root (),
                                                                            util::dependency::match_check_options |
                                                                            util::dependency::match_collect_options));
                            
                            if (matches.empty ()) {
                                installed = false;
                                matches = dependency.match (env.get_config_root (), env.get_package_tree_root (),
                                                            util::dependency::match_check_options |
                                                            util::dependency::match_collect_options);
                                if (matches.empty ()) {
                                    throw dependency_calc::failure ("no match for dependency " + dependency.get_string ());
                                }
                            }
                            
                            util::node_path_list::const_iterator best_match_i (std::max_element (matches.begin (),
                                                                                                 matches.end (),
                                                                                                 version_node_path_compare ()));
                            
                            dep_graph_vertex_descriptor_type dependee_vertex (add_version_path (dep_graph,
                                                                                                *best_match_i,
                                                                                                dependency.get_slot (),
                                                                                                installed, dep_graph_info));
                            to_visit_queue.push_back (dependee_vertex);
                            
                            dep_graph_edge_descriptor_type edge (add_edge (dependee_vertex,
                                                                           dependor_vertex,
                                                                           edge_info (dependency),
                                                                           dep_graph).first);
                            
#ifdef DEP_CALC_DEBUG
                            
                            std::string source_vertex_name (vertex_name_map[dependor_vertex].version_path ?
                                                            (vertex_name_map[dependor_vertex].version_path->get_string () +
                                                             (vertex_name_map[dependor_vertex].slot ? ("::" + *vertex_name_map[dependor_vertex].slot) : "")) :
                                                            "(not set)");
                            std::string target_vertex_name (vertex_name_map[dependee_vertex].version_path ?
                                                            (vertex_name_map[dependee_vertex].version_path->get_string () +
                                                             (vertex_name_map[dependee_vertex].slot ? ("::" + *vertex_name_map[dependee_vertex].slot) : "")) :
                                                            "(not set)");
                            std::cerr << "added edge:" << std::endl;
                            std::cerr << "    source vertex: " << source_vertex_name << std::endl;
                            std::cerr << "    target vertex: " << target_vertex_name << std::endl;
                            std::cerr << "    dependency:    " << dependency.get_string () << std::endl;
                            std::cerr << std::endl;
#endif
                        } else {
                            // we have already seen this package/slot combination
                            dependee_vertex = dependee_vertex_i->second;
                            
                            vertex_info dependee_vertex_info (vertex_name_map[dependee_vertex]);
                            assert (dependee_vertex_info.version_path);
                            
                            if (!dependency.check (env.get_config_root (),
                                                   dependee_vertex_info.installed ?
                                                   boost::shared_ptr<const builtins::category_node> (env.get_installed_tree_root ()) :
                                                   env.get_package_tree_root (),
                                                   *dependee_vertex_info.version_path,
                                                   util::dependency::match_check_options | util::dependency::match_collect_options)) {
                                // we need to find a new version that works with all these dependencies.
                                bool installed (true);
                                util::node_path_list new_dependee_paths (find_new_dependees (dep_graph, dependency, dependee_vertex,
                                                                                             dep_graph_info,
                                                                                             env.get_config_root (),
                                                                                             env.get_installed_tree_root (), true, false));
                                if (new_dependee_paths.empty ()) {
                                    installed = false;
                                    new_dependee_paths = find_new_dependees (dep_graph, dependency, dependee_vertex,
                                                                             dep_graph_info,
                                                                             env.get_config_root (),
                                                                             env.get_package_tree_root (), true, true);
                                }
                                remove_dependees (dep_graph, dependee_vertex, dep_graph_info);
                                dependee_vertex_info.version_path = *std::max_element (new_dependee_paths.begin (),
                                                                                       new_dependee_paths.end (),
                                                                                       version_node_path_compare ());
                                dependee_vertex_info.installed = installed;
                                
                                vertex_name_map[dependee_vertex] = dependee_vertex_info;
                                // we need to revisit this vertex
                                to_visit_queue.push_back (dependee_vertex);
                            }
                            
                            // we might have to redo edges for non-slot dependencies
                            redo_no_slot_dependencies (dep_graph, dependee_vertex, dep_graph_info, env, to_visit_queue);
                        }
                        
                    } else {
                        
                        // package has slots, dependency specifies no slot.
                        if (!find_no_slot_dependency_for_slotted_package (dep_graph,
                                                                          dependor_vertex,
                                                                          dependency,
                                                                          dep_graph_info,
                                                                          env.get_config_root (),
                                                                          env.get_installed_tree_root (),
                                                                          true,
                                                                          false,
                                                                          to_visit_queue)) {
                            find_no_slot_dependency_for_slotted_package (dep_graph,
                                                                         dependor_vertex,
                                                                         dependency,
                                                                         dep_graph_info,
                                                                         env.get_config_root (),
                                                                         env.get_package_tree_root (),
                                                                         false,
                                                                         true,
                                                                         to_visit_queue);
                        }
                    }
                    
                } else {
                    // TODO: add installed package support
                    // package has no slots
                    vertices_by_package_path_slot_map_type::iterator dependee_vertex_i
                        (dep_graph_info.vertices_by_package_path_slot_map.find (std::make_pair (dependency.get_path (),
                                                                                                dependency.get_slot ())));
                    
                    if (dependee_vertex_i == dep_graph_info.vertices_by_package_path_slot_map.end ()) {
                        // we have never seen this package.
                        // just create a new vertex for it and add an edge.
                        bool installed (true);
                        util::node_path_list matches (dependency.match (env.get_config_root (), env.get_installed_tree_root (),
                                                                        util::dependency::match_check_options |
                                                                        util::dependency::match_collect_options));
                        
                        if (matches.empty ()) {
                            installed = false;
                            matches = dependency.match (env.get_config_root (), env.get_package_tree_root (),
                                                        util::dependency::match_check_options |
                                                        util::dependency::match_collect_options);
                            if (matches.empty ()) {
                                throw dependency_calc::failure ("no match for dependency " + dependency.get_string ());
                            }
                        }
                        
                        util::node_path_list::const_iterator best_match_i (max_element (matches.begin (), matches.end (), version_node_path_compare ()));
                        dep_graph_vertex_descriptor_type dependee_vertex (add_version_path (dep_graph,
                                                                                            *best_match_i,
                                                                                            dependency.get_slot (),
                                                                                            installed, dep_graph_info));
                        to_visit_queue.push_back (dependee_vertex);
                        
                        dep_graph_edge_descriptor_type edge (add_edge (dependee_vertex,
                                                                       dependor_vertex,
                                                                       edge_info (dependency),
                                                                       dep_graph).first);
#ifdef DEP_CALC_DEBUG
                        std::string source_vertex_name (vertex_name_map[dependor_vertex].version_path ?
                                                        (vertex_name_map[dependor_vertex].version_path->get_string () +
                                                         (vertex_name_map[dependor_vertex].slot ? ("::" + *vertex_name_map[dependor_vertex].slot) : "")) :
                                                        "(not set)");
                        std::string target_vertex_name (vertex_name_map[dependee_vertex].version_path ?
                                                        (vertex_name_map[dependee_vertex].version_path->get_string () +
                                                         (vertex_name_map[dependee_vertex].slot ? ("::" + *vertex_name_map[dependee_vertex].slot) : "")) :
                                                        "(not set)");
                        std::cerr << "added edge:" << std::endl;
                        std::cerr << "    source vertex: " << source_vertex_name << std::endl;
                        std::cerr << "    target vertex: " << target_vertex_name << std::endl;
                        std::cerr << "    dependency:    " << dependency.get_string () << std::endl;
                        std::cerr << std::endl;
#endif
                        
                    } else {
                        // we have already seen this package
                        dependee_vertex = dependee_vertex_i->second;
                        
                        vertex_info dependee_vertex_info (vertex_name_map[dependee_vertex]);
                        
                        if (!dependency.check (env.get_config_root (),
                                               dependee_vertex_info.installed ?
                                               boost::shared_ptr<const builtins::category_node> (env.get_installed_tree_root ()) :
                                               env.get_package_tree_root (),
                                               *dependee_vertex_info.version_path,
                                               util::dependency::match_check_options | util::dependency::match_collect_options)) {
                            // we need to find a new version that works with all these dependencies.
                            bool installed (true);
                            util::node_path_list new_dependee_paths (find_new_dependees (dep_graph, dependency, dependee_vertex,
                                                                                         dep_graph_info,
                                                                                         env.get_config_root (),
                                                                                         env.get_installed_tree_root (), false, false));
                            if (new_dependee_paths.empty ()) {
                                installed = false;
                                new_dependee_paths = find_new_dependees (dep_graph, dependency, dependee_vertex,
                                                                         dep_graph_info,
                                                                         env.get_config_root (),
                                                                         env.get_package_tree_root (), false, true);
                            }
                            remove_dependees (dep_graph, dependee_vertex, dep_graph_info);
                            dependee_vertex_info.version_path = *std::max_element (new_dependee_paths.begin (),
                                                                                   new_dependee_paths.end (),
                                                                                   version_node_path_compare ());
                            dependee_vertex_info.installed = installed;
                            
                            vertex_name_map[dependee_vertex] = dependee_vertex_info;
                            // we need to revisit this vertex
                            to_visit_queue.push_back (dependee_vertex);
                        }
                        
                    }
                }
                
            }
            
            void
            visit_vertex (dep_graph_type &dep_graph,
                          dep_graph_vertex_descriptor_type vertex,
                          dep_graph_info &dep_graph_info,
                          const environment &env,
                          to_visit_queue_type &to_visit_queue)
            {
                assert (in_degree (vertex, dep_graph) == 0); // this vertex shouldn't depend on anything yet
                
                boost::property_map<dep_graph_type, boost::vertex_name_t>::type vertex_name_map (get (boost::vertex_name, dep_graph));
                boost::property_map<dep_graph_type, boost::edge_name_t>::type edge_name_map (get (boost::edge_name, dep_graph));
                boost::property_map<dep_graph_type, boost::vertex_index_t>::type vertex_index_map (get (boost::vertex_index, dep_graph));
                
                vertex_info vertex_info (vertex_name_map[vertex]);
                assert (vertex_info.version_path);
                
#ifdef DEP_CALC_DEBUG
                std::cerr << "visiting vertex " << (vertex_info.version_path ?
                                                    (vertex_info.version_path->get_string () +
                                                     (vertex_info.slot ? ("::" + *vertex_info.slot) : "")) :
                                                    "(not set)") << std::endl;
#endif
                
                boost::shared_ptr<const spec::node> node (vertex_info.version_path->match (env.get_package_tree_root (), "mpak:version"));
                assert (node);
                
                if (node->has_data ("mpak:dependency")) {
                    boost::shared_ptr<const builtins::dependency_node_data> dependency_node_data
                        (boost::dynamic_pointer_cast<const builtins::dependency_node_data> (node->get_data ("mpak:dependency")));
                    for (builtins::dependency_node_data::dependency_iterator dep_i (dependency_node_data->begin_dependencies ("build"));
                         dep_i != dependency_node_data->end_dependencies ("build"); ++dep_i) {
                        add_dep_graph_dependency (dep_graph, vertex, *dep_i, dep_graph_info, env, to_visit_queue);
                    }
                }
            }
            
            // shamelessly ripped from the BGL example program file_dependencies.cpp
            struct cycle_detector
                : public boost::dfs_visitor<>
            {
                cycle_detector (bool& has_cycle) 
                    : has_cycle_ (has_cycle)
                {
                }
                
                template <class edge_type_, class graph_type_>
                void back_edge(edge_type_, graph_type_ &)
                {
                    this->has_cycle_ = true;
                }
                
            protected:
                bool &has_cycle_;
            };
        }
        
        const util::node_path_list
        dependency_calc::
        calculate (const environment &env)
        {
            dep_graph_type dep_graph;
            dep_graph_info dep_graph_info;
            to_visit_queue_type to_visit_queue;
            
            // add a vertex which the base (config and command line) dependencies will come from
            dep_graph_vertex_descriptor_type root_vertex (add_version_path (dep_graph, boost::optional<util::node_path> (), boost::optional<std::string> (), false, dep_graph_info));
            for (dependency_vector_type_::const_iterator dep_i (this->dependencies_.begin ());
                 dep_i != this->dependencies_.end (); ++dep_i) {
                add_dep_graph_dependency (dep_graph, root_vertex, *dep_i, dep_graph_info, env, to_visit_queue);
            }

            boost::property_map<dep_graph_type, boost::vertex_name_t>::type vertex_name_map (get (boost::vertex_name, dep_graph));
            while (!to_visit_queue.empty ()) {
                std::cerr << "visit queue: " << std::endl;
                for (to_visit_queue_type::const_iterator i (to_visit_queue.begin ()); i != to_visit_queue.end (); ++i) {
                    const boost::optional<util::node_path> &version_path (vertex_name_map[*i].version_path);
                    std::cerr << '\t' << (version_path ? version_path->get_string () : "(not set)") << std::endl;
                }
                
                visit_vertex (dep_graph, to_visit_queue.front (), dep_graph_info, env, to_visit_queue);
                to_visit_queue.pop_front ();
            }
            
            // add indexes to vertices
            boost::graph_traits<dep_graph_type>::vertex_iterator vertex_i, end_vertex_i;
            boost::property_map<dep_graph_type, boost::vertex_index_t>::type vertex_index_map (get (boost::vertex_index, dep_graph));
            unsigned index (0);
            for (boost::tie (vertex_i, end_vertex_i) = vertices (dep_graph); vertex_i != end_vertex_i; ++vertex_i, ++index) {
                vertex_index_map[*vertex_i] = index;
            }
            
#ifdef DEP_CALC_DEBUG
            boost::property_map<dep_graph_type, boost::edge_name_t>::type edge_name_map (get (boost::edge_name, dep_graph));
            boost::graph_traits<dep_graph_type>::edge_iterator edge_i, end_edge_i;
            std::cerr << "dep_graph vertices:" << std::endl;
            for (boost::tie (vertex_i, end_vertex_i) = vertices (dep_graph); vertex_i != end_vertex_i; ++vertex_i) {
                unsigned vertex_index (vertex_index_map[*vertex_i]);
                vertex_info vertex_info (vertex_name_map[*vertex_i]);
                std::cerr << "index:            " << vertex_index << std::endl;
                std::cerr << "    version_path: " << (vertex_info.version_path ? vertex_info.version_path->get_string () : "(not set)") << std::endl;
                std::cerr << "    slot:         " << (vertex_info.slot ? *vertex_info.slot : "(not set)") << std::endl;
                std::cerr << "    installed:    " << vertex_info.installed << std::endl;
            }
            std::cerr << std::endl;
            
            std::cerr << "dep_graph edges:" << std::endl;
            for (boost::tie (edge_i, end_edge_i) = edges (dep_graph); edge_i != end_edge_i; ++edge_i) {
                unsigned source_vertex_index (vertex_index_map[source (*edge_i, dep_graph)]);
                unsigned target_vertex_index (vertex_index_map[target (*edge_i, dep_graph)]);
                edge_info edge_info (edge_name_map[*edge_i]);
                std::cerr << "source index:     " << source_vertex_index << std::endl;
                std::cerr << "    target index: " << target_vertex_index << std::endl;
                std::cerr << "    dependency:   " << edge_info.dependency.get_string () << std::endl;
            }
            std::cerr << std::endl;
#endif
            
            // note that we don't check for circular dependencies
            // before this point, because at some point we will handle
            // them specially.
            bool has_cycle (false);
            boost::depth_first_search (dep_graph, boost::visitor (cycle_detector (has_cycle)));
            if (has_cycle) {
                throw failure ("cyclic dependencies detected");
            }
            
            typedef std::list<dep_graph_vertex_descriptor_type> sort_output_type;
            sort_output_type sort_output;
            boost::topological_sort (dep_graph, std::front_inserter (sort_output));
            
            util::node_path_list ret;
            for (sort_output_type::const_iterator i (sort_output.begin ()), end (sort_output.end ()); i != end; ++i) {
                const vertex_info &vertex_info (vertex_name_map[*i]);
                assert (!vertex_info.version_path ? (!vertex_info.slot && !vertex_info.installed) : true);
                if (vertex_info.version_path && !vertex_info.installed)
                    ret.push_back (*vertex_info.version_path);
            }
            
            return ret;
        }
    }
}
