// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/include/mpak/util/dependency.hh,v 1.6 2004/07/07 02:40:42 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__UTIL__DEPENDENCY_HH__
#define __MPAK__UTIL__DEPENDENCY_HH__

#include <mpak/defs.hh>
#include <mpak/util/node_path.hh>
#include <mpak/util/node_path_list.hh>
#include <mpak/util/version_spec.hh>
#include <mpak/util/fwd.hh>
#include <mpak/spec/fwd.hh>
#include <mpak/builtins/fwd.hh>
#include <mpak/builtins/category_node.hh>

#include <boost/optional.hpp>
#include <boost/shared_ptr.hpp>

#include <vector>
#include <string>

namespace mpak
{
    namespace util
    {
        class dependency
        {
        public:
            class failure
                : public std::runtime_error
            {
            public:
                inline failure (const std::string &what)
                    : runtime_error (what)
                {
                }
            };
            
            enum compare_type {
                compare_none,
                compare_less,
                compare_less_equal,
                compare_equal,
                compare_greater_equal,
                compare_greater,
                compare_not_equal,
            };
            
        private:
            node_path path_;
            boost::optional<std::string> slot_;
            compare_type compare_type_;
            boost::optional<version_spec> version_spec_;
            std::vector<std::string> options_;
            
        public:
            dependency (void)
                : path_ (),
                  slot_ (),
                  compare_type_ (compare_none),
                  version_spec_ (),
                  options_ ()
            {
            }
            
            explicit
            dependency (const std::string &dep_string)
                : path_ (),
                  slot_ (),
                  compare_type_ (compare_none),
                  version_spec_ (),
                  options_ ()
            {
                this->parse_string (dep_string);
            }
            
            dependency (const dependency &that)
                : path_ (that.path_),
                  slot_ (that.slot_),
                  compare_type_ (that.compare_type_),
                  version_spec_ (that.version_spec_),
                  options_ (that.options_)
            {
            }
            
            void swap (dependency &that)
            {
                this->path_.swap (that.path_);
                boost::swap (this->slot_, that.slot_);
                std::swap (this->compare_type_, that.compare_type_);
                boost::swap (this->version_spec_, that.version_spec_);
                std::swap (this->options_, that.options_);
            }
            
            dependency &
            operator= (const dependency &that)
            {
                dependency new_dependency (that);
                this->swap (new_dependency);
                return *this;
            }
            
            void
            clear (void)
            {
                dependency dep;
                this->swap (dep);
            }
            
            void
            parse_string (const std::string &string);
            
            const std::string get_string (void) const;
            
            const node_path &
            get_path (void)
                const
            {
                return this->path_;
            }
            
            void
            set_path (const node_path &that_path)
            {
                this->path_ = that_path;
            }
            
            const compare_type
            get_compare_type (void)
                const
            {
                return this->compare_type_;
            }
            
            void
            set_compare_type (const compare_type new_compare_type)
            {
                this->compare_type_ = new_compare_type;
            }
            
            const boost::optional<std::string> &
            get_slot (void)
                const
            {
                return this->slot_;
            }
            
            void
            set_slot (const boost::optional<std::string> &slot)
            {
                this->slot_ = slot;
            }
            
            const boost::optional<version_spec> &
            get_version_spec (void)
                const
            {
                return this->version_spec_;
            }
            
            void
            set_version_spec (const boost::optional<version_spec> &new_version_spec)
            {
                this->version_spec_ = new_version_spec;
            }
            
            bool
            valid (void)
                const
            {
                if (!this->path_.valid ()) {
                    return false;
                }
                if (this->compare_type_ != compare_none) {
                    if (!this->version_spec_) {
                        return false;
                    }
                    if (!this->version_spec_->valid ()) {
                        return false;
                    }
                }
                return true;
            }
            
            const std::vector<std::string>
            get_options (void)
            {
                return this->options_;
            }
            
            void
            set_options (const std::vector<std::string> &options)
            {
                this->options_ = options;
            }
            
            enum match_options
            {
                match_check_options   = 0x1 << 0,
                match_collect_options = 0x1 << 1,
                match_defaults        = 0
            };
            
            const node_path_list
            match (const boost::shared_ptr<const builtins::config_node> &config_root,
                   const boost::shared_ptr<const builtins::category_node> &root_node,
                   match_options flags = match_defaults)
                const;
            
            bool
            check (const boost::shared_ptr<const builtins::config_node> &config_root,
                   const boost::shared_ptr<const builtins::category_node> &root_node,
                   const node_path &node_path,
                   match_options flags = match_defaults)
                const;
            
            bool
            empty (void)
                const
            {
                assert (this->valid ());
                return (this->path_.empty () &&
                        !this->slot_ &&
                        (this->compare_type_ == compare_none) &&
                        !this->version_spec_ &&
                        this->options_.empty ());
            }
            
            bool operator < (const dependency &that)
                const
            {
                if (this->path_ < that.path_)
                    return true;
                else if (this->path_ > that.path_)
                    return false;
                
                if (this->slot_ < that.slot_)
                    return true;
                else if (this->slot_ > that.slot_)
                    return false;
                
                if (this->compare_type_ < that.compare_type_)
                    return true;
                else if (this->compare_type_ > that.compare_type_)
                    return false;
                
                if (this->version_spec_ < that.version_spec_)
                    return true;
                else if (this->version_spec_ > that.version_spec_)
                    return false;
                
                if (this->options_ < that.options_)
                    return true;
                else if (this->options_ > that.options_)
                    return false;
                
                return false;
            }
            
            bool operator == (const dependency &that)
                const
            {
                if (this->path_ != that.path_)
                    return false;
                if (this->slot_ != that.slot_)
                    return false;
                if (this->compare_type_ != that.compare_type_)
                    return false;
                if (this->version_spec_ != that.version_spec_)
                    return false;
                if (this->options_ != that.options_)
                    return false;
                return true;
            }
            
            bool operator!= (const dependency &that)
                const
            {
                return !this->operator == (that);
            }
            
            bool operator> (const dependency &that)
                const
            {
                return that.operator< (*this);
            }
            
            bool operator<= (const dependency &that)
                const
            {
                return !this->operator> (that);
            }
            
            bool operator>= (const dependency &that)
                const
            {
                return !this->operator< (that);
            }
            
        private:
            bool check_options_ (const boost::shared_ptr<const builtins::config_node> &config_root,
                                 const boost::shared_ptr<const builtins::category_node> &root_node,
                                 const node_path &node_path,
                                 match_options flags)
                const;
            bool check_helper_ (const boost::shared_ptr<const builtins::config_node> &config_root,
                                const boost::shared_ptr<const builtins::category_node> &root_node,
                                const node_path &node_path,
                                const boost::shared_ptr<const builtins::version_node> &version_node,
                                match_options flags)
                const;
        };
        
        inline dependency::match_options
        operator& (dependency::match_options options1, dependency::match_options options2)
        {
            return static_cast<dependency::match_options> (static_cast<unsigned> (options1) & static_cast<unsigned> (options2));
        }
        
        inline dependency::match_options
        operator| (dependency::match_options options1, dependency::match_options options2)
        {
            return static_cast<dependency::match_options> (static_cast<unsigned> (options1) | static_cast<unsigned> (options2));
        }
        
        inline dependency::match_options
        operator^ (dependency::match_options options1, dependency::match_options options2)
        {
            return static_cast<dependency::match_options> (static_cast<unsigned> (options1) ^ static_cast<unsigned> (options2));
        }
        
        inline dependency::match_options &
        operator&= (dependency::match_options options1, dependency::match_options options2)
        {
            return options1 = options1 & options2;
        }
        
        inline dependency::match_options &
        operator|= (dependency::match_options options1, dependency::match_options options2)
        {
            return options1 = options1 | options2;
        }
        
        inline dependency::match_options &
        operator^= (dependency::match_options options1, dependency::match_options options2)
        {
            return options1 = options1 ^ options2;
        }
        
        inline dependency::match_options
        operator~ (dependency::match_options options)
        {
            return static_cast<dependency::match_options> (~static_cast<unsigned> (options));
        }
    }
}

namespace std
{
    inline void
    swap (mpak::util::dependency &dependency1, mpak::util::dependency &dependency2)
    {
        dependency1.swap (dependency2);
    }
}

#endif // ifndef __MPAK__UTIL__DEPENDENCY_HH__
