// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/include/mpak/util/cow_vector.hh,v 1.1 2004/07/07 01:54:57 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__UTIL_COW_VECTOR_HH__
#define __MPAK__UTIL_COW_VECTOR_HH__

#include <boost/shared_ptr.hpp>

#include <vector>
#include <new>

namespace mpak
{
    namespace util
    {
        /** \class cow_vector
         *  \brief Implements a copy-on-write vector.
         *  
         *  Note if !cow_vector->unique () then all non-const
         *  operations will invalidate iterators. In addition, all
         *  non-const operations automatically execute
         *  cow_vector->make_unique (), so 2 or more successive
         *  non-const operations (with no copies in between) are safe,
         *  provided there are no race conditions. Thus, you can
         *  safely do cow_vector->begin () immediately followed by
         *  cow_vector->end ().
         *
         *  This could probably be made more efficient by not
         *  implementing it in terms of std::vector, but I'm lazy.
         */
        template<typename value_type_, typename allocator_type_ = typename std::vector<value_type_>::allocator_type>
        class cow_vector
        {
        private:
            typedef std::vector<value_type_, allocator_type_> vector_type_;
            boost::shared_ptr<vector_type_> vector_;
            
        public:
            typedef typename vector_type_::value_type value_type;
            typedef typename vector_type_::size_type size_type;
            typedef typename vector_type_::reference reference;
            typedef typename vector_type_::const_reference const_reference;
            typedef typename vector_type_::pointer pointer;
            typedef typename vector_type_::const_pointer const_pointer;
            typedef typename vector_type_::iterator iterator;
            typedef typename vector_type_::const_iterator const_iterator;
            typedef typename vector_type_::reverse_iterator reverse_iterator;
            typedef typename vector_type_::const_reverse_iterator const_reverse_iterator;
            typedef typename vector_type_::allocator_type allocator_type;
            
            cow_vector (void)
                : vector_ (new vector_type_)
            {
            }
            
            cow_vector (const cow_vector &that)
                : vector_ (that.vector_)
            {
            }
            
            template<typename input_iterator_type_>
            cow_vector (const input_iterator_type_ &first, const input_iterator_type_ &last)
                : vector_ (new vector_type_ (first, last))
            {
            }
            
            cow_vector (const allocator_type &allocator)
                : vector_ (new vector_type_ (allocator))
            {
            }
            
            cow_vector (size_type size, const value_type &value, const allocator_type &allocator = allocator_type ())
                : vector_ (new vector_type_ (size, value, allocator))
            {
            }
            
            cow_vector (size_type size)
                : vector_ (new vector_type_ (size))
            {
            }
            
            ~cow_vector (void)
            {
            }
            
            allocator_type
            get_allocator (void)
                const
            {
                return this->vector_->get_allocator ();
            }
            
            template<typename input_iterator_type_>
            void
            assign (const input_iterator_type_ &first, const input_iterator_type_ &last)
            {
                cow_vector new_cow_vector (first, last);
                this->swap (new_cow_vector);
            }
            
            void
            assign (size_type size, const value_type &value)
            {
                cow_vector new_cow_vector (size, value);
                this->swap (new_cow_vector);
            }
            
            iterator begin (void)
            {
                return this->get_vector_ ().begin ();
            }
            
            iterator end (void)
            {
                return this->get_vector_ ().end ();
            }
            
            const_iterator begin (void)
                const
            {
                return this->get_vector_ ().begin ();
            }
            
            const_iterator end (void)
                const
            {
                return this->get_vector_ ().end ();
            }
            
            reverse_iterator rbegin (void)
            {
                return this->get_vector_ ().rbegin ();
            }
            
            reverse_iterator rend (void)
            {
                return this->get_vector_ ().rend ();
            }
            
            const_reverse_iterator rbegin (void)
                const
            {
                return this->get_vector_ ().rbegin ();
            }
            
            const_reverse_iterator rend (void)
                const
            {
                return this->get_vector_ ().rend ();
            }
            
            bool
            empty (void)
                const
            {
                return this->get_vector_ ().empty ();
            }
            
            size_type
            size (void)
                const
            {
                return this->get_vector_ ().size ();
            }
            
            size_type
            max_size (void)
                const
            {
                return this->get_vector_ ().max_size ();
            }
            
            void
            resize (size_type size, const value_type &value)
            {
                this->get_vector_ ().resize (size, value);
            }
            
            void
            resize (size_type size)
            {
                this->get_vector_ ().resize (size);
            }
            
            void
            reserve (size_type size)
            {
                this->get_vector_ ().reserve (size);
            }
            
            size_type
            capacity (void)
                const
            {
                return this->get_vector_ ().capacity ();
            }
            
            const_reference
            front (void)
                const
            {
                return this->get_vector_ ().front ();
            }
            
            const_reference
            back (void)
                const
            {
                return this->get_vector_ ().back ();
            }
            
            reference
            front (void)
            {
                return this->get_vector_ ().front ();
            }
            
            reference
            back (void)
            {
                return this->get_vector_ ().back ();
            }
            
            reference
            operator[] (size_type index)
            {
                return this->get_vector_ ().operator[] (index);
            }
            
            const_reference
            operator[] (size_type index)
                const
            {
                return this->get_vector_ ().operator[] (index);
            }
            
            reference
            at (size_type index)
            {
                return this->get_vector_ ().at (index);
            }
            
            const_reference
            at (size_type index)
                const
            {
                return this->get_vector_ ().at (index);
            }
            
            void swap (cow_vector &that)
            {
                this->vector_.swap (that.vector_);
            }
            
            void clear (void)
            {
                cow_vector new_nodes;
                this->swap (new_nodes);
            }
            
            void
            push_back (const value_type &value)
            {
                this->get_vector_ ().push_back (value);
            }
            
            void
            pop_back (void)
            {
                this->get_vector_ ().pop_back ();
            }
            
            iterator
            insert (iterator position, const value_type &value)
            {
                return this->get_vector_ ().insert (position, value);
            }
            
            void
            insert (iterator position, size_type size, const value_type &value)
            {
                this->get_vector_ ().insert (position, size, value);
            }
            
            template<typename input_iterator_type_>
            void
            insert (iterator position, const input_iterator_type_ &first, const input_iterator_type_ &last)
            {
                return this->get_vector_ ().insert (position, first, last);
            }
            
            iterator
            erase (iterator first, iterator last)
            {
                return this->erase (first, last);
            }
            
            void
            unique (void)
            const
            {
                return this->vector_.unique ();
            }
            
            void
            make_unique (void)
            {
                if (!this->vector_.unique ()) {
                    boost::shared_ptr<vector_type_> new_nodes (new vector_type_(*this->vector_));
                    this->vector_.swap (new_nodes);
                }
            }
            
            cow_vector &
            operator= (const cow_vector &that)
            {
                cow_vector new_cow_vector (that);
                this->swap (new_cow_vector);
                return *this;
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator == (const cow_vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () == that.get_vector_ ();
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator != (const cow_vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () != that.get_vector_ ();
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator < (const cow_vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () < that.get_vector_ ();
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator > (const cow_vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () > that.get_vector_ ();
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator <= (const cow_vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () <= that.get_vector_ ();
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator >= (const cow_vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () >= that.get_vector_ ();
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator == (const std::vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () == that;
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator != (const std::vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () != that;
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator < (const std::vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () < that;
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator > (const std::vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () > that;
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator <= (const std::vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () <= that;
            }
            
            template<typename that_value_type_, typename that_allocator_type_>
            bool
            operator >= (const std::vector<that_value_type_, that_allocator_type_> &that)
                const
            {
                return this->get_vector_ () >= that;
            }
            
        private:
            const vector_type_ &
            get_vector_ (void)
                const
            {
                return *this->vector_;
            }
            
            vector_type_ &
            get_vector_ (void)
            {
                this->make_unique ();
                return *this->vector_;
            }
            
        };
        
        template<typename value_type_, typename allocator_type_,
                 typename cow_value_type_, typename cow_allocator_type_>
        bool
        operator== (const std::vector<value_type_, allocator_type_> &vector,
                    const cow_vector<cow_value_type_, cow_allocator_type_> &cow_vector)
        {
            return cow_vector == vector;
        }
        
        template<typename value_type_, typename allocator_type_,
                 typename cow_value_type_, typename cow_allocator_type_>
        bool
        operator!= (const std::vector<value_type_, allocator_type_> &vector,
                    const cow_vector<cow_value_type_, cow_allocator_type_> &cow_vector)
        {
            return cow_vector != vector;
        }
        
        template<typename value_type_, typename allocator_type_,
                 typename cow_value_type_, typename cow_allocator_type_>
        bool
        operator< (const std::vector<value_type_, allocator_type_> &vector,
                    const cow_vector<cow_value_type_, cow_allocator_type_> &cow_vector)
        {
            return cow_vector > vector;
        }
        
        template<typename value_type_, typename allocator_type_,
                 typename cow_value_type_, typename cow_allocator_type_>
        bool
        operator> (const std::vector<value_type_, allocator_type_> &vector,
                    const cow_vector<cow_value_type_, cow_allocator_type_> &cow_vector)
        {
            return cow_vector < vector;
        }
        
        template<typename value_type_, typename allocator_type_,
                 typename cow_value_type_, typename cow_allocator_type_>
        bool
        operator<= (const std::vector<value_type_, allocator_type_> &vector,
                    const cow_vector<cow_value_type_, cow_allocator_type_> &cow_vector)
        {
            return cow_vector >= vector;
        }
        
        template<typename value_type_, typename allocator_type_,
                 typename cow_value_type_, typename cow_allocator_type_>
        bool
        operator>= (const std::vector<value_type_, allocator_type_> &vector,
                    const cow_vector<cow_value_type_, cow_allocator_type_> &cow_vector)
        {
            return cow_vector <= vector;
        }
        
    }
}

#endif // ifndef __MPAK__UTIL_COW_VECTOR_HH__
