// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/include/mpak/util/checksummer.hh,v 1.2 2004/06/19 06:27:58 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__BUILD__CHECKSUMMER_HH__
#define __MPAK__BUILD__CHECKSUMMER_HH__

#include <boost/scoped_ptr.hpp>
#include <boost/filesystem/path.hpp>

#include <vector>
#include <streambuf>
#include <stdexcept>
#include <fstream>
#include <sstream>

namespace mpak
{
    namespace util
    {
        namespace detail
        {
            class checksummer_impl_;
        }
        
        class checksummer
        {
        public:
            class failure
                : public std::runtime_error
            {
            public:
                inline failure (const std::string &what)
                    : runtime_error (what)
                {
                }
            };
            
            enum algorithm {
                algorithm_md2,
                algorithm_md5,
                algorithm_sha,
                algorithm_sha1,
                algorithm_mdc2,
                algorithm_ripemd160,
                algorithm_default = algorithm_sha1,
            };
            
            typedef std::vector<unsigned char> checksum_type;
            
        private:
            algorithm algorithm_;
            boost::scoped_ptr<detail::checksummer_impl_> impl_;
            
        public:
            checksummer (const algorithm a = algorithm_default);
            ~checksummer (void);
            
            checksum_type
            checksum (std::streambuf *const sb)
                const;
            
            checksum_type
            checksum (const std::string &string)
                const
            {
                std::istringstream ss (string);
                return this->checksum (ss.rdbuf ());
            }
            
            checksum_type
            checksum (const boost::filesystem::path &file_path)
                const
            {
                std::filebuf fb;
                std::string file_string (file_path.native_file_string ());
                if (!fb.open (file_string.c_str (), std::ios_base::in))
                    throw checksummer::failure ("could not open file: " + file_string);
                return this->checksum (&fb);
            }
            
            const algorithm
            get_algorithm (void)
                const
            {
                return this->algorithm_;
            }
            
            void
            set_algorithm (const algorithm a);
            
            static const std::string
            algorithm_to_string (const algorithm a);
            static const algorithm
            string_to_algorithm (const std::string &string);
            
            static const checksum_type
            string_to_checksum (const std::string &string);
            static const std::string
            checksum_to_string (const checksum_type &checksum);
        };
    }
}

#endif // ifndef __MPAK__BUILD__CHECKSUMMER_HH__
