// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/include/mpak/spec/node.hh,v 1.5 2004/07/06 17:20:31 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__SPEC__NODE_HH__
#define __MPAK__SPEC__NODE_HH__

#include <mpak/defs.hh>
#include <mpak/spec/fwd.hh>
#include <mpak/spec/node_data.hh>
#include <mpak/spec/context.hh>

#include <boost/shared_ptr.hpp>
#include <boost/enable_shared_from_this.hpp>
#include <boost/weak_ptr.hpp>
#include <boost/optional.hpp>
#include <boost/utility.hpp>
#include <boost/iterator/transform_iterator.hpp>

#include <string>
#include <map>
#include <stdexcept>
#include <cassert>

namespace mpak
{
    namespace spec
    {
        class node
            : public boost::noncopyable,
              public boost::enable_shared_from_this<node>
        {
            friend class context;
            
        public:
            class failure
                : public std::runtime_error
            {
            public:
                inline failure (const std::string &what)
                    : runtime_error (what)
                {
                }
            };
            
        private:
            typedef std::map<std::pair<std::string, std::string>, boost::shared_ptr<node> > child_map_type_;
            typedef std::map<std::string, boost::shared_ptr<node_data> > data_map_type_;
            
            const std::string type_;
            const std::string name_;
            boost::optional<boost::weak_ptr<node> > parent_;
            child_map_type_ children_;
            data_map_type_ data_map_;
            mutable context &context_;
            mutable bool initialized_;
            const bool serialized_init_;
            bool dirty_;
            
        private:
            struct child_iterator_transform_
            {
                typedef child_map_type_::value_type::second_type &result_type;
                
                result_type
                operator () (child_map_type_::value_type & v)
                    const
                {
                    return v.second;
                }
            };
            
            struct child_const_iterator_transform_
            {
                typedef const boost::shared_ptr<const node> result_type;
                
                result_type
                operator () (const child_map_type_::value_type &v)
                    const
                {
                    return v.second;
                }
            };
            
            struct data_const_iterator_transform_
            {
                typedef std::pair<data_map_type_::value_type::first_type, const boost::shared_ptr<const node_data> > result_type;
                
                result_type
                operator () (const data_map_type_::value_type &v)
                    const
                {
                    return v;
                }
            };
            
        public:
            typedef boost::transform_iterator<child_iterator_transform_, child_map_type_::iterator> child_iterator;
            typedef boost::transform_iterator<child_const_iterator_transform_, child_map_type_::const_iterator> child_const_iterator;
            typedef data_map_type_::iterator data_iterator;
            typedef boost::transform_iterator<data_const_iterator_transform_, data_map_type_::const_iterator> data_const_iterator;
            
            inline
            node (const std::string &type, const std::string &name,
                  context &context,
                  const bool serialized_init = false)
                : type_ (type),
                  name_ (name),
                  parent_ (),
                  data_map_ (),
                  context_ (context),
                  initialized_ (false),
                  serialized_init_ (serialized_init),
                  dirty_ (false)
            {
            }
            
        protected:
            virtual
            void init (void)
                const;
            
        public:
            virtual
            ~node (void);
            
            inline const std::string &
            get_type (void)
                const
            {
                return this->type_;
            }
            
            inline const std::string &
            get_name (void)
                const
            {
                return this->name_;
            }
            
            inline const boost::optional<const boost::weak_ptr<node> >
            get_parent (void)
            {
                return boost::optional<const boost::weak_ptr<node> > (this->parent_);
            }
            
            inline const boost::optional<const boost::weak_ptr<const node> >
            get_parent (void)
                const
            {
                return boost::optional<const boost::weak_ptr<const node> > (this->parent_);
            }
            
            inline bool
            has_child (const std::string &type, const std::string &name)
                const
            {
                this->init ();
                return this->children_.find (std::make_pair (type, name)) != this->children_.end ();
            }
            
            inline const boost::shared_ptr<node>
            get_child (const std::string &type, const std::string &name)
            {
                this->init ();
                child_map_type_::iterator i (this->children_.find (std::make_pair (type, name)));
                assert (i != this->children_.end ());
                i->second->init ();
                return i->second;
            }
            
            inline const boost::shared_ptr<const node>
            get_child (const std::string &type, const std::string &name)
                const
            {
                this->init ();
                child_map_type_::const_iterator i (this->children_.find (std::make_pair (type, name)));
                assert (i != this->children_.end ());
                i->second->init ();
                return i->second;
            }
            
            inline void
            add_child (const boost::shared_ptr<node> &node)
            {
                this->init ();
                assert (!node->parent_);
                if (!this->children_.insert (child_map_type_::value_type (std::make_pair (node->type_, node->name_), node)).second)
                    throw failure ("could not add child");
                node->parent_ = this->shared_from_this ();
            }
            
            inline void
            remove_child (const std::string &type, const std::string &name)
            {
                this->init ();
                child_map_type_::iterator child = this->children_.find (std::make_pair (type, name));
                assert (child != this->children_.end ());
                child->second->parent_.reset ();
                this->children_.erase (child);
            }
            
            inline bool
            has_data (const std::string &key)
                const
            {
                this->init ();
                return this->data_map_.find (key) != this->data_map_.end ();
            }
            
            inline const boost::shared_ptr<const node_data>
            get_data (const std::string &key)
                const
            {
                this->init ();
                data_map_type_::const_iterator i (this->data_map_.find (key));
                if (i == this->data_map_.end ()) {
                        throw failure ("data for key not found");
                } else {
                    return i->second;
                }
            }
            
            inline const boost::shared_ptr<node_data>
            get_data (const std::string &key)
            {
                this->init ();
                data_map_type_::const_iterator i (this->data_map_.find (key));
                if (i == this->data_map_.end ()) {
                        throw failure ("data for key not found");
                } else {
                    return i->second;
                }
            }
            
            inline void
            add_data (const std::string &key, const boost::shared_ptr<node_data> &data)
            {
                std::pair<data_map_type_::iterator, bool> p (this->data_map_.insert (data_map_type_::value_type (key, data)));
                if (!p.second)
                    throw failure ("could not insert data");
            }
            
            bool
            children_empty (void)
                const
            {
                this->init ();
                return this->children_.empty ();
            }
            
            child_iterator
            begin_children ()
            {
                this->init ();
                return child_iterator (this->children_.begin ());
            }
            
            child_const_iterator
            begin_children ()
                const
            {
                this->init ();
                return child_const_iterator (this->children_.begin ());
            }
            
            child_iterator
            end_children ()
            {
                this->init ();
                return child_iterator (this->children_.end ());
            }
            
            child_const_iterator
            end_children ()
                const
            {
                this->init ();
                return child_const_iterator (this->children_.end ());
            }
            
            data_iterator
            begin_data ()
            {
                this->init ();
                return data_iterator (this->data_map_.begin ());
            }
            
            data_const_iterator
            begin_data ()
                const
            {
                this->init ();
                return data_const_iterator (this->data_map_.begin ());
            }
            
            data_iterator
            end_data ()
            {
                this->init ();
                return data_iterator (this->data_map_.end ());
            }
            
            data_const_iterator
            end_data ()
                const
            {
                this->init ();
                return data_const_iterator (this->data_map_.end ());
            }
            
            const bool
            is_dirty (void)
                const
            {
                return this->dirty_;
            }
            
            void
            set_dirty (const bool dirty)
            {
                this->dirty_ = dirty;
            }
        };
    }
}

#endif // ifndef __MPAK__SPEC__NODE_HH__
