// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/spec/grammar.tcc,v 1.3 2004/05/29 15:52:12 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__SPEC__GRAMMAR_TCC__
#define __MPAK__SPEC__GRAMMAR_TCC__

#include <mpak/spec/command_info.hh>

#include <new>
#include <string>

#include <boost/shared_ptr.hpp>
#include <boost/spirit/core.hpp>
#include <boost/spirit/attribute.hpp>
#include <boost/spirit/utility.hpp>
#include <boost/spirit/phoenix.hpp>
#include <boost/spirit/actor.hpp>

namespace mpak
{
    namespace spec
    {
        namespace detail
        {
            //////////////////////////////////////////////////
            // space_grammar
            
            template<class scanner_type_>
            space_grammar::definition<scanner_type_>::
            definition (const space_grammar &space_grammar)
            {
                using boost::spirit::ch_p;
                using boost::spirit::eol_p;
                using boost::spirit::space_p;
                
                start_         = (space_p - eol_p) | (ch_p ('\\') >> eol_p);
                
                BOOST_SPIRIT_DEBUG_NODE(start_);
            }
            
            template<class scanner_type_>
            const boost::spirit::rule<scanner_type_> &
            space_grammar::definition<scanner_type_>::
            start (void) const
            {
                return this->start_;
            }
            
            //////////////////////////////////////////////////
            // string_grammar
            
            template<class scanner_type_>
            string_grammar::definition<scanner_type_>::
            definition (const string_grammar &self)
            {
                using boost::spirit::ch_p;
                using boost::spirit::lex_escape_ch_p;
                using boost::spirit::space_p;
                
                using phoenix::arg1;
                using phoenix::arg2;
                using phoenix::construct_;
                
                start_         = qstring
                    | mlstring
                    | uqstring;
                qstring        = ch_p('\"') >>
                    (*(lex_escape_ch_p - ch_p ('\"')))[self.string_ = construct_<std::string> (arg1, arg2)] >>
                    ch_p ('\"');
                mlstring       = ch_p ('[') >>
                    (*(lex_escape_ch_p - (ch_p ('[') | ch_p (']'))))[self.string_ = construct_<std::string> (arg1, arg2)] >>
                    ch_p (']');
                uqstring       = (+(lex_escape_ch_p -
                                    (ch_p ('[') |
                                     ch_p (']') |
                                     ch_p ('{') |
                                     ch_p ('}') |
                                     ch_p ('\"') |
                                     ch_p ('\'') |
                                     ch_p (';') |
                                     ch_p ('#') |
                                     space_p)))[self.string_ = construct_<std::string> (arg1, arg2)];
                
                BOOST_SPIRIT_DEBUG_NODE(start_);
                BOOST_SPIRIT_DEBUG_NODE(qstring);
                BOOST_SPIRIT_DEBUG_NODE(mlstring);
                BOOST_SPIRIT_DEBUG_NODE(uqstring);
            }
            
            template<class scanner_type_>
            const boost::spirit::rule<scanner_type_, typename string_grammar::definition<scanner_type_>::context_type> &
            string_grammar::definition<scanner_type_>::
            start (void) const
            {
                return this->start_;
            }
            
            //////////////////////////////////////////////////
            // argument_list_grammar
            
            template<class scanner_type_>
            argument_list_grammar::definition<scanner_type_>::
            definition (const argument_list_grammar &self)
            {
                using boost::spirit::ch_p;
                using boost::spirit::epsilon_p;
                
                using phoenix::arg1;
                using phoenix::bind;
                using phoenix::new_;
                using phoenix::construct_;
                
                start_         = argument >> *(+space >> argument);
                argument       = string[bind (&argument_vector::push_back) (*self.arguments, arg1)]
                    | block;
                block          = ch_p ('{') >> command_list(construct_<boost::shared_ptr<command_info_list> > (new_<command_info_list> ()))
                    [bind (&argument_vector::push_back) (*self.arguments, arg1)] >>
                    *space >> ch_p ('}');
                
                BOOST_SPIRIT_DEBUG_NODE(start_);
                BOOST_SPIRIT_DEBUG_NODE(argument);
                BOOST_SPIRIT_DEBUG_NODE(string);
                BOOST_SPIRIT_DEBUG_NODE(command_list);
                BOOST_SPIRIT_DEBUG_NODE(block);
                BOOST_SPIRIT_DEBUG_NODE(space);
            }
            
            template<class scanner_type_>
            const boost::spirit::rule<scanner_type_, typename argument_list_grammar::definition<scanner_type_>::context_type> &
            argument_list_grammar::definition<scanner_type_>::
            start (void) const
            {
                return this->start_;
            }
            
            //////////////////////////////////////////////////
            // command_grammar
            
            template<class scanner_type_>
            command_grammar::definition<scanner_type_>::
            definition (const command_grammar &self)
            {
                using boost::spirit::alpha_p;
                using boost::spirit::alnum_p;
                using boost::spirit::anychar_p;
                using boost::spirit::ch_p;
                using boost::spirit::end_p;
                using boost::spirit::eol_p;
                using boost::spirit::epsilon_p;
                
                using boost::spirit::swap_a;
                
                using phoenix::arg1;
                using phoenix::arg2;
                using phoenix::bind;
                using phoenix::construct_;
                using phoenix::var;
                using phoenix::new_;
                
                start_         = *space >> qidentifier >>
                    !(+space >>
                      argument_list(construct_<boost::shared_ptr<argument_vector> > (new_<argument_vector> ()))
                      [bind (&command_info::arguments) (self.command_info_) = arg1]);
                qidentifier    = (identifier >>
                                  *(ch_p (':') >> identifier))
                    [bind (&command_info::identifier) (self.command_info_) = construct_<std::string> (arg1, arg2)];
                identifier     = ((alpha_p | ch_p ('_')) >>
                                  *(alnum_p | ch_p ('_') | ch_p ('+') | ch_p ('-')));
                
                BOOST_SPIRIT_DEBUG_NODE(start_);
                BOOST_SPIRIT_DEBUG_NODE(argument_list);
                BOOST_SPIRIT_DEBUG_NODE(qidentifier);
                BOOST_SPIRIT_DEBUG_NODE(identifier);
                BOOST_SPIRIT_DEBUG_NODE(space);
            }
            
            template<class scanner_type_>
            const boost::spirit::rule<scanner_type_, typename command_grammar::definition<scanner_type_>::context_type> &
            command_grammar::definition<scanner_type_>::
            start (void) const
            {
                return this->start_;
            }
            
            //////////////////////////////////////////////////
            // command_list_grammar
            
            template<class scanner_type_>
            command_list_grammar::definition<scanner_type_>::
            definition (const command_list_grammar &self)
            {
                using boost::spirit::alpha_p;
                using boost::spirit::alnum_p;
                using boost::spirit::anychar_p;
                using boost::spirit::ch_p;
                using boost::spirit::end_p;
                using boost::spirit::eol_p;
                using boost::spirit::epsilon_p;
                
                using phoenix::new_;
                using phoenix::arg1;
                using phoenix::bind;
                
                start_         = *(!command[bind (&command_info_list::push_back) (*(self.command_infos), arg1)] >> command_sep);
                command_sep    = *space >> (ch_p (';') | eol_p | comment);
                comment        = epsilon_p (~ch_p ('\\')) >> ch_p ('#') >> *(anychar_p - eol_p) >> eol_p;
                
                BOOST_SPIRIT_DEBUG_NODE(start_);
                BOOST_SPIRIT_DEBUG_NODE(command);
                BOOST_SPIRIT_DEBUG_NODE(command_sep);
                BOOST_SPIRIT_DEBUG_NODE(comment);
                BOOST_SPIRIT_DEBUG_NODE(space);
            }
            
            template<class scanner_type_>
            const boost::spirit::rule<scanner_type_, typename command_list_grammar::definition<scanner_type_>::context_type> &
            command_list_grammar::definition<scanner_type_>::
            start (void) const
            {
                return this->start_;
            }
            
            
            //////////////////////////////////////////////////
            // grammar
            
            template<class scanner_type_>
            grammar::definition<scanner_type_>::
            definition (const grammar &self)
            {
                using phoenix::arg1;
                using phoenix::construct_;
                using phoenix::new_;
                
                start_ = command_list(construct_<boost::shared_ptr<command_info_list> > (new_<command_info_list> ()))[self.command_infos = arg1];
                BOOST_SPIRIT_DEBUG_NODE (start_);
                BOOST_SPIRIT_DEBUG_NODE (command_list);
            }
            
            template<class scanner_type_>
            const boost::spirit::rule<scanner_type_, typename grammar::definition<scanner_type_>::context_type> &
            grammar::definition<scanner_type_>::
            start (void) const
            {
                return this->start_;
            }
        }
    }
}

#endif // ifndef __MPAK_SPEC__GRAMMAR_H__
