// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/builtins/stages.cc,v 1.9 2004/07/07 02:40:42 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/builtins/stages.hh>
#include <mpak/build/stage.hh>
#include <mpak/build/fetcher.hh>
#include <mpak/build/environment.hh>
#include <mpak/spec/context.hh>
#include <mpak/spec/node.hh>
#include <mpak/util/node_path.hh>
#include <mpak/builtins/mpak_node.hh>
#include <mpak/builtins/config_node.hh>
#include <mpak/builtins/category_node.hh>
#include <mpak/builtins/package_node.hh>
#include <mpak/builtins/version_node.hh>
#include <mpak/builtins/dependency_node_data.hh>
#include <mpak/builtins/sources_node_data.hh>
#include <mpak/builtins/script_node_data.hh>
#include <mpak/builtins/installed_node_data.hh>
#include <mpak/builtins/timestamp_node_data.hh>
#include <mpak/util/file_stat.hh>
#include <mpak/util/file_operations.hh>
#include <mpak/util/checksummer.hh>
#include <mpak/util/dependency.hh>
#include <mpak/util/node_path_list.hh>

#include <boost/shared_ptr.hpp>
#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/filesystem/convenience.hpp>
#include <boost/filesystem/exception.hpp>
#include <boost/spirit/phoenix.hpp>
#include <boost/iterator/reverse_iterator.hpp>

#include <fstream>
#include <list>
#include <cstdlib>
#include <string>
#include <algorithm>
#include <unistd.h>
#include <sys/wait.h>
#include <time.h>
#include <sys/time.h>
#include <sys/stat.h>

#include <iostream>

namespace mpak
{
    namespace builtins
    {
        namespace stages
        {
            namespace
            {
                const boost::filesystem::path
                make_path_from_node_path (const util::node_path &node_path)
                {
                    boost::filesystem::path ret;
                    for (util::node_path::element_const_iterator i (node_path.begin_elements ()); i != node_path.end_elements (); ++i) {
                        ret /= *i;
                    }
                    return ret;
                }
                
                const boost::filesystem::path
                make_workdir_path (const util::node_path &node_path,
                                   const build::environment &env)
                {
                    const boost::shared_ptr<const builtins::config_node> config_node (env.get_config_root ());
                    assert (config_node);
                    boost::filesystem::path workdir (env.get_config_root ()->get_root_dir () / config_node->get_mpaktmp_dir () / make_path_from_node_path (node_path));
                    return workdir;
                }
            }
            
            clean::
            ~clean (void)
            {
            }
            
            void
            clean::
            execute (const util::node_path &node_path,
                     const boost::shared_ptr<builtins::version_node> &version_node,
                     const build::environment &env)
                    const
            {
                boost::filesystem::path workdir (make_workdir_path (node_path, env));
                // boost::filesystem::remove has trouble w/ symlinks that point to non-existant files
                // otherwise we'd just use that
                util::traverse_tree (workdir, &util::remove, util::traverse_tree_postorder);
            }
            
            check::
            ~check (void)
            {
            }
            
            void
            check::
            execute (const util::node_path &node_path,
                     const boost::shared_ptr<builtins::version_node> &version_node,
                     const build::environment &env)
                    const
            {
                boost::shared_ptr<const spec::node> node (node_path.parent ().match (env.get_package_tree_root (), "mpak:package"));
                assert (node);
                boost::shared_ptr<const builtins::package_node> package_node (boost::dynamic_pointer_cast<const builtins::package_node> (node));
                assert (package_node);
                
                if (version_node->get_slot ()) {
                    if (!package_node->has_slot (*version_node->get_slot ()))
                        throw failure ("package " + node_path.parent ().get_string () + " has no slot " + *version_node->get_slot () + " (in version " + version_node->get_name () + ")");
                } else {
                    if (package_node->has_slots ()) {
                        throw failure ("package " + node_path.parent ().get_string () + " has slots, no slot specified in version " + version_node->get_name ());
                    }
                }
            }
            
            setup::
            ~setup (void)
            {
            }
            
            void
            setup::
            execute (const util::node_path &node_path,
                     const boost::shared_ptr<builtins::version_node> &version_node,
                     const build::environment &env)
                    const
            {
                boost::filesystem::path workdir (make_workdir_path (node_path, env));
                create_directories (workdir);
                create_directories (workdir / "tmp");
                create_directories (workdir / "scripts");
                create_directories (workdir / "status");
                create_directories (workdir / "build");
                create_directories (workdir / "dest");
            }
            
            dependencies::
            ~dependencies (void)
            {
            }
            
            void
            dependencies::
            execute (const util::node_path &node_path,
                     const boost::shared_ptr<builtins::version_node> &version_node,
                     const build::environment &env)
                    const
            {
                if (version_node->has_data ("mpak:dependency")) {
                    boost::shared_ptr<const dependency_node_data> data (boost::dynamic_pointer_cast<const dependency_node_data> (version_node->get_data ("mpak:dependency")));
                    for (dependency_node_data::dependency_iterator i (data->begin_dependencies ("build")),
                             end (data->end_dependencies ("build")); i != end; ++i) {
                        // if any packages need to be installed, we fail
                        const util::node_path_list matches (i->match (env.get_config_root (),
                                                                      env.get_installed_tree_root (),
                                                                      util::dependency::match_check_options));
                        if (matches.empty ()) {
                            throw failure ("dependencies not met: build " + i->get_string ());
                        }
                    }
                }
            }
            
            fetch::
            ~fetch (void)
            {
            }
            
            namespace
            {
                bool do_checksum (const boost::shared_ptr<const spec::node> &node,
                                  const boost::filesystem::path &local_file_path)
                {
                        boost::shared_ptr<const sources_node_data> sources_data (boost::dynamic_pointer_cast<const sources_node_data> (node->get_data ("mpak:source")));
                        assert (sources_data);
                        if (sources_data->has_checksum (local_file_path.leaf ())) {
                            std::pair<util::checksummer::algorithm, util::checksummer::checksum_type> checksum_pair (sources_data->get_checksum (local_file_path.leaf ()));
                            util::checksummer checksummer (checksum_pair.first);
                            util::checksummer::checksum_type checksum (checksummer.checksum (local_file_path));
                            if (checksum != checksum_pair.second)
                                return false;
                        }
                        return true;
                }
                
                void
                fetch_helper (const boost::shared_ptr<const spec::node> &node,
                              const build::environment &env,
                              const std::string &location,
                              const std::string &filename)
                {
                    assert (!filename.empty ());
                    assert (!location.empty ());
                    const boost::shared_ptr<const builtins::config_node> config_node (env.get_config_root ());
                    if ((*location.begin () != '@') && (location != "!")) {
                        assert (config_node);
                        const boost::filesystem::path local_file_path (env.get_config_root ()->get_root_dir () / config_node->get_source_dir () / make_path_from_node_path (util::node_path (node)) / filename);
                        boost::filesystem::create_directories (local_file_path.branch_path ());
                        
                        unsigned n (0);
                        if (!do_checksum (node, local_file_path)) {
                            build::fetcher f (location, local_file_path);
                            for (; n < 3; n++) {
                                try {
                                    f.fetch ();
                                } catch (build::fetcher::failure &f) {
                                    continue;
                                }
                                if (do_checksum (node, local_file_path))
                                    break;
                            }
                        }
                        
                        if (n == 3)
                            throw build::stage::failure ("could not fetch " + location);
                        
                    } else if (location == "!") {
                        // nothing to be fetched, just need to make sure we have the file
                        const boost::filesystem::path local_file_path (env.get_config_root ()->get_root_dir () / config_node->get_source_dir () / make_path_from_node_path (util::node_path (node)) / filename);
                        if (!exists (local_file_path)) {
                            throw build::stage::failure (local_file_path.native_file_string () + " does not exist, and is not fetchable");
                        }
                        
                        if (!do_checksum (node, local_file_path))
                            throw build::stage::failure ("checksum failed on " + local_file_path.native_file_string ());
                    } else {
                        util::node_path node_path (location.substr (1));
                        boost::shared_ptr<const spec::node> new_node (node_path.match (env.get_package_tree_root (), "mpak:version"));
                        if (!new_node) {
                            new_node = node_path.match (env.get_package_tree_root (), "mpak:package");
                            if (!new_node) {
                                new_node = node_path.match (env.get_package_tree_root (), "mpak:category");
                                if (!new_node) {
                                    throw build::stage::failure ("could not find match for " + location.substr (1));
                                }
                            }
                        }
                        boost::shared_ptr<const sources_node_data> sources_data (boost::dynamic_pointer_cast<const sources_node_data> (new_node->get_data ("mpak:source")));
                        if (!sources_data) {
                            throw build::stage::failure ("could not find source " + filename + " in node " + location.substr (1));
                        }
                        const std::string &new_location (sources_data->get_source (filename));
                        
                        fetch_helper (new_node, env, new_location, filename);
                        
                        // make symlink from other source to our source dir
                        // first get the path (relative to source_dir) in which link will be put.
                        const boost::filesystem::path node_relative_path (make_path_from_node_path (util::node_path (node)));
                        boost::filesystem::path target_path;
                        // then for each element in node_relative_path, add '..' to the link target
                        for (boost::filesystem::path::iterator i (node_relative_path.begin ()); i != node_relative_path.end (); ++i)
                            target_path /= "..";
                        // then add the relative path to the link target
                        // I know, confusing...
                        target_path /= make_path_from_node_path (util::node_path (new_node));
                        target_path /= filename;
                        const boost::filesystem::path symlink_path (env.get_config_root ()->get_root_dir () / config_node->get_source_dir () / node_relative_path / filename);
                        boost::filesystem::create_directories (symlink_path.branch_path ());
                        if (exists (symlink_path)) {
                            remove (symlink_path);
                        }
                        util::make_symbolic_link (target_path.native_file_string (),
                                                  symlink_path);
                    }
                }
            }
            
            void
            fetch::
            execute (const util::node_path &node_path,
                     const boost::shared_ptr<builtins::version_node> &version_node,
                     const build::environment &env)
                    const
            {
                const boost::shared_ptr<const spec::node> package_tree_node (node_path.match (env.get_package_tree_root (),
                                                                                              "mpak:version"));
                assert (package_tree_node);
                
                const boost::shared_ptr<const builtins::version_node> package_tree_version_node (boost::dynamic_pointer_cast<const builtins::version_node> (package_tree_node));
                assert (package_tree_version_node);
                
                if (package_tree_version_node->has_data ("mpak:source")) {
                    boost::shared_ptr<const sources_node_data> sources_data (boost::dynamic_pointer_cast<const sources_node_data> (package_tree_version_node->get_data ("mpak:source")));
                    assert (sources_data);
                    for (sources_node_data::sources_iterator i (sources_data->begin_sources ()); i != sources_data->end_sources (); ++i) {
                        fetch_helper (package_tree_version_node, env, i->second, i->first);
                    }
                }
            }
            
            script::
            ~script (void)
            {
            }
            
            bool
            script::
            is_complete (const util::node_path &node_path,
                         const boost::shared_ptr<const builtins::version_node> &version_node,
                         const build::environment &env)
                    const
            {
                const boost::filesystem::path workdir_path (make_workdir_path (node_path, env));
                const boost::filesystem::path statusdir_path (workdir_path / "status");
                const boost::filesystem::path completefile_path (statusdir_path / (this->name_ + ".complete"));
                if (!exists (completefile_path)) {
                    return false;
                }
                return true;
            }
            
            void
            script::
            execute (const util::node_path &node_path,
                     const boost::shared_ptr<builtins::version_node> &version_node,
                     const build::environment &env)
                const
            {
                if (!this->pre_execute_.empty ()) {
                    this->pre_execute_ (node_path, version_node, env);
                }
                
#define EXEC_FAILURE_CODE 127
                if (!version_node->has_data ("mpak:script")) {
                    return;
                }
                
                const boost::filesystem::path workdir_path (make_workdir_path (node_path, env));
                const boost::shared_ptr<const builtins::config_node> config_node (env.get_config_root ());
                assert (config_node);
                
                const boost::shared_ptr<const script_node_data> script_data (boost::dynamic_pointer_cast<const builtins::script_node_data>
                                                                             (version_node->get_data ("mpak:script")));
                
                const boost::filesystem::path scriptfile_path (workdir_path / "scripts" / this->name_);
                std::fstream script_os (scriptfile_path.native_file_string ().c_str (), std::ios_base::out);
                if (!script_os)
                    throw failure ("could not open script file for writing: " + scriptfile_path.native_file_string ());
                
                if (script_data->has_script ("global")) {
                    script_os << script_data->get_script ("global");
                }
                if (script_data->has_script (this->name_)) {
                    script_os << script_data->get_script (this->name_);
                }
                
                script_os.close ();
                
                // TODO: fix this interpreter setting
                std::string interpreter (script_data->get_interpreter () ? (*script_data->get_interpreter ()) : "/bin/bash");
                std::string::size_type interpreter_slash_pos (interpreter.rfind ('/'));
                std::string interpreter_basename (interpreter.substr (interpreter_slash_pos + 1));
                boost::filesystem::path sourcedir_path (env.get_config_root ()->get_root_dir () / config_node->get_source_dir () / make_path_from_node_path (node_path));
                
                pid_t pid;
                if ((pid = fork ()) == 0) {
                    // child process
                    
                    // do as little as possible here; if it can be
                    // done before the fork, don't do it here, since
                    // it's impossible to throw exceptions back to the
                    // parent process :)
                    
                    // first set up the process environment
                    for (builtins::mpak_node::env_iterator i (config_node->begin_env ()); i != config_node->end_env (); ++i) {
                        setenv (("MPAK_" + i->first).c_str (), i->second.c_str (), 1);
                    }
                    for (builtins::mpak_node::env_iterator i (version_node->begin_env ()); i != version_node->end_env (); ++i) {
                        setenv (("MPAK_" + i->first).c_str (), i->second.c_str (), 1);
                    }
                    setenv ("MPAK_ROOTDIR", env.get_config_root ()->get_root_dir ().native_directory_string ().c_str (), 1);
                    setenv ("MPAK_WORKDIR", workdir_path.native_directory_string ().c_str (), 1);
                    setenv ("MPAK_TMPDIR", (workdir_path / "tmp").native_directory_string ().c_str (), 1);
                    setenv ("MPAK_BUILDDIR", (workdir_path / "build").native_directory_string ().c_str (), 1);
                    setenv ("MPAK_DESTDIR", (workdir_path / "dest").native_directory_string ().c_str (), 1);
                    setenv ("MPAK_SOURCEDIR", sourcedir_path.native_directory_string ().c_str (), 1);
                    setenv ("MPAK_PKGDIR", (env.get_config_root ()->get_root_dir () / env.get_config_root ()->get_package_tree_dir () / make_path_from_node_path (node_path)).native_directory_string ().c_str (), 1);
                    setenv ("MPAK_PKGROOT", env.get_config_root ()->get_package_tree_dir ().native_directory_string ().c_str (), 1);
                    setenv ("MPAK_FULL_VERSION", version_node->get_spec ().get_string ().c_str (), 1);
                    setenv ("MPAK_VERSION", version_node->get_spec ().get_string_no_revision ().c_str (), 1);
                    setenv ("MPAK_PACKAGE_PATH", node_path.parent ().get_string ().c_str (), 1);
                    setenv ("MPAK_PACKAGE", node_path.parent ().leaf ().c_str (), 1);
                    
                    // this seems like a good idea right now...
                    setenv ("HOME", workdir_path.native_directory_string ().c_str (), 1);
                    
                    chdir ((workdir_path / "build").native_directory_string ().c_str ());
                    
                    // then execute the script
                    if (execlp (interpreter.c_str (), interpreter_basename.c_str (), scriptfile_path.native_file_string ().c_str (), NULL) == -1)
                        exit (EXEC_FAILURE_CODE);
                } else {
                    // parent
                    if (pid == -1) {
                        throw failure ("could not fork");
                    }
                    int status;
                    if (waitpid (pid, &status, 0) == -1) {
                        throw failure ("wait failed");
                    }
                    if (status != EXIT_SUCCESS) {
                        if (status == EXEC_FAILURE_CODE) {
                            throw failure ("exec failed, interpreter was " + interpreter);
                        } else {
                            throw failure ("script failed: " + this->name_);
                        }
                    }
                }
                
                // create status file
                std::ofstream status_fs ((workdir_path / "status" / (this->name_ + ".complete")).native_file_string ().c_str (), std::ios_base::out);
                status_fs << "complete" << std::endl;
                status_fs.close ();
                
                if (!this->post_execute_.empty ()) {
                    this->post_execute_ (node_path, version_node, env);
                }
            }
            
            void install_script_pre_execute (const util::node_path &node_path,
                                             const boost::shared_ptr<const builtins::version_node> &,
                                             const build::environment &env)
            {
                boost::filesystem::path workdir (make_workdir_path (node_path, env));
                util::traverse_tree (workdir / "dest", &util::remove, util::traverse_tree_postorder);
            }
            
            merge::
            ~merge (void)
            {
            }
            
            namespace
            {
                class file_stat_collector
                {
                    const boost::filesystem::path root_path_;
                    const boost::shared_ptr<builtins::installed_node_data> installed_node_data_;
                    const util::checksummer::algorithm checksum_algo_;
                    
                public:
                    file_stat_collector (const boost::filesystem::path &root_path,
                                         const boost::shared_ptr<builtins::installed_node_data> &installed_node_data,
                                         util::checksummer::algorithm checksum_algo)
                        : root_path_ (root_path),
                          installed_node_data_ (installed_node_data),
                          checksum_algo_ (checksum_algo)
                    {
                    }
                    
                    void operator () (const boost::filesystem::path &path)
                    {
                        boost::filesystem::path rel_path (boost::filesystem::path ("/") / util::make_relative_path (path, this->root_path_, false));
                        util::file_stat file_stat (path, this->checksum_algo_);
                        this->installed_node_data_->add_file_stat (rel_path, file_stat);
                    }
                };
            }
            
            namespace
            {
                void
                unmerge_files (const boost::shared_ptr<const builtins::installed_node_data> &installed_node_data,
                               const build::environment &env)
                {
                    typedef std::list<boost::filesystem::path> deferred_list_type;
                    deferred_list_type deferred_dirs, deferred_symlinks;
                    
                    boost::reverse_iterator<builtins::installed_node_data::file_stat_iterator> i (installed_node_data->end_file_stats ());
                    boost::reverse_iterator<builtins::installed_node_data::file_stat_iterator> end (installed_node_data->begin_file_stats ());
                    
                    // The ideas behind how unmerging should work was
                    // blatantly, err, appropriated from Gentoo
                    // Linux's portage system, especially the way we
                    // defer directory and symlink removal till later.
                    // Props and thanks to the Gentoo Linux team!
                    for (; i != end; ++i) {
                        const boost::filesystem::path &path (env.get_config_root ()->get_root_dir () / i->first);
                        if (!exists (path) && !symbolic_link_exists (path)) {
                            std::cerr << "--- !exist  " << i->first.native_file_string () << std::endl;
                            continue;
                        }
                        
                        const util::file_stat &stored_file_stat (i->second);
                        const util::file_stat real_file_stat (path, stored_file_stat.checksum_algo);
                        
                        if (stored_file_stat.type != real_file_stat.type) {
                            std::cerr << "--- !type   " << i->first.native_file_string () << std::endl;
                            continue;
                        }
                        
                        switch (real_file_stat.type) {
                        case util::file_stat::type_directory:
                            deferred_dirs.push_back (i->first);
                            break;
                        case util::file_stat::type_chardev:
                        case util::file_stat::type_blkdev:
                            if (stored_file_stat.devnum != real_file_stat.devnum) {
                                std::cerr << "--- !devnum " << i->first.native_file_string () << std::endl;
                                break;
                            }
                            if (util::remove (path)) {
                                if (real_file_stat.type == util::file_stat::type_chardev) 
                                    std::cerr << "<<< chardev " << i->first.native_file_string () << std::endl;
                                else
                                    std::cerr << "<<< blkdev  " << i->first.native_file_string () << std::endl;
                            } else
                                std::cerr << "--- !exist  " << i->first.native_file_string () << std::endl;
                            break;
                        case util::file_stat::type_regular:
                            if (stored_file_stat.mtime != real_file_stat.mtime) {
                                std::cerr << "--- !mtime  " << i->first.native_file_string () << std::endl;
                                break;
                            }
                            if (stored_file_stat.size != real_file_stat.size) {
                                std::cerr << "--- !size   " << i->first.native_file_string () << std::endl;
                                break;
                            }
                            if (stored_file_stat.checksum != real_file_stat.checksum) {
                                std::cerr << "--- !chksum " << i->first.native_file_string () << std::endl;
                                break;
                            }
                            if (util::remove (path))
                                std::cerr << "<<< regular " << i->first.native_file_string () << std::endl;
                            else
                                std::cerr << "--- !exist  " << i->first.native_file_string () << std::endl;
                            break;
                        case util::file_stat::type_symlink:
                            deferred_symlinks.push_back (i->first);
                            break;
                        case util::file_stat::type_fifo:
                            if (util::remove (path))
                                std::cerr << "<<< fifo    " << i->first.native_file_string () << std::endl;
                            else
                                std::cerr << "--- !exist  " << i->first.native_file_string () << std::endl;
                            break;
                        case util::file_stat::type_socket:
                            if (util::remove (path))
                                std::cerr << "<<< socket  " << i->first.native_file_string () << std::endl;
                            else
                                std::cerr << "--- !exist  " << i->first.native_file_string () << std::endl;
                            break;
                        }
                    }
                    
                    bool removed_smth (true);
                    while (removed_smth && !(deferred_dirs.empty () && deferred_symlinks.empty ())) {
                        removed_smth = false;
                        
                        for (deferred_list_type::iterator i (deferred_symlinks.begin ()); i != deferred_symlinks.end (); ++i) {
                            const boost::filesystem::path &path (env.get_config_root ()->get_root_dir () / *i);
                            
                            if (!exists (path) && symbolic_link_exists (path)) { // if this is a dangling symlink
                                try {
                                    util::remove (path);
                                    std::cerr << "<<< symlink " << i->native_file_string () << std::endl;
                                    removed_smth = true;
                                } catch (boost::filesystem::filesystem_error &e) {
                                    std::cerr << "!!! error   " << i->native_file_string () << std::endl;
                                }
                                i = deferred_symlinks.erase (i);
                            }
                        }
                        
                        for (deferred_list_type::iterator i (deferred_dirs.begin ()); i != deferred_dirs.end (); ++i) {
                            const boost::filesystem::path &path (env.get_config_root ()->get_root_dir () / *i);
                            
                            if (is_empty (path)) {
                                try {
                                    util::remove (path);
                                    std::cerr << "<<< dir     " << i->native_directory_string () << std::endl;
                                    removed_smth = true;
                                } catch (boost::filesystem::filesystem_error &e) {
                                    std::cerr << "!!! error   " << i->native_directory_string () << std::endl;
                                }
                                i = deferred_dirs.erase (i);
                            }
                        }
                        
                    }
                    
                    for (deferred_list_type::const_iterator i (deferred_symlinks.begin ()); i != deferred_symlinks.end (); ++i) {
                        const boost::filesystem::path &path (env.get_config_root ()->get_root_dir () / *i);
                        if (exists (path)) {
                            std::cerr << "--- !target " << i->native_file_string () << std::endl;
                        } else if (symbolic_link_exists (path)) {
                            std::cerr << "!!! error   " << i->native_file_string () << std::endl;
                        } else {
                            std::cerr << "--- !exist  " << i->native_file_string () << std::endl;
                        }
                    }
                    
                    for (deferred_list_type::const_iterator i (deferred_dirs.begin ()); i != deferred_dirs.end (); ++i) {
                        const boost::filesystem::path &path (env.get_config_root ()->get_root_dir () / *i);
                        if (exists (path) && !is_empty (path)) {
                            std::cerr << "--- !empty  " << i->native_directory_string () << std::endl;
                        } else if (!exists (path)) {
                            std::cerr << "--- !exist  " << i->native_directory_string () << std::endl;
                        } else {
                            std::cerr << "!!! error   " << i->native_directory_string () << std::endl;
                        }
                    }
                }
            }
            
            namespace
            {
                class update_mtime
                {
                    struct timeval tv_;
                    
                public:
                    update_mtime (struct timeval tv)
                        : tv_ (tv)
                    {
                    }
                    
                    void
                    operator () (const boost::filesystem::path &path)
                    {
                        struct stat stat;
                        lstat (path.native_file_string ().c_str (), &stat);
                        struct timeval tv[2] = { { 0, 0 },
                                                 this->tv_ };
                        if (!S_ISLNK (stat.st_mode))
                            utimes (path.native_file_string ().c_str (), tv);
                    }
                };
            }
            
            void
            merge::
            execute (const util::node_path &node_path,
                     const boost::shared_ptr<builtins::version_node> &version_node,
                     const build::environment &env)
                    const
            {
                const boost::shared_ptr<const builtins::config_node> config_node (env.get_config_root ());
                assert (config_node);
                
                const boost::filesystem::path &root_dir (config_node->get_root_dir ());
                if (!exists (root_dir) || !is_directory (root_dir)) {
                    throw failure ("root path " + root_dir.native_directory_string () + " does not exist");
                }
                
                boost::filesystem::path workdir (make_workdir_path (node_path, env));
                boost::filesystem::path destdir (workdir / "dest");
                
                // first update mtimes
                struct timeval tv;
                gettimeofday (&tv, 0);
                util::traverse_tree (destdir, update_mtime (tv), util::traverse_tree_postorder);
                
                // collect the file_stats
                boost::shared_ptr<builtins::installed_node_data> installed_node_data (new builtins::installed_node_data);
                util::traverse_tree (destdir, file_stat_collector (destdir, installed_node_data, util::checksummer::algorithm_default));
                version_node->add_data ("mpak:installed", installed_node_data);
                
                // copy the files into the filesystem
                util::duplicate_contents (destdir, root_dir);
                
                // set timestamp
                boost::shared_ptr<builtins::timestamp_node_data> timestamp_node_data (new timestamp_node_data);
                version_node->add_data ("mpak:timestamp", timestamp_node_data);
                
                // now insert the version_node into the installed tree
                
                // we expect the path passed to be of the form:
                //   category1:category2:...:categoryN:package:1.0
                // so trim off last 2 elements.
                assert (std::distance (node_path.begin_elements (), node_path.end_elements ()) >= 3);
                util::node_path::element_const_iterator element (node_path.begin_elements ());
                util::node_path::element_const_iterator end_of_categories (node_path.end_elements ());
                --end_of_categories;
                --end_of_categories;
                
                // first descend all categories, creating categories as necessary
                boost::shared_ptr<builtins::category_node> category_node (env.get_installed_tree_root ());
                boost::shared_ptr<builtins::category_node> child_category_node;
                
                for (; element != end_of_categories; ++element) {
                    if (category_node->has_child ("mpak:category", *element)) {
                        boost::shared_ptr<spec::node> child_node (category_node->get_child ("mpak:category", *element));
                        child_category_node = boost::dynamic_pointer_cast<builtins::category_node> (child_node);
                        assert (child_category_node);
                    } else {
                        child_category_node.reset (new builtins::category_node ("mpak:category", *element,
                                                                                *env.get_installed_tree_context ()));
                        category_node->add_child (child_category_node);
                    }
                    category_node->set_dirty (true);
                    category_node = child_category_node;
                }
                
                // get/create package node
                boost::shared_ptr<builtins::package_node> package_node;
                if (category_node->has_child ("mpak:package", *element)) {
                    boost::shared_ptr<spec::node> child_node (category_node->get_child ("mpak:package", *element));
                    package_node = boost::dynamic_pointer_cast<builtins::package_node> (child_node);
                    assert (package_node);
                } else {
                    package_node.reset (new builtins::package_node ("mpak:package", *element,
                                                                    *env.get_installed_tree_context ()));
                    category_node->add_child (package_node);
                }
                category_node->set_dirty (true);
                ++element;
                
                // insert new version_node, while saving old one.
                boost::shared_ptr<builtins::version_node> old_version_node;
                if (package_node->has_child ("mpak:version", *element)) {
                    boost::shared_ptr<spec::node> old_node (package_node->get_child ("mpak:version", *element));
                    old_version_node = boost::dynamic_pointer_cast<builtins::version_node> (old_node);
                    assert (old_version_node);
                    
                    package_node->remove_child ("mpak:version", *element);
                }
                package_node->add_child (version_node);
                package_node->set_dirty (true);
                version_node->set_dirty (true);
                
                env.get_installed_tree_context ()->get_pickler ()->remove ("mpak:version", node_path);
                env.get_installed_tree_context ()->write_nodes (env.get_installed_tree_root ());
                
                if (old_version_node && old_version_node->has_data ("mpak:installed")) {
                    boost::shared_ptr<const builtins::installed_node_data> old_installed_node_data (boost::dynamic_pointer_cast<const builtins::installed_node_data> (old_version_node->get_data ("mpak:installed")));
                    assert (old_installed_node_data);
                    
                    unmerge_files (old_installed_node_data, env);
                }
            }
            
            unmerge::
            ~unmerge (void)
            {
            }
            
            void
            unmerge::
            execute (const util::node_path &node_path,
                     const boost::shared_ptr<builtins::version_node> &version_node,
                     const build::environment &env)
                    const
            {
                if (version_node->has_data ("mpak:installed")) {
                    boost::shared_ptr<const builtins::installed_node_data> installed_node_data (boost::dynamic_pointer_cast<const builtins::installed_node_data> (version_node->get_data ("mpak:installed")));
                    assert (installed_node_data);
                    unmerge_files (installed_node_data, env);
                }
                
                boost::shared_ptr<spec::node> installed_tree_version_node (node_path.match_nc (env.get_installed_tree_root (), "mpak:version"));
                boost::optional<const boost::weak_ptr<spec::node> > opt_weak_package_node (installed_tree_version_node->get_parent ());
                assert (opt_weak_package_node);
                boost::shared_ptr<spec::node> package_node (*opt_weak_package_node);
                assert (package_node->get_type () == "mpak:package");
                
                package_node->remove_child (installed_tree_version_node->get_type (), installed_tree_version_node->get_name ());
                
                boost::shared_ptr<spec::node> node (package_node);
                boost::optional<boost::weak_ptr<spec::node> > opt_weak_parent_node;
                while (opt_weak_parent_node = node->get_parent ()) {
                    boost::shared_ptr<spec::node> parent_node (*opt_weak_parent_node);
                    if (node->children_empty ()) {
                        const util::node_path child_node_path (node);
                        const std::string child_node_type (node->get_type ());
                        parent_node->remove_child (node->get_type (), node->get_name ());
                        env.get_installed_tree_context ()->get_pickler ()->remove (child_node_type, child_node_path);
                        parent_node->set_dirty (true);
                    } else {
                        node->set_dirty (true);
                    }
                    node = parent_node;
                }
                
                env.get_installed_tree_context ()->get_pickler ()->remove ("mpak:version", node_path);
                env.get_installed_tree_context ()->write_nodes (env.get_installed_tree_root ());
            }
        }
    }
}
