// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/libmpak/mpak/build/manager.cc,v 1.5 2004/07/06 17:20:31 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <config.h>

#include <mpak/build/manager.hh>
#include <mpak/build/stage.hh>
#include <mpak/build/environment.hh>
#include <mpak/spec/node.hh>
#include <mpak/builtins/mpak_node.hh>
#include <mpak/builtins/version_node.hh>
#include <mpak/builtins/category_node.hh>
#include <mpak/builtins/config_node.hh>

#include <boost/shared_ptr.hpp>
#include <boost/weak_ptr.hpp>
#include <boost/optional.hpp>
#include <boost/graph/adjacency_list.hpp>
#include <boost/graph/topological_sort.hpp>
#include <boost/graph/depth_first_search.hpp>
#include <boost/graph/visitors.hpp>

#include <string>
#include <map>
#include <deque>
#include <stdexcept>

namespace mpak
{
    namespace build
    {
        namespace
        {
            void
            merge_data (const boost::shared_ptr<const spec::node> &source,
                        const boost::shared_ptr<spec::node> &target)
            {
                boost::shared_ptr<const builtins::mpak_node> source_mpak_node (boost::dynamic_pointer_cast<const builtins::mpak_node> (source));
                assert (source_mpak_node);
                boost::shared_ptr<builtins::mpak_node> target_mpak_node (boost::dynamic_pointer_cast<builtins::mpak_node> (target));
                assert (target_mpak_node);
                
                target_mpak_node->set_description (source_mpak_node->get_description ());
                for (builtins::mpak_node::env_iterator i (source_mpak_node->begin_env ()); i != source_mpak_node->end_env (); ++i) {
                    target_mpak_node->set_env (i->first, i->second);
                }
                
                for (spec::node::data_const_iterator i (source->begin_data ()); i != source->end_data (); ++i) {
                    if (target->has_data (i->first)) {
                        i->second->merge (target->get_data (i->first));
                    } else {
                        boost::shared_ptr<spec::node_data> node_data (i->second->clone ());
                        target->add_data (i->first, node_data);
                    }
                }
                
            }
            
            void
            collect_settings (const boost::shared_ptr<const builtins::mpak_node> &root,
                              const util::node_path &node_path,
                              const boost::shared_ptr<builtins::version_node> &version_node,
                              bool must_exist)
            {
                assert (2 <= node_path.elements_size ());
                util::node_path::elements_size_type num_category_elements (node_path.elements_size () - 2);
                
                merge_data (root, version_node);
                
                boost::shared_ptr<const spec::node> node (root);
                for (util::node_path::elements_size_type n (0); n < num_category_elements; ++n) {
                    if (node->has_child ("mpak:category", node_path.element_at (n))) {
                        node = node->get_child ("mpak:category", node_path.element_at (n));
                    } else {
                        if (must_exist) {
                            assert (false);
                        } else {
                            return;
                        }
                    }
                    
                    merge_data (node, version_node);
                }
                
                if (node->has_child ("mpak:package", node_path.element_at (num_category_elements))) {
                    node = node->get_child ("mpak:package", node_path.element_at (num_category_elements));
                } else {
                    if (must_exist) {
                        assert (false);
                    } else {
                        return;
                    }
                }
                
                merge_data (node, version_node);
                
                if (node->has_child ("mpak:version", node_path.element_at (num_category_elements + 1))) {
                    node = node->get_child ("mpak:version", node_path.element_at (num_category_elements + 1));
                } else {
                    if (must_exist) {
                        assert (false);
                    } else {
                        return;
                    }
                }
                
                merge_data (node, version_node);
            }
            
            // shamelessly ripped from the BGL example program file_dependencies.cpp
            struct cycle_detector
                : public boost::dfs_visitor<>
            {
                cycle_detector (bool& has_cycle) 
                    : has_cycle_ (has_cycle)
                {
                }
                
                template <class edge_type_, class graph_type_>
                void back_edge(edge_type_, graph_type_ &)
                {
                    this->has_cycle_ = true;
                }
                
            protected:
                bool &has_cycle_;
            };
        }
        
        void
        manager::
        execute_stage (const std::string &name,
                       const util::node_path &node_path,
                       const environment &env)
            const
        {
            typedef boost::property<boost::vertex_name_t, std::string,
                boost::property<boost::vertex_color_t, boost::default_color_type,
                boost::property<boost::vertex_index_t, unsigned> > > stage_graph_vertex_property_type;
            typedef boost::adjacency_list<boost::vecS, boost::vecS, boost::directedS,
                                          stage_graph_vertex_property_type> stage_graph_type;
            typedef boost::graph_traits<stage_graph_type>::vertex_descriptor stage_graph_vertex_descriptor_type;
            typedef std::map<std::string, stage_graph_vertex_descriptor_type> stage_vertex_map_type;
            
            stage_graph_type stage_graph;
            stage_vertex_map_type stage_vertex_map;
            std::deque<std::string> stage_queue;
            
            stage_queue.push_back (name);
            bool success (stage_vertex_map.insert (std::make_pair (name, add_vertex (name, stage_graph))).second);
            assert (success);
            
            while (!stage_queue.empty ()) {
                const std::string &dependor_stage (stage_queue.front ());
                
                if (this->stage_map_.find (dependor_stage) == this->stage_map_.end ())
                    throw std::logic_error ("stage " + dependor_stage + " not found");
                
                stage_vertex_map_type::const_iterator dependor_vertex_i (stage_vertex_map.find (dependor_stage));
                assert (dependor_vertex_i != stage_vertex_map.end ());
                
                for (dependency_map_type_::const_iterator i (this->dependency_map_.lower_bound (dependor_stage)),
                         end (this->dependency_map_.upper_bound (dependor_stage));
                     i != end; ++i) {
                    const std::string &dependee_stage (i->second);
                    
                    stage_vertex_map_type::const_iterator dependee_vertex_i (stage_vertex_map.find (dependee_stage));
                    if (dependee_vertex_i == stage_vertex_map.end ()) {
                        boost::tie (dependee_vertex_i, success) = stage_vertex_map.insert (std::make_pair (dependee_stage,
                                                                                                           add_vertex (dependee_stage,
                                                                                                                       stage_graph)));
                        assert (success);
                        
                        stage_queue.push_back (dependee_stage);
                    }
                    
                    boost::add_edge (dependor_vertex_i->second, dependee_vertex_i->second, stage_graph);
                }
                
                stage_queue.pop_front ();
            }
            
            bool has_cycle (false);
            boost::depth_first_search (stage_graph, boost::visitor (cycle_detector (has_cycle)));
            assert (!has_cycle);
            
            typedef std::list<stage_graph_vertex_descriptor_type> sort_output_type;
            sort_output_type sort_output;
            boost::topological_sort (stage_graph, std::back_inserter (sort_output));
            
            boost::property_map<stage_graph_type, boost::vertex_name_t>::type vertex_name_map (get (boost::vertex_name, stage_graph));
            
            for (sort_output_type::const_iterator stage_vertex_i (sort_output.begin ());
                 stage_vertex_i != sort_output.end (); ++stage_vertex_i) {
                const std::string &stage_name (vertex_name_map[*stage_vertex_i]);
                
                stage_map_type_::const_iterator i (this->stage_map_.find (stage_name));
                assert (i != this->stage_map_.end ());
                
                // then execute stage
                boost::shared_ptr<const builtins::mpak_node> source_tree_root;
                boost::shared_ptr<spec::context> source_tree_context;
                switch (i->second.second) {
                case stage_use_package_tree:
                    source_tree_root = env.get_package_tree_root ();
                    source_tree_context = env.get_package_tree_context ();
                    break;
                case stage_use_installed_tree:
                    source_tree_root = env.get_installed_tree_root ();
                    source_tree_context = env.get_installed_tree_context ();
                    break;
                default:
                    assert (false);
                }
                
                boost::shared_ptr<const spec::node> source_tree_node (node_path.match (source_tree_root, "mpak:version"));
                if (!source_tree_node)
                    throw failure ("could not find node " + node_path.get_string ());
                
                boost::shared_ptr<const builtins::version_node> source_tree_version_node (boost::dynamic_pointer_cast<const builtins::version_node> (source_tree_node));
                assert (source_tree_version_node);
                
                boost::shared_ptr<builtins::version_node> version_node (new builtins::version_node ("mpak:version",
                                                                                                    source_tree_node->get_name (),
                                                                                                    *source_tree_context));
                version_node->set_slot (source_tree_version_node->get_slot ());
                collect_settings (env.get_config_root (), node_path, version_node, false);
                collect_settings (source_tree_root, node_path, version_node, true);
                
                const boost::shared_ptr<const stage> &stage (i->second.first);
                if ((i->first == name) || !stage->is_complete (node_path, version_node, env)) {
                    stage->execute (node_path, version_node, env);
                }
            }
        }
    }
}
