// -*- mode: c++; indent-tabs-mode: nil; c-basic-offset: 4 -*-
// $Header: /home/pgavin/cvsroot/mpak/include/mpak/util/node_path.hh,v 1.6 2004/07/07 01:54:57 pgavin Exp $
// mpak - the advanced package manager
// Copyright (C) 2003 Peter Gavin
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef __MPAK__UTIL__NODE_PATH_HH__
#define __MPAK__UTIL__NODE_PATH_HH__

#include <mpak/defs.hh>
#include <mpak/spec/fwd.hh>
#include <mpak/util/cow_vector.hh>

#include <boost/shared_ptr.hpp>

#include <vector>
#include <string>
#include <stdexcept>

namespace mpak
{
    namespace util
    {
        class node_path
        {
        public:
            class failure
                : public std::runtime_error
            {
            public:
                inline failure (const std::string &what)
                    : runtime_error (what)
                {
                }
            };
            
        private:
            typedef cow_vector<std::string> element_vector_;
            element_vector_ elements_;
            
        public:
            typedef element_vector_::size_type elements_size_type;
            typedef element_vector_::iterator element_iterator;
            typedef element_vector_::const_iterator element_const_iterator;
            
            node_path (void)
                : elements_ ()
            {
            }
            
            node_path (const std::string &path_string)
                : elements_ ()
            {
                this->parse_string (path_string);
            }
            
            node_path (const node_path &that)
                : elements_ (that.elements_)
            {
            }
            
        private:
            node_path (const element_vector_ &elements)
                : elements_ (elements)
            {
            }
            
        public:
            explicit
            node_path (const boost::shared_ptr<const spec::node> &node)
                : elements_ ()
            {
                this->read_node (node);
            }
            
            void swap (node_path &that)
            {
                this->elements_.swap (that.elements_);
            }
            
            node_path &
            operator= (const node_path &that)
            {
                node_path new_node_path (that);
                this->swap (new_node_path);
                return *this;
            }
            
            void
            clear (void)
            {
                node_path new_node_path;
                this->swap (new_node_path);
            }
            
            bool empty (void)
                const
            {
                return this->elements_.empty ();
            }
            
            void
            push_back_element (const std::string &element)
            {
                this->elements_.push_back (element);
            }
            
            const element_iterator
            begin_elements (void)
            {
                return this->elements_.begin ();
            }
            
            const element_const_iterator
            begin_elements (void)
                const
            {
                return this->elements_.begin ();
            }
            
            const element_iterator
            end_elements (void)
            {
                return this->elements_.end ();
            }
            
            const element_const_iterator
            end_elements (void)
                const
            {
                return this->elements_.end ();
            }
            
            const std::string &
            element_at (elements_size_type index)
                const
            {
                return this->elements_.at (index);
            }
            
            const elements_size_type
            elements_size (void)
                const
            {
                return this->elements_.size ();
            }
            
            const std::string &
            leaf (void)
                const
            {
                return this->elements_.back ();
            }
            
            const node_path
            parent (void)
                const
            {
                if (!this->elements_.size ())
                    throw failure ("root path has no parent");
                return node_path (element_vector_ (this->elements_.begin (), this->elements_.end () - 1));
            }
            
            void
            parse_string (const std::string &string);
            
            const std::string get_string (void) const;
            
            const boost::shared_ptr<const spec::node>
            match (const boost::shared_ptr<const spec::node> &root_node, const std::string &node_type)
                const;
            
            const boost::shared_ptr<spec::node>
            match_nc (const boost::shared_ptr<spec::node> &root_node, const std::string &node_type)
                const;
            
            void
            read_node (const boost::shared_ptr<const spec::node> &node);
            
            bool
            valid (void)
                const
            {
                return true;
            }
            
            bool operator < (const node_path &that)
                const
            {
                return this->elements_ < that.elements_;
            }
            
            bool operator == (const node_path &that)
                const
            {
                return this->elements_ == that.elements_;
            }
            
            bool operator > (const node_path &that)
                const
            {
                return this->elements_ > that.elements_;
            }
            
            bool operator != (const node_path &that)
                const
            {
                return this->elements_ != that.elements_;
            }
            
            bool operator >= (const node_path &that)
                const
            {
                return this->elements_ >= that.elements_;
            }
            
            bool operator <= (const node_path &that)
                const
            {
                return this->elements_ <= that.elements_;
            }
            
        private:
            const boost::shared_ptr<const spec::node>
            match_helper_ (const boost::shared_ptr<const spec::node> &category_node,
                           const element_vector_::size_type element_index,
                           const std::string &node_type)
                const;
            
            const boost::shared_ptr<spec::node>
            match_helper_nc_ (const boost::shared_ptr<spec::node> &category_node,
                           const element_vector_::size_type element_index,
                           const std::string &node_type)
                const;
            
            void
            read_node_helper_ (const boost::shared_ptr<const spec::node> &node);
        };
    }
}

namespace std
{
    inline void
    swap (mpak::util::node_path &node_path1, mpak::util::node_path &node_path2)
    {
        node_path1.swap (node_path2);
    }
}

#endif // ifndef __MPAK__UTIL__NODE_PATH_HH__
