/*******************************************************************
 *                   Mathematical Object Library                   *
 *  Class polynom : declaration for polynoms and its derivatives   *
 *                      simula.plus@cemes.fr                       *
 *	               GNU/linux version 3.2.0                     *
 *              software under General Public License              *
 *******************************************************************
 * copyright © 2002,2003,2004,2005,2006,2008,2009,2011,2012 COLLARD Christophe
 * copyright © 2002,2003,2004,2005,2006,2008,2009,2011,2012 Centre National de la Recherche Scientifique
 * copyright © 2002,2003,2004,2005,2006,2008,2009,2011 Arts et Métiers ParisTech
 * copyright © 2002,2003,2004,2005,2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2002,2003,2004,2005,2006,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2002,2003,2004,2005,2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2011 Laboratoire d'Etude des Microstructures et de Mécanique des Matériaux (LEM3 - CNRS)
 * copyright © 2011,2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 *******************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::polynom
    \brief polynom library \n

    \htmlonly 
    <FONT color="#838383">

    polynom belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006, 2008, 2009, 2011, 2012 Christophe COLLARD \n
            copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2008, 2009, 2011, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2008, 2009, 2011 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	    copyright \htmlonly &#169; 2011 Laboratoire d'Etude des Microstructures et de M&#233;canique des Mat&#233;riaux (LEM3 - CNRS) \endhtmlonly \n
            copyright \htmlonly &#169; 2011, 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.2.0
    \date 2002-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type polynom
#endif

#ifndef __polynoms_hpp
#define __polynoms_hpp


#ifndef __stdlib_h
#include <stdlib.h>
#endif

#ifndef __iostream
#include <iostream>
#endif

#ifndef __fstream
#include <fstream>
#endif

#ifndef __assert_h
#include <assert.h>
#endif

#ifndef __math_h
#include <math.h>
#endif

#ifndef __parameters_h
#include "parameters.h"
#endif

#ifndef __maths_hpp
#include "maths.hpp"
#endif

#ifndef __vectors_hpp
#include "vectors.hpp"
#endif

using namespace std;

namespace mol
{


//=================================================
template <class T> class polynom : public vector<T>
//=================================================
{
  // Px[0] = coef in X^0 , ... , Px[n] = coef in X^n
  using vector<T>::size;
  using vector<T>::p;

  public:    
    polynom () : vector<T>() {}       // default constructor
    polynom (int n, bool init=true, T val=0) : vector<T> (n+1,init,val) {}  // constructor with the polynom degree
    polynom (const polynom<T>& P) : vector<T>(P) {}   // constructor with defaults values for the polynom coefficients
    polynom (polynom<T>* P) : vector<T>(P) {}         // copy constructor with temporary objects
    ~polynom () {}                 // destructor

    T* operator () () {return p;}
    void assign (int);  // allocates memory for the polynom but doen't initilize to zero its components (use it with great care)
    int dim () const {return 1;} // returns space dimension
    T& operator [] (int) const;  // returns the n_th coefficient of the polynom P(X) (coef in X^n)
    template <class Tf> friend polynom<Tf> operator + (const polynom<Tf>&, const polynom<Tf>&); // overloads operator + for polynoms
    template <class Tf> friend polynom<Tf> operator - (const polynom<Tf>&, const polynom<Tf>&); // overloads operator - for polynoms
    template <class Tf> friend polynom<Tf> operator * (const polynom<Tf>&, const polynom<Tf>&); // defines the scalar product
    void  operator += (const polynom<T>&);
    void  operator -= (const polynom<T>&);
    template <class Tf> friend ostream& operator << (ostream&, const polynom<Tf>&);
    //    friend  istream& operator >> <T> (istream&, const polynom<T>&){exit(1);}
    virtual void permute (int,int);  // elements permutation

    // new methods
    int degree () {return size-1;}
    T operator () (T) const;          // returns the value Px(X)
    virtual polynom<T> D_dx (int=1);  // returns the n_th derivative
    virtual T d_dx (T);               // returns the derivative value at x
    virtual T dn_dx (int, T);         // returns the n_th derivative value at x
    vector<T> operator () (const vector<T>&) const;  // return the value Px(X) where X is a one element vector
    virtual vector<T> d_dx (const vector<T>&);       // returns the derivative value at x
    virtual vector<T> dn_dx (int, const vector<T>&); // returns the n_th derivative value at x
    template <class Tf> friend polynom<Tf> abs (const polynom<Tf>&);  // give the absolute value of a polynom

    // invalidates the following methods
    virtual long double norm () const;
    virtual T norminf () const;
    template <class Tf> friend void order (polynom<Tf>&);
    template <class Tf> friend polynom<Tf> subvector (const polynom<Tf>&, int, int);
    template <class Tf> friend polynom<Tf> merge (const polynom<Tf>&, const polynom<Tf>&);
};


//=====Private methods for polynoms===========================================


//=====Public methods for polynoms============================================


//-----Overloaded methods for polynoms----------------------------------------


//------------------------------------------------
template <class T> void polynom<T>::assign (int n)
//------------------------------------------------
{
  assert (n>0);
  vector<T>::assign(n+1);
}


//---------------------------------------------------------
template <class T> T& polynom<T>::operator [] (int n) const
//---------------------------------------------------------
{
  assert (n>=0);
  assert (n<size); // polynom 0->n and its coefficients are stored 1->n+1
  return vector<T>::operator[](n+1);
}


//-------------------------------------------------------------------------------------
template <class Tf> polynom<Tf> operator + (const polynom<Tf>& a, const polynom<Tf>& b)
//-------------------------------------------------------------------------------------
{ 
  assert (a.size && b.size);
  polynom<Tf> v;

  if (a.size > b.size)
    { v = a;
      v += b;
    }
  else
    { v = b;
      v += a;
    }

  return v;
}


//-------------------------------------------------------------------------------------
template <class Tf> polynom<Tf> operator - (const polynom<Tf>& a, const polynom<Tf>& b)
//-------------------------------------------------------------------------------------
{
  assert (a.size && b.size);
  polynom<Tf> v;

  if (a.size > b.size)
    { v = a;
      v -= b;
    }
  else
    { v = b;
      v *= -1;
      v += a;
    }

  return v;
}


//-------------------------------------------------------------------------------------
template <class Tf> polynom<Tf> operator * (const polynom<Tf>& a, const polynom<Tf>& b)
//-------------------------------------------------------------------------------------
{
  assert (a.size && b.size);
  polynom<Tf> P(a.size+b.size-2);

  for (int i=0; i<a.size; i++)
    for (int j=0; j<b.size; j++)
      P.p[i+j] += a.p[i] * b.p[j];

  return P;
}


//-------------------------------------------------------------------
template <class T> void polynom<T>::operator += (const polynom<T>& v)
//-------------------------------------------------------------------
{
  assert (size >= v.size);
  for (int i=0; i<v.size; i++)
    p[i] += v.p[i];
}


//-------------------------------------------------------------------
template <class T> void polynom<T>::operator -= (const polynom<T>& v)
//-------------------------------------------------------------------
{
  assert (size>=v.size);
  for (int i=0; i<v.size; i++)
    p[i] -= v.p[i];
}


//-------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, const polynom<Tf>& p)
//-------------------------------------------------------------------------
{
  assert (p.size);
  int deg = p.size-1;

  s << "P(X) = ";
  if (!deg) s << p[0];
  else 
    { if (p[deg]!=1) s << p[deg];
      s << " X^" << deg;
      for (deg--; deg>0; deg--)
	if (abs(p[deg]) > epsilon)
	  { if ( p[deg] > 0) s << "  +  ";
	    else s << "  -  ";
	    if (abs(p[deg]) != 1) s << abs(p[deg]);
	    s << " X";
	    if (deg != 1) s << "^" << deg;
	  }
      if (p[0])
	{ if ( p[0] > 0) s << "  +  ";
	  else s << "  -  ";
	  s << abs(p[0]);
	}
    }
  s << endl;

  return s;
}


//--------------------------------------------------------
template <class T> void polynom<T>::permute (int i, int j)
//--------------------------------------------------------
{
  vector<T>::permute(i+1,j+1);
}


//-----New methods for polynoms------------------------------------------------


//------------------------------------------------------
template <class T> T polynom<T>::operator () (T x) const
//------------------------------------------------------
{
  assert (size);

  int deg = size-1;
  T value=p[0], xi=x;

  for (int i=1; i<=deg; i++, xi*=x)
    value += p[i] * xi;

  return value;
}


//----------------------------------------------------
template <class T> polynom<T> polynom<T>::D_dx (int n)
//----------------------------------------------------
{
  assert (size);  // polynom must exists
  assert (n>0);
  int deg = size-1;

  polynom<T> Qx;

  if (deg<n) Qx =& polynom<T>(0); // (P(x) = constant) => (P'(X) = 0)
  else
    { Qx = deg-n; // derivative polynom degree
      long int coef = factorial (n);
      for (int i=n; i<=deg; coef = coef/((++i)-n) * i)
	Qx.p[i-n] = p[i] * coef;
    }

  return Qx;
}


//---------------------------------------------
template <class T> T polynom<T>::d_dx (T value)
//---------------------------------------------
{
  return D_dx()(value);
}


//-----------------------------------------------------
template <class T> T polynom<T>::dn_dx (int n, T value)
//-----------------------------------------------------
{
  return D_dx(n)(value);
}


//-----------------------------------------------------------------------------
template <class T> vector<T> polynom<T>::operator () (const vector<T>& v) const
//-----------------------------------------------------------------------------
{
  assert (v.dim()==1);
  vector<T> u(1,true,(*this)(v[1]));
  return u;
}


//----------------------------------------------------------------
template <class T> vector<T> polynom<T>::d_dx (const vector<T>& v)
//----------------------------------------------------------------
{
  assert (v.dim()==1);
  vector<T> u(1,true,D_dx()(v[1]));
  return u;
}


//------------------------------------------------------------------------
template <class T> vector<T> polynom<T>::dn_dx (int n, const vector<T>& v)
//------------------------------------------------------------------------
{
  assert (v.dim()==1);
  vector<T> u(1,true,D_dx(n)(v[1]));
  return u;
}


/*
//--------------------------------------------------
template <class T> void polynom<T>::Legendre(int dg)
//--------------------------------------------------
{
  // computes the Legendre polynom using the formula : Ln(X) = (2n-1)/n X Ln-1(X) - (n-1)/n Ln-2 = a_n X^n + a_(n-1) X^(n-1) + ...
  vector<T> v(dg);
  P alpha, beta, val; //     alpha=(2n-1)/n     beta=(n-1)/n

  if (deg<0)  AssignMemory (dg);
  else Px[0] = v; // initialisation of the polynom coefficients

  (*this)(0) = 1; // Lo(X) = 1
  for (int n=1; n<=dg; n++)
    { alpha = 2 - 1 / (P) n;
      beta = 1 - 1 / (P) n;
      // Ln(X) = (2n-1)/n * b_(n-1) X^n + (2n-1)/n * b_(n-2) X^(n-1) + [ (2n-1)/n * b_(n-3) - (n-1)/n c_(n-2)] X^(n-2) + ...
      (*this)(n) = alpha * (*this)(n-1); // ....................... a_n (empty) = (2n-1)/n * b_(n-1) X^n
      v[dg-n+1] = (*this)(n-1); // ..................................... v[n] (empty) = b_(n-1)
      if (n>1) (*this)(n-1) = alpha * (*this)(n-2); // .................. a_(n-1) = (2n-1)/n * b_(n-2)
      for (int j=n-2; j>=1; j--)
	{ val = (*this)(j); // ................................. val = c_j
	  (*this)(j) = alpha * (*this)(j-1) - beta * v[dg-j]; // .. a_(j) = (2n-1)/n * b_(j-1) - (n-1)/n c_(j)
	  v[dg-j] = val; // ....................................... c_j = b_j
	}
      val = (*this)(0);
      (*this)(0) = - beta * v[dg]; // ............................... a_0 = (n-1)/n c_0
      v[dg] = val;
    }
}

*/


//--------------------------------------------------------
template <class Tf> polynom<Tf> abs (const polynom<Tf>& P)
//--------------------------------------------------------
{ 
  assert (P.size);
  polynom<Tf> Q(P.size-1);  // allocates memory without initializing the vector components to zero

  for (int i=0; i<P.size; i++)
    Q.p[i] = abs(P.p[i]);

  return Q;
}


//-----Unvalidates some methods for polynoms-----------------------------------


//------------------------------------------------------
template <class T> long double polynom<T>::norm () const
//------------------------------------------------------
{
  cout << "ERROR : 'long double  polynom<T>::norm () const' is not available for polynoms" << endl;
  exit(1);
}


//-----------------------------------------------
template <class T> T polynom<T>::norminf () const
//-----------------------------------------------
{
  cout << "ERROR : 'T polynom<T>::norminf () const' is not available for polynoms" << endl;
  exit(1);
}


//-------------------------------------------
template <class Tf> void order (polynom<Tf>&)
//-------------------------------------------
{
  cout << "ERROR : 'void order(polynom<T>&)' is not available for polynoms" << endl;
  exit(1);
}


//----------------------------------------------------------------------
template <class Tf> polynom<Tf> subvector (const polynom<Tf>&, int, int)
//----------------------------------------------------------------------
{
  cout << "ERROR : 'polynom<T> subvector(const polynom<T>&, int, int)' is not available for polynoms" << endl;
  exit(1);
}


//----------------------------------------------------------------------------
template <class Tf> polynom<Tf> merge (const polynom<Tf>&, const polynom<Tf>&)
//----------------------------------------------------------------------------
{
  cout << "ERROR : 'polynom<T> merge(const polynom<T>&, const polynom<T>&)' is not available for polynoms" << endl;
  exit(1);
}


}


#endif
