/***************************************************************
 *                 Mathematical Object Library                 *
 *          class function: declaration for functions          *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 2.4.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2005,2006,2008,2009 COLLARD Christophe
 * copyright © 2005,2006,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2005,2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::function
    \brief function library \n

    \htmlonly 
    <FONT color="#838383">

    function belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \author copyright \htmlonly &#169; \endhtmlonly 2005, 2006, 2008, 2009 Christophe COLLARD \n
	    copyright \htmlonly &#169; 2005, 2006, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	    copyright \htmlonly &#169; 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
    \version 2.4.0
    \date 2005-2010
    \bug none
    \warning under construction (an interpreter will be available later)
*/

#ifndef __cplusplus
#error Must use C++ for the type function
#endif

#if !defined(_FUNCTIONS_H)
#define _FUNCTIONS_H

#if !defined(__VECTORS_H)
#include "vectors.h"
#endif

using namespace std;

namespace mol
{


//===============================
template <class T> class function
//===============================
{
  private :
    int n,m;
    char** f; // application from Rn -> Rm

  public :
    function ();    // default constructor
    function (int, int); // constructor with space dimension (mettre a jour le module heterogeneites plastiques pour supprimer ce constructeur)

    int dim () const {return n;}  // returns space dimension
//    int size () const {return n;} // returns space dimension
    vector<T> operator () (const T&, const vector<T>&) const; // returns the value of the function f(t,x(t))
    vector<T> operator () (const vector<T>&) const; // returns the value of the function f(x) at x
    char*& operator [] (int) const;  // returns the expression of the ith component of the function (cannot be used to allocate a function)
    T operator () (const vector<T>*) const; // returns the value of the function f(x,y,z,...) at x,y,z,...
    template <class M> friend  ostream& operator << (ostream&, const function<M>&);
    //    friend  istream& operator >> <T> (istream&, const function<T>&);
};


//=====Private methods for integration===========================================


//=====Public methods for integration============================================


//-----------------------------------------
template <class T> function<T>::function ()
//-----------------------------------------
{
  n = m = 0;
}


/*!
  \brief constructor for functions \f$ f : \boldsymbol{R}^n \longrightarrow \boldsymbol{R}^m \f$ 
  \param n
  \param m
*/

//-------------------------------------------------------
template <class T> function<T>::function (int nl, int ml)
//-------------------------------------------------------
{
  assert (nl>0 && ml>0);
  n = nl;
  f = new char* [m=ml];
}


//--------------------------------------------------------------
template <class T> char*& function<T>::operator [] (int i) const
//--------------------------------------------------------------
{
  assert (n && m);
  assert ((i>0) && (i<=m));
  return (f[i-1]);
}


//------------------------------------------------------------------------------------------
template <class T> vector<T> function<T>::operator () (const T& t, const vector<T>& x) const
//------------------------------------------------------------------------------------------
{
  assert (n && m);
  assert (x.dim() == n);
  vector<T> value(m);

  if (m==1)
    { if (f[0]=="y") value = x;
      else if (f[0]=="-y^2") value[1] = -1.*(x*x);
      else {cout << f[0] << endl; exit(1);}
    }
  else if (m==2)
    { if ( (f[0] == "y[2]") && (f[1] == "y[1] + y[2] - (2*sin(t)+cos(t))") )
	{ value[1] = x[2];
	  value[2] = x[1] + x[2] - (2*sin(t) + cos(t));
	}
      else if (  (f[0] == "y[2]") && (f[1] == "-y[2]*y[2] - t*y[1] + t*log(1+t)") )
	{ value[1] =  x[2];
	  value[2] = -power(x[2],2) - t*x[1] + t*log(1+t);
	}
      else if ( (f[0] == "y[2]") && (f[1] == "y[2]^2 - t*y[1] + (t-1)*sin(t) - cos^2(t)") )
	{ value[1] =   x[2];
	  value[2] = power(x[2],2) - t*x[1] + (t-1)*sin(t) - power(cos(t),2);
	}
      else exit(1);
    }
  else exit(1);

  return value;
}


//------------------------------------------------------------------------------
template <class T> vector<T> function<T>::operator () (const vector<T>& x) const
//------------------------------------------------------------------------------
{
  assert (n && m && n==x.dim());
  vector<T> value(m,false);

  //  for (int i=0; i<dimension; i++)
  //    assert (x[i].dim());

  if ((f[0])=="x²") value[1] = power(x[1],2);
  else if ((f[0])=="x²/y") value[1] = power(x[1],2) / x[2];
  else if ((f[0])=="x*exp(x*y)") value[1] = x[1] * exp(x[1]*x[2]);
  else if ((f[0])=="x²/y*sin(z)") value[1] = power(x[1],2) / x[2] * sin(x[3]);
  else if ((f[0])=="x²*z²*exp(yz)") value[1] = power(x[1],2) * power(x[3],2) * exp(x[2]*x[3]);
  else if ((f[0])=="heterogeneites plastiques")
    { // partie heterogeneites plastiques
      // T nu = 0.3;
      T mu = 80000;
      // T lambda = 120000;
      T k3 = 520000; // 3k

      vector<T> w;
      w = 3;
      w[1] = x[1] - x[4];
      w[2] = x[2] - x[5];
      w[3] = x[3] - x[6];

      T norme = w.norm(); // w
      T normx = norme * norme; // w^2
      T w2 = power(w[2],2); // w_2 * w_2
      T w3 = power(w[3],2); // w_3 * w_3

      //cout << w;

      value[1] = - 15 * (k3+mu) * w2 * w3 / normx;
      value[1] += 1.5 * (k3-2*mu) * (w2+w3);
      value[1] /= normx;
      value[1] += 3 * mu;
      value[1] /= (4 * pi() * mu * (k3+4*mu) * normx * norme);
    }
  else exit(1);

  return value;


  /*


  //cout << value << endl;
//exit (0);

//  T value = 2*nu-1;
//  value -= 3* nu * (w2+w3) / normx; // 3 nu (w_2^2 + w_3^2) / w^2
//  norme *= normx; // w^3
//  normx *= normx; // w^4
//  value += 15 *w2*w3/normx;
//  value /= 8 *(1-nu)*mu*pi()*norme;

  return value;
  //  return  1/(8.*(1-nu)*mu*pi()*power(w.norm(),3)) * 
  //          (15 * power(w[2],2) * power(w[3],2) / power(w.norm(),4)
  //	   - 3 * nu * (power(w[2],2) + power(w[3],2)) / power(w.norm(),2)
  //	   - (1-2*nu));
  */
}


//----------------------------------------------------------------------
template <class T> T function<T>::operator () (const vector<T>* x) const
//----------------------------------------------------------------------
{
  assert (n && m);
  T value;

/*
  vector<S> u = x[0];
  vector<S> v = x[1];

  S value = 1/(8.*(1-nu)*mu*pi()*power((u-v).norm(),3)) * 
    (15 * power((v[2]-u[2]),2) * power((v[3]-u[3]),2) / power((u-v).norm(),4)
     - 3 * nu * (power((v[2]-u[2]),2) + power((v[3]-u[3]),2)) / power((u-v).norm(),2)
     - (1-2*nu));

  //  T value = u[1] * u[2] * u[3];
*/

  return value;
}


//--------------------------------------------------------------------------
template <class T> ostream& operator << (ostream& s, const function<T>& fct)
//--------------------------------------------------------------------------
{
  for (int i=0; i<fct.m; i++)
    s << fct.f[i] << endl;

  return s;
}


}


#endif
