/***************************************************************
 *                 Mathematical Object Library                 *
 *     class symmatrix:  declarations for symmetric matrix     *
 *                    simula+@metz.ensam.fr                    *
 *                   GNU/linux version 2.2.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2002,2003,2004,2005,2006,2007,2008,2009 COLLARD Christophe
 * copyright © 2002,2003,2004 CREUSE Emmanuel
 * copyright © 2002,2003,2004,2005,2006,2007,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2002,2003,2004,2005,2006,2007 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::symmatrix
    \brief symmetric matrix library \n

    \htmlonly 
    <FONT color="#838383">

    symmatrix belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly


    \authors copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009 Christophe COLLARD \n
             copyright \htmlonly &#169; \endhtmlonly 2002, 2003, 2004 Emmanuel CREUSE \n
	     copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2002, 2003, 2004, 2005, 2006, 2007 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly
    \version 2.2.0
    \date 2002-2009
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type symmatrix
#endif

#if !defined(_SYMMATRIX_H)
#define _SYMMATRIX_H


#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__STDLIB_H)
#include <stdlib.h>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__PARAMETERS_H)
#include "../parameters.h"
#endif

#if !defined(__VECTORS_H)
#include "vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "matrix.h"
#endif

using namespace std;

namespace mol
{


//===========================================================
template <class T> class symmatrix : public virtual matrix<T>
//===========================================================
{
  using matrix<T>::nbRows;
  using matrix<T>::nbColumns;
  using matrix<T>::mrows;

  public:
    symmatrix () : matrix<T>() {};        // default constructor
    symmatrix (int size, bool=true, T=0); // this constructor allocates memory
    symmatrix (int, int, bool=true, T=0); // this constructor allocates memory
    symmatrix (const symmatrix<T>&);      // copy constructor
    symmatrix (symmatrix<T>*);            // copy constructor for temporary objects
    symmatrix (const matrix<T>&);         // cast constructor matrix -> symmatrix

    virtual int Columns () const {return nbColumns;}     // returns the # of columns
    virtual void assign  (int,int);  // assign memory without initilizing the symmetric matrix components to zero
    virtual T&   operator () (int,int) const;
    symmatrix<T>& operator = (const symmatrix<T>&);
    virtual symmatrix<T>& operator = (symmatrix<T>*);  // allocates the data of a temporary object to a real object (without copying data)
    symmatrix<T>& operator = (const matrix<T>&);
    virtual symmatrix<T>& operator &= (const symmatrix<T>&);  // copies symmatrix without size check (use this operator with great care)
    virtual symmatrix<T>& operator &= (symmatrix<T>*); // allocates the data from the right hand temporary object to the left hand object (deletes temporary object) - no size check - no copy (use this operator with great care)
    template <class Tf> friend symmatrix<Tf> operator + (const symmatrix<Tf>&, const symmatrix<Tf>&); // overloads plus operator for symmatrix
    template <class Tf> friend symmatrix<Tf> operator - (const symmatrix<Tf>&, const symmatrix<Tf>&); // overloads plus operator for symmatrix
    //    S * S' is not symmetric when S and S' are symmetric
    template <class Tf> friend symmatrix<Tf> operator * (const symmatrix<Tf>&, const Tf&);  // overloads symmatrix * scalar
    template <class Tf> friend symmatrix<Tf> operator * (const Tf&, const symmatrix<Tf>&);  // overloads scalar * symmatrix
    //    S / S' is not symmetric when S and S' are symmetric
    template <class Tf> friend symmatrix<Tf> operator / (const symmatrix<Tf>&, const Tf&);  // overloads symmatrix / scalar
    template <class Tf> friend symmatrix<Tf> operator / (const Tf&, const symmatrix<Tf>&);  // overloads scalar / symmatrix
    void operator += (const symmatrix<T>&); // overloads += operator for symmatrix/symmatrix
    void operator += (const matrix<T>&);    // overloads += operator for symmatrix/matrix (which must be symmetric)
    void operator -= (const symmatrix<T>&); // overloads -= operator for symmatrix/symmatrix
    void operator -= (const matrix<T>&);    // overloads -= operator for symmatrix/matrix (which must be symmetric)
    template <class Tf> friend bool operator == (const symmatrix<Tf>&, const symmatrix<Tf>&); // optimizes == operator of symmatrix
    template <class Tf> friend bool operator != (const symmatrix<Tf>&, const symmatrix<Tf>&); // same optimization
    friend symmatrix<T> t (const symmatrix<T>& mat) {return mat;}        // optimizes tranposition operator for symmatrix
    template <class Tf> friend symmatrix<Tf> gauss (const symmatrix<Tf>&);  // computes symmatrix inversion with gauss algorithm
    template <class Tf> friend symmatrix<Tf> abs (const symmatrix<Tf>&);    // returns the symmatrix absolute value
    template <class Tf> friend symmatrix<Tf> Id2s (int);                    // returns identity matrix
};


//=====Private methods for symmatrix=============================================


//=====Public methods for symmatrix==============================================


//-----------------------------------------------------------------------
template <class T> symmatrix<T>::symmatrix (int size, bool init, T value)
//-----------------------------------------------------------------------
{
  assert (size > 0);

  mrows = new vector<T> [nbRows=nbColumns=size];
  for (int i=1; i<=size; i++)
    mrows[i-1] =& vector<T> (i,init,value);
}


//------------------------------------------------------------------------------------
template <class T> symmatrix<T>::symmatrix (int rows, int columns, bool init, T value)
//------------------------------------------------------------------------------------
{
  assert (rows>0 && columns>0);
  assert (rows==columns); // this matrix must be squared

  mrows = new vector<T> [nbRows=nbColumns=rows];
  for (int i=1; i<=rows; i++)
    mrows[i-1] =& vector<T> (i,init,value);
}


//------------------------------------------------------------------
template <class T> symmatrix<T>::symmatrix (const symmatrix<T>& mat)
//------------------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns);
  assert (mat.nbRows==mat.nbColumns);  // this matrix must be a squared in order to be symmetric

  mrows = new vector<T> [nbRows=nbColumns=mat.nbRows];
  for (int i=0; i<mat.nbRows; i++)
    mrows[i] = mat.mrows[i];
}


//------------------------------------------------------------
template <class T> symmatrix<T>::symmatrix (symmatrix<T>* mat)
//------------------------------------------------------------
{
  assert ((*mat).nbRows);

  nbRows = (*mat).nbRows;         // copy the size of the temporary objet
  nbColumns = (*mat).nbColumns;   // copy the size of the temporary objet
  mrows = (*mat).mrows;           // copy the address of the temporary objet
  (*mat).nbRows = (*mat).nbColumns = 0;  // set temporary object size to zero (so the data stored at p won't be delete by the destructor)
}


//---------------------------------------------------------------
template <class T> symmatrix<T>::symmatrix (const matrix<T>& mat)
//---------------------------------------------------------------
{
  assert (mat.Rows() && mat.Columns());
  assert (mat.Rows() == mat.Columns());  // this matrix must be a squared in order to be symmetric

  mrows = new vector<T> [nbRows=mat.Rows()];
  nbColumns = mat.Columns();
  for (int i=1; i<=nbRows; i++)
    { mrows[i-1] = i;
      for (int j=1; j<=i; j++)
 	{ assert (abs(mat(i,j)-mat(j,i))<epsilon); // needed by symmatrix = matrix
	  (*this)(i,j) = mat(i,j);
	}
    }
}


//------------------------------------------------------------------
template <class T> void symmatrix<T>::assign (int rows, int columns)
//------------------------------------------------------------------
{
  // computation of matrix row vectors
  assert ((rows>0) && (columns>0));
  assert (!nbRows);
  assert (rows==columns);  // this matrix must be a squared in order to be symmetric

  mrows = new vector<T> [nbRows=nbColumns=rows];
  for (int i=1; i<=rows;i++)
    mrows[i-1] = i; // allocates memory - better than mrows[i] = vector<T>(i+1) -> no temporary object -
}


//-------------------------------------------------------------------------
template <class T> T& symmatrix<T>::operator () (int row, int column) const
//-------------------------------------------------------------------------
{
  assert ((row>0)&&(row<=nbRows));
  assert ((column>0)&&(column<=nbColumns));

  if (column>row) // take the symmetric element (which is stored)
    {int k=column; column=row; row=k;}

  return (mrows[row-1])[column];
}


//---------------------------------------------------------------------------------
template <class T> symmatrix<T>& symmatrix<T>::operator = (const symmatrix<T>& mat)
//---------------------------------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns);

  if (!nbRows) create (mat.nbRows,mat.nbColumns);
  else assert (nbRows == mat.nbRows);

  for (int i=0; i<nbRows; i++)
    mrows[i] = mat.mrows[i];

  return *this;
}


//------------------------------------------------------------------------------
template <class T> symmatrix<T>& symmatrix<T>::operator = (const matrix<T>& mat)
//------------------------------------------------------------------------------
{
  assert (mat.Rows());
  assert (mat.Rows() == mat.Columns());

  if (!nbRows) assign (mat.Rows(),mat.Columns());  // allocates memory for symmatrix (no initialization)
  else assert ( (nbRows==nbColumns) && (nbRows==mat.Rows()) && (nbColumns==mat.Columns()) );

  for (int i=1; i<=nbRows; i++)
    for (int j=1; j<=i; j++)
      { assert (abs(mat(i,j)-mat(j,i))<epsilon); // needed by symmatrix = matrix
        (*this)(i,j) = mat(i,j);
      }

  return *this;
}


//---------------------------------------------------------------------------
template <class T> symmatrix<T>& symmatrix<T>::operator = (symmatrix<T>* mat)
//---------------------------------------------------------------------------
{
  assert ((*mat).nbRows && (*mat).nbColumns);

  if (nbRows)
    { assert ( (nbRows == (*mat).nbRows) && (nbColumns == (*mat).nbColumns) );
      delete [] mrows;
    }
  else
    { nbRows = (*mat).nbRows;
      nbColumns = (*mat).nbColumns;
    }
  mrows = (*mat).mrows;
  (*mat).nbRows = (*mat).nbColumns = 0;

  return *this;
}


//----------------------------------------------------------------------------------
template <class T> symmatrix<T>& symmatrix<T>::operator &= (const symmatrix<T>& mat)
//----------------------------------------------------------------------------------
{
  if (!mat)
    { if (nbRows) delete [] mrows;
      nbRows = nbColumns = 0;
    }

  else
    { assert (mat[1].dim()==1 && mat[mat.nbRows].dim()==mat.nbColumns); // forbids the acces to matrix && spmatrix
      if (nbRows != mat.nbRows)
	{ if (nbRows) delete [] mrows;
	  mrows = new vector<T> [nbRows = mat.nbRows];  // allocates memory with the = operator for vector (no initialization)
	  nbColumns = mat.nbColumns;
	}
      for (int i=1; i<=mat.nbRows; i++)
	(*this)[i] = mat[i];
    }

  return *this;
}


//----------------------------------------------------------------------------
template <class T> symmatrix<T>& symmatrix<T>::operator &= (symmatrix<T>* mat)
//----------------------------------------------------------------------------
{
  if (!(*mat).nbRows)
    { if (nbRows) delete [] mrows;
      nbRows = nbColumns = 0;
    }

  else
    { if (nbRows) delete [] mrows;
      nbRows = (*mat).nbRows;
      nbColumns = (*mat).nbColumns;
      mrows = (*mat).mrows;
      (*mat).nbRows = (*mat).nbColumns = 0;
    }

  return *this;
}


//-------------------------------------------------------------------------------------------------
template <class Tf> symmatrix<Tf> operator + (const symmatrix<Tf>& mat1, const symmatrix<Tf>& mat2)
//-------------------------------------------------------------------------------------------------
{
  assert ( mat1.Rows()==mat2.Rows() ); // matrices must have the same size
  assert (mat1.Rows() && mat1.Columns());
  symmatrix<Tf> mat = mat1;
  mat += mat2;
  return mat;
}


//-------------------------------------------------------------------------------------------------
template <class Tf> symmatrix<Tf> operator - (const symmatrix<Tf>& mat1, const symmatrix<Tf>& mat2)
//-------------------------------------------------------------------------------------------------
{
  assert ( mat1.Rows()==mat2.Rows() ); // matrices must have the same size
  assert (mat1.Rows() && mat1.Columns());
  symmatrix<Tf> mat = mat1;
  mat -= mat2;
  return mat;
}


//------------------------------------------------------------------------------------
template <class Tf> symmatrix<Tf> operator * (const symmatrix<Tf>& mat, const Tf& elt)
//------------------------------------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns);
  symmatrix<Tf> mp = mat;
  mp *= elt;
  return mp;
}


//------------------------------------------------------------------------------------
template <class Tf> symmatrix<Tf> operator * (const Tf& elt, const symmatrix<Tf>& mat)
//------------------------------------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns);
  symmatrix<Tf> mp = mat;
  mp *= elt;
  return mp;
}


//------------------------------------------------------------------------------------
template <class Tf> symmatrix<Tf> operator / (const symmatrix<Tf>& mat, const Tf& elt)
//------------------------------------------------------------------------------------
{
  assert (mat.nbRows&&mat.nbColumns);
  assert (elt); // can't divide by zero
  symmatrix<Tf> mp = mat;
  mp /= elt;
  return mp;
}


//------------------------------------------------------------------------------------
template <class Tf> symmatrix<Tf> operator / (const Tf& elt, const symmatrix<Tf>& mat)
//------------------------------------------------------------------------------------
{
  assert (mat.nbRows&&mat.nbColumns);
  assert (elt); // can't divide by zero
  symmatrix<Tf> mp =& gauss(mat);
  mp *= elt;
  return mp;
}


//-------------------------------------------------------------------------
template <class T> void symmatrix<T>::operator += (const symmatrix<T>& mat)
//-------------------------------------------------------------------------
{
  assert(nbRows && nbColumns);
  assert ((nbRows==mat.nbRows) && (nbColumns==mat.nbColumns)); // matrices must have the same size
  for (int i=0; i<nbRows; i++)
    mrows[i] += mat.mrows[i];
}


//----------------------------------------------------------------------
template <class T> void symmatrix<T>::operator += (const matrix<T>& mat)
//----------------------------------------------------------------------
{
  assert(nbRows && nbColumns);
  assert ( (nbRows==mat.Rows()) && (nbColumns==mat.Columns()) ); // matrices must have the same size
  for (int i=1; i<=nbRows; i++)
    for (int j=1; j<=i; j++)
      { assert(abs(mat(i,j)-mat(j,i))<epsilon);
        (*this)(i,j) += mat(i,j);
      }
}


//-------------------------------------------------------------------------
template <class T> void symmatrix<T>::operator -= (const symmatrix<T>& mat)
//-------------------------------------------------------------------------
{
  assert(nbRows && nbColumns);
  assert ((nbRows==mat.nbRows) && (nbColumns==mat.nbColumns)); // matrices must have the same size
  for (int i=0; i<nbRows; i++)
    mrows[i] -= mat.mrows[i];
}


//----------------------------------------------------------------------
template <class T> void symmatrix<T>::operator -= (const matrix<T>& mat)
//----------------------------------------------------------------------
{
  assert ((nbRows==mat.Rows()) && (nbColumns==mat.Columns())); // matrices must have the same size
  assert((nbRows)&&(nbColumns));
  for (int i=1; i<=nbRows; i++)
    for (int j=i; j<=nbColumns; j++)
      { assert(abs(mat(i,j)-mat(j,i))<epsilon);
        (*this)(i,j) -= mat(i,j);
      }
}


//-----------------------------------------------------------------------------------------
template <class Tf> bool operator == (const symmatrix<Tf>& mat1, const symmatrix<Tf>& mat2)
//-----------------------------------------------------------------------------------------
{
  bool result=( (mat1.nbRows==mat2.nbRows) && (mat2.nbColumns==mat2.nbColumns) );
  for (int i=1; i<=mat1.nbRows && result; i++)
    result *= (mat1[i] == mat2[i]);
  return result;
}


//-----------------------------------------------------------------------------------------
template <class Tf> bool operator != (const symmatrix<Tf>& mat1, const symmatrix<Tf>& mat2)
//-----------------------------------------------------------------------------------------
{
  return !(mat1 == mat2);
}


//---------------------------------------------------------------
template <class Tf> symmatrix<Tf> gauss(const symmatrix<Tf>& mat)
//---------------------------------------------------------------
{
  // optimiser imperativement l'algo !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  symmatrix<Tf> smt;
  matrix<Tf> mt;
  mt = mat; // copie pour la convertion en trop
  smt = gauss(mt); // copie pour la convertion en trop + ADAPTER GAUSS
  return smt;
}


//--------------------------------------------------------------
template <class Tf> symmatrix<Tf> abs (const symmatrix<Tf>& mat)
//--------------------------------------------------------------
{
  assert (mat.nbColumns);
  symmatrix<Tf> mtx;
  mtx.create(mat.nbRows,mat.nbColumns);

  for (int i=1; i<=mat.nbRows; i++)
    mtx[i] =& abs(mat[i]);

  return mtx;
}


//--------------------------------------------
template <class Tf> symmatrix<Tf> Id2s (int n)
//--------------------------------------------
{
  assert (n > 0);
  symmatrix<Tf> I(n,n);

  for (int i=1; i<=n; i++)
    I(i,i) = 1;

  return I;
}


}


#endif
