/***************************************************************
 *                 Mathematical Object Library                 *
 *     class tensor2 : declarations for 2nd order tensors      *
 *                    simula+@metz.ensam.fr                    *
 *		     GNU/linux version 0.3.6                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2004,2005,2006 COLLARD Christophe
 * copyright  2004,2005,2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  2004,2005,2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! \class tensor2
    \brief 2nd order tensor library \n

    \htmlonly 
    <FONT color="#838383">

    tensor2 belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    Let \f$ E \f$ be a Euclidean space of dimension \f$ n \f$ with basis vectors \f$ \{ e_i \} = (e_1,\ldots,e_n) \f$. A \f$ 2^\text{nd} \f$ order Euclidean tensor \f$ T \f$ is defined as a bilinear application from \f$ E \times E \longrightarrow \Bbb{R} \f$. We use a matrix storage for these tensors. All the basic operations are then already defined, so we just add some specific tensors operators to this class. \n

    \author copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006 Christophe COLLARD \n
	    copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	    copyright \htmlonly &#169; \endhtmlonly 2004, 2005, 2006 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.3.6
    \date 2004-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type tensor2
#endif

#if !defined(_TENSORS2_H)
#define _TENSORS2_H


#if !defined(__VECTORS_H)
#include "vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "matrix.h"
#endif

using namespace std;


//=================================================
template <class T> class tensor2 : public matrix<T>
//=================================================
{
  using matrix<T>::nbRows;
  using matrix<T>::nbColumns;

  public :  
    tensor2 () : matrix<T> () {} 		     // default constructor
    tensor2 (int i, bool init=true, T value=0) : matrix<T> (i,i,init,value) {}         // this constructor allocates memory
    tensor2 (int i, int j, bool init=true, T value=0) : matrix<T> (i,j,init,value) {}  // this constructor allocates memory
    tensor2 (const tensor2<T>& t) : matrix<T>(t) {}  // copy constructor
    tensor2 (const matrix<T>&);                      // translation constructor (converts matrix -> tensor)
    tensor2 (tensor2<T>* t) : matrix<T>(t) {}        // copy constructor for temporary objects
    tensor2 (matrix<T>* m)  : matrix<T>(m) {}        // copy constructor for temporary objects
    ~tensor2 (){}  // destructor

    virtual int dim1 () const {return nbRows;}    // returns the tensor 1st component size : i
    virtual int dim2 () const {return nbColumns;} // returns the tensor 2nd component size : j
    template <class FT> friend FT operator | (const tensor2<FT>&, const tensor2<FT>&);  // A|B = Aij Bij
    template <class FT> friend tensor2<FT> sph (const tensor2<FT>&);  // computes the sperical part of a tensor
    template <class FT> friend tensor2<FT> dev (tensor2<FT>);         // computes the deviatoric part of a tensor
};


//=====Private methods for tensor2=============================================

//=====Public methods for tensor2==============================================


//-----------------------------------------------------------------------------
template <class T> tensor2<T>::tensor2 (const matrix<T>& mat) : matrix<T> (mat)
//-----------------------------------------------------------------------------
{
  //  cout << "WARNING : USING CONVERSION MATRIX -> TENSOR \n";
}


/*!
  \brief Computes the double contracted tensor product for second order tensors.

  It we note \f$ T \f$ and \f$ R \f$ two second order tensors, then the double contracted tensor product reads \f$ T_{ij} R_{ij} \f$.

  \param t1 \f$ 2^\text{nd} \f$ order tensor
  \param t2 \f$ 2^\text{nd} \f$ order tensor
  \return double contracted tensor product (scalar)
*/

//------------------------------------------------------------------------------
template <class FT> FT operator | (const tensor2<FT>& t1, const tensor2<FT>& t2)
//------------------------------------------------------------------------------
{
  assert (t1.nbRows);
  assert (t1.nbRows==t2.nbRows);

  FT sum=0;
  for (int i=1; i<=t1.nbRows; i++)
    for (int j=1; j<=t1.nbColumns; j++)
      sum += t1(i,j) * t2(i,j);

  return sum;
}


/*!
  \brief Computes the spherical part of a \f$ 2^\text{nd} \f$ order tensor.

  If we denote by \f$ Sph(T) \f$ the spherical part of the tensor \f$ T \f$, then it reads : \n
  \f$ \displaystyle Sph(T) = \frac{1}{n} Tr (T) \; G \f$, \n
  where \f$ Tr(T) = T_{ii} = T_{11} + \ldots + T_{nn} \f$ is the tensor trace
  and \f$ G \f$ is the metric tensor given by \n
  \f$ G = g_{ij} e^i \otimes e^j = g^{ij} e_i \otimes e_j = g_i^j e^i \otimes e_j = g_j^i e_i \otimes e^j \f$ \n
  with \n
  \f$ \begin{aligned}
  g_{ij} &= G(e_i,e_j) = e_i \cdot e_j \\
  g^{ij} &= G(e^i,e^j) = e^i \cdot e^j \\
  g_i^j  &= G(e_i,e^j) = e_i \cdot e^j = \delta_i^j \\
  g^i_j  &= G(e^i,e_j) = e^i \cdot e_j = \delta_j^i
  \end{aligned}
  \f$ \n
where \f$ \delta \f$ is the Kronecker symbol, where \f$ \{ e_i \f$} is the covariant basis and where \f$ \{ e^i \f$} is the contravariant basis.

  \param tsr \f$ 2^\text{nd} \f$ order tensor \f$ T \f$
  \return spherical part of \f$ T \f$
*/

//----------------------------------------------------------
template <class FT> tensor2<FT> sph (const tensor2<FT>& tsr)
//----------------------------------------------------------
{
  int n;
  n = tsr.dim1();
  assert (n>0 && n==tsr.dim2());

  FT trace = tr(tsr) / (FT) n;
  tensor2<FT> sp(n,n);

  for (int i=1; i<=n; i++)
    sp (i,i) = trace;

  return sp;
}


/*!
  \brief Computes the deviatoric part of a \f$ 2^\text{nd} \f$ order tensor.

  If we denote by \f$ Dev(T) \f$ the deviatoric part of the tensor \f$ T \f$, then it reads : \n
  \f$ \displaystyle Dev(T) = T - Sph(T) \f$, \n
  where \f$ Sph(T) \f$ is the spherical part of \f$ T \f$ (see \ref sph).

  \param tsr \f$ 2^\text{nd} \f$ order tensor \f$ T \f$
  \return deviatoric part of \f$ T \f$
*/

//---------------------------------------------------
template <class FT> tensor2<FT> dev (tensor2<FT> tsr)
//---------------------------------------------------
{
  int n = tsr.dim1();
  assert (n>0 && n==tsr.dim2());

  FT trace = tr(tsr) / (FT) n;

  for (int i=1; i<=n; i++)
    tsr (i,i) -= trace;

  return tsr;
}


#endif
