/***************************************************************
 *                 Mathematical Object Library                 *
 *      class spmatrix : declarations for sparse matrices      *
 *                    simula+@metz.ensam.fr                    *
 *		     GNU/linux version 0.0.8                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright  2004,2005 LAGOUCHE Sophie
 * copyright  2004,2005 CREUSE Emmanuel
 * copyright  2004,2005 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554)
 * copyright  2004,2005 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
 ***************************************************************/

/*! 
    \class spmatrix
    \brief sparse matrix library

    \htmlonly 
    <FONT color="#838383">

    spmatrix belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly 2004, 2005 Sophie LAGOUCHE \n
             copyright \htmlonly &#169; \endhtmlonly 2004, 2005 Emmanuel CREUSE \n
	     copyright \htmlonly &#169; \endhtmlonly 2004, 2005 Laboratoire de Physique et Mcanique des Matriaux (LPMM - UMR 7554) \n
	     copyright \htmlonly &#169; \endhtmlonly 2004, 2005 Laboratoire de Mathmatiques et ses Applications de Valenciennes (LAMAV - EA 4015)
    \version 0.0.8
    \date 2004-2006
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type sparse matrix.
#endif

#if !defined(_SP_MATRIX)
#define _SP_MATRIX


#if !defined(__IOSTREAM_H)
#include <iostream>
#endif

#if !defined(__FSTREAM_H)
#include <fstream>
#endif

#if !defined(__ASSERT_H)
#include <assert.h>
#endif

#if !defined(__PRECISION_H)
#include "precision.h"
#endif

#if !defined(__MATHS_H)
#include "maths.h"
#endif

#if !defined(__VECTORS_H)
#include "vectors.h"
#endif

#if !defined(__MATRIX_H)
#include "matrix.h"
#endif

using namespace std;


//==================================================
template <class T> class spmatrix : public matrix<T>
//==================================================
{
  using matrix<T>::nbRows;
  using matrix<T>::nbColumns;
  using matrix<T>::mrows;

  private:
    matrix<int> columns_position;
    int real_nbCol;
    vector<int> nbRowsElements;
	
  public:
    spmatrix();
    spmatrix(int,int,int);
    spmatrix (const spmatrix<T>&);
    ~spmatrix ();
	
    void set (int, int, T);
    void add (int , int, T);
    T output (int, int) const;
    template <class Tf> friend ostream& operator << (ostream&, const spmatrix<Tf>&);  
    spmatrix<T> resize (int);
    virtual int Row_Elements () const {return nbColumns;}          // returns the number of max. non zero coefficients per row
	
    virtual spmatrix<T>& operator = (const spmatrix<T>&);
    template <class Tf> friend spmatrix<Tf> operator + (const spmatrix<Tf>& , const spmatrix<Tf>& );
    template <class Tf> friend spmatrix<Tf> operator - (const spmatrix<Tf>& , const spmatrix<Tf>& );
    matrix<T> full ();  
    template <class Tf> friend spmatrix<Tf> operator * (const Tf&, const spmatrix<Tf>&);
    template <class Tf> friend vector<Tf>   operator * (const spmatrix<Tf>&, const vector<Tf>&);
    template <class Tf> friend matrix<Tf>   operator * (spmatrix<Tf>&, spmatrix<Tf>&);
    template <class Tf> friend spmatrix<Tf> operator / (const spmatrix<Tf>&, const Tf&);
    template <class Tf> friend spmatrix<Tf> t (const spmatrix<Tf>&);
    template <class Tf> friend bool operator == (const spmatrix<Tf>&, const spmatrix<Tf>&);
    template <class Tf> friend bool operator != (const spmatrix<Tf>&, const spmatrix<Tf>&);

    template <class Tf> friend Tf trace (const spmatrix<Tf>&);   
    template <class Tf> friend Tf max (const spmatrix<Tf>&);  
};


//=====Public methods for spmatrix==============================================


/*!
  \brief default constructor of the spmatrix class.
*/

//-----------------------------------------
template <class T> spmatrix<T>::spmatrix ()
//-----------------------------------------
{
  nbRows = nbColumns = real_nbCol = 0;
}


/*!
   \brief constructor of the spmatrix class. The arguments are three integers
    row, column and non_zero_elements. row is the number of row of the sparse matrix.
    column is the number of column of the sparse matrix. non_zero_element is the maximal
    number of non-zero element per row of the sparse matrix.
   \param row integer
   \param column integer
   \param non_zero_element integer

*/

//-----------------------------------------------------------------------------------------------------------------------
template <class T> spmatrix<T>::spmatrix (int row, int column, int non_zero_elements) : matrix<T>(row, non_zero_elements)
//-----------------------------------------------------------------------------------------------------------------------
{
  assert (column>0 && column>=non_zero_elements);
  if (column == non_zero_elements) cout << "WARNING : you are using sparse matrix to store a full matrix \n";

  real_nbCol = column;
  columns_position = &matrix<int>(row,non_zero_elements);
  nbRowsElements = &vector<int>(row);
}


/*!
  \brief copy constructor of the spmatrix class.
  \param mat  reference on spmatrix object
*/

//---------------------------------------------------------------
template <class T> spmatrix<T>::spmatrix (const spmatrix<T>& mat)
//---------------------------------------------------------------
{
  assert (mat.nbRows);

  columns_position = mat.columns_position;
  real_nbCol = mat.real_nbCol;
  nbRowsElements = mat.nbRowsElements;

  nbColumns = mat.nbColumns;
  mrows = new vector<T> [nbRows = mat.nbRows];
  for (int i=0; i<mat.nbRows; i++)
    mrows[i] = mat.mrows[i];
}


/*!
   \brief destructor of the spmatrix class.
*/

//------------------------------------------
template <class T> spmatrix<T>::~spmatrix ()
//------------------------------------------
{
  // destruction of the memory allocated for the spmatrix
  if (nbRows) 
    delete [] mrows; // free memory only if it's been affected 

  nbRows = nbColumns = real_nbCol = 0;
}


/*!
  \brief  assignment, set operator for the spmatrix class.The arguments are three integers
    i, j and coeff. i is the indication of row. j is the indication of column. 
    coeff is the element to affect.
  \param i integer
  \param j integer
  \param coeff <T> 
*/

//--------------------------------------------------------------
template <class T> void spmatrix<T>::set (int i, int j, T coeff)
//--------------------------------------------------------------
{
  assert (i>0 && i<=nbRows);
  assert (j>0 && j<=real_nbCol);

  int r = nbRowsElements[i];

  if (abs(coeff)>epsilon)
    { bool done = true;
      for (int k=1; k<=r; k++)
	if (columns_position(i,k) == j)
	  { mrows[i-1][k]=coeff;
	    done = false;
	  }
     
      if (done) 
	{if (r+1>nbColumns)
	    cout << "Warning : line " << i << " already contains " << nbColumns << " non-zero elements, that's the maximum !";
	  else
	    { columns_position(i,r+1)=j;
	      mrows[i-1][r+1]=coeff;
	      nbRowsElements[i]=r+1;
	    }
	}
    }

  else
    for (int k=1; k<=r; k++)
      if (columns_position(i,k) == j)
	{ mrows[i-1][k] = mrows[i-1][r];
	  mrows[i-1][r] = coeff;
	  columns_position(i,k) = columns_position(i,r);
	  columns_position(i,r) = 0; 
	  nbRowsElements[i]-=1;
	}	
}


/*!
   \brief add coef to the element (i,j) of a spmatrix.
   \param i integer
   \param j integer
   \param coef <T>
*/

//-------------------------------------------------------------
template <class T> void spmatrix<T>::add (int i, int j, T coef)
//-------------------------------------------------------------
{
  assert (i>0 && i<=nbRows);
  assert (j>0 && j<=real_nbCol);

  int r = nbRowsElements[i];

  if (abs(coef)>epsilon)
    { bool done = true;
      for (int k=1; k<=r; k++)
	if (columns_position(i,k) == j)
	  { if (abs(mrows[i-1][k] + coef)<epsilon)
	      { mrows[i-1][k] = mrows[i-1][r];
		columns_position(i,k) = columns_position(i,r);
		mrows[i-1][r] = 0;
		columns_position(i,r) = 0;
		nbRowsElements[i] -= 1;
		done = false;
	      }
            else
	      { mrows[i-1][k] += coef;
		done = false;
	      }
	  }

      if (done) 
	{ if (r+1>nbColumns)
	    cout << "Warning : it's impossible to add this non-zero element because line " << i << " already contains " << nbColumns << " non-zero elements, that's the maximum !";
	  else
	    { columns_position(i,r+1)=j;
	      mrows[i-1][r+1]=coef;
	      nbRowsElements[i]=r+1;
	    }
	}
    }
}


/*!
  \brief access or display operator for the spmatrix class.The arguments are two integers
    i and j. i is the indication of row. j is the indication of column.
  \param i integer
  \param j integer
  \return <T>

*/

//----------------------------------------------------------
template <class T> T spmatrix<T>::output(int i, int j) const
//----------------------------------------------------------
{
  assert (i>0 && i<=nbRows);
  assert (j>0 && j<=real_nbCol);
  T value = 0;

  int r = nbRowsElements[i];

  for(int k=1; k<=r; k++)
    if (columns_position(i,k) == j)
      value = mrows[i-1][k];
 
 return value;
}


/*!
   \brief overloads output stream for spmatrix class.
*/

//----------------------------------------------------------------------------
template <class Tf> ostream& operator << (ostream& s, const spmatrix<Tf>& mat)
//----------------------------------------------------------------------------
{
  assert (mat.nbRows>0 && mat.nbColumns>0);

  int r;

  s << "matrix size : (" << mat.nbRows << "," << mat.real_nbCol << ") \n";
  s << "maximum number of non zero elements per line : " << mat.nbColumns << endl;

  if (mat.nbRows<=15 && mat.real_nbCol<=15)
    { for (int i=1; i<=mat.nbRows; i++, s<<endl)
	for (int j=1; j<=mat.real_nbCol; j++)
	  s << mat.output(i,j) << " ";
      s << endl;
    }

  else
    { cout << "columns indices with associated coefficients : \n" ;
      for (int i=1; i<=mat.nbRows; i++, s<<endl)
	{ cout << "line " << i << endl;
          r = mat.nbRowsElements[i];
          for (int j=1; j<=r; j++)
            s << mat.columns_position[i-1][j] << "   " << mat.mrows[i-1][j] << endl;
	}
    }

  return s;
}


/*!
   \brief resize sparse matrix in case of more zeros are stocked
   \param n integer
   \return spmatrix object
*/

//--------------------------------------------------------
template <class T> spmatrix<T> spmatrix<T>::resize (int n)
//--------------------------------------------------------
{
  spmatrix<T> mat (nbRows, real_nbCol, n);

  for (int i=1; i<=nbRows; i++)
    { if (nbRowsElements[i]>n) mat.nbRowsElements[i] = n;
      else mat.nbRowsElements[i] = nbRowsElements[i];
      for (int j=1; j<=n; j++)
	{ mat.columns_position(i,j) = columns_position(i,j);
	  mat.mrows[i-1][j] = mrows[i-1][j];
	}
    }

  return mat;
}


/*!
  \brief assignment operator for the spmatrix class.
  \param mat reference on spmatrix object
  \return  reference on spmatrix object
*/

//------------------------------------------------------------------------------
template <class T> spmatrix<T>& spmatrix<T>::operator = (const spmatrix<T>& mat)
//------------------------------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns && mat.real_nbCol);

  if (!nbRows)
    { mrows = new vector<T> [nbRows=mat.nbRows]; // allocates memory for rows (no initialization)
      nbColumns = mat.nbColumns;
      real_nbCol = mat.real_nbCol;
      nbRowsElements = new vector<int> (nbRows);
    }
  else assert ( (nbRows==mat.nbRows) && (nbColumns==mat.nbColumns) && (real_nbCol==mat.real_nbCol) );

  for (int i=0; i<mat.nbRows; i++)
    mrows[i] = mat.mrows[i];

  columns_position = mat.columns_position;
  nbRowsElements = mat.nbRowsElements;

  return *this;
}


/*!
   \brief operator + for the spmatrix class.The arguments are two spmatrix objects
    mat1 and mat2. mat1 and mat2 must have the same size and contain the same number
    of non-zero element per row.
  \param mat1 reference on spmatrix object
  \param mat2 reference on spmatrix object
  \return mat1+mat2 spmatrix object 
*/

//----------------------------------------------------------------------------------------------
template <class Tf> spmatrix<Tf> operator + (const spmatrix<Tf>& mat1, const spmatrix<Tf>& mat2)
//----------------------------------------------------------------------------------------------
{
  //vrification de la compatibilit des matrices
  assert ( (mat1.nbRows == mat2.nbRows)  &&  (mat1.real_nbCol == mat2.real_nbCol) );
  assert (mat1.nbColumns && mat2.nbColumns);

  bool done = true;
  int nbEleMax = 0;
  vector<Tf> temp;
  int r1,r2;

  //cration de la matrice rsultat
  int size = min(mat1.nbColumns+mat2.nbColumns,mat1.real_nbCol);
  spmatrix<Tf> sum ( mat1.nbRows, mat1.real_nbCol, size);

  for (int i=1; i<=mat1.nbRows; i++)
    { r1 = mat1.nbRowsElements[i];
      r2 = mat2.nbRowsElements[i];
      if (r1<=r2)
	{ temp = mat1.mrows[i-1];
	  for (int k2=1; k2<=r2; k2++)
	    { done = true;
	      for (int k1=1; k1<=r1; k1++)
		if (mat2.columns_position(i,k2) == mat1.columns_position(i,k1))
		  { sum.mrows[i-1][sum.nbRowsElements[i]+1] = mat2.mrows[i-1][k2]+temp[k1];
		    temp[k1] = 0;
		    done = false;
		    //on vrifie que la somme ne donne pas un coefficient nul
		    if (abs(sum.mrows[i-1][sum.nbRowsElements[i]+1])>epsilon)
		      { sum.columns_position(i,sum.nbRowsElements[i]+1) = mat2.columns_position(i,k2);
			sum.nbRowsElements[i]+=1;
		      }
		  }
	      if (done)
		{ sum.columns_position(i,sum.nbRowsElements[i]+1) = mat2.columns_position(i,k2);
		  sum.mrows[i-1][sum.nbRowsElements[i]+1] = mat2.mrows[i-1][k2];
		  sum.nbRowsElements[i]+=1;
		}
	    }
	  for(int t=1; t<=r1; t++)
	    if (temp[t])
	      { sum.columns_position(i,sum.nbRowsElements[i]+1) = mat1.columns_position(i,t);
		sum.mrows[i-1][sum.nbRowsElements[i]+1] = temp[t];
		sum.nbRowsElements[i]+=1;
	      }
	  if (sum.nbRowsElements[i]>nbEleMax)  nbEleMax = sum.nbRowsElements[i];
	}

      else
	{ temp = mat2.mrows[i-1];
	  for (int k1=1; k1<=r1; k1++)
	    { done = true;
	      for (int k2=1; k2<=r2; k2++)
		if (mat1.columns_position(i,k1) == mat2.columns_position(i,k2))
		  { sum.mrows[i-1][sum.nbRowsElements[i]+1] = mat1.mrows[i-1][k1]+temp[k2];
		    temp[k2] = 0;
		    done = false;
		    //on vrifie que la somme ne donne pas un coefficient nul
		    if (abs(sum.mrows[i-1][sum.nbRowsElements[i]+1])>epsilon)
		      { sum.columns_position(i,sum.nbRowsElements[i]+1) = mat1.columns_position(i,k1);
			sum.nbRowsElements[i]+=1;
		      }
		  }
	      if (done)
		{ sum.columns_position(i,sum.nbRowsElements[i]+1) = mat1.columns_position(i,k1);
		  sum.mrows[i-1][sum.nbRowsElements[i]+1] = mat1.mrows[i-1][k1];
		  sum.nbRowsElements[i]+=1;
		}
	    }
	  for(int t=1; t<=r2; t++)
	    if (temp[t])
	      { sum.columns_position(i,sum.nbRowsElements[i]+1) = mat2.columns_position(i,t);
		sum.mrows[i-1][sum.nbRowsElements[i]+1] = temp[t];
		sum.nbRowsElements[i]+=1;
	      }
	  if (sum.nbRowsElements[i]>nbEleMax)  nbEleMax = sum.nbRowsElements[i];
	}
    }

  if (nbEleMax < size) return sum.resize(nbEleMax);
  else return sum;   
}


/*!
  \brief operator - for the spmatrix class.The arguments are two spmatrix object
    mat1 and mat2. mat1 and mat2 must have the same size and contain the same number
    of non-zero element per row.
  \param mat1 reference on spmatrix object
  \param mat2 reference on spmatrix object
  \return mat1-mat2 spmatrix object 
*/ 
 
 //---------------------------------------------------------------------------------------------
template <class Tf> spmatrix<Tf> operator - (const spmatrix<Tf>& mat1, const spmatrix<Tf>& mat2)
//----------------------------------------------------------------------------------------------
{
  //vrification de la compatibilit des matrices
  assert ( (mat1.nbRows == mat2.nbRows)  &&  (mat1.real_nbCol == mat2.real_nbCol) );
  assert (mat1.nbColumns && mat2.nbColumns);

  bool done = true;
  int nbEleMax = 0;
  vector<Tf> temp;
  int r1,r2;

  //cration de la matrice rsultat
  int size = min(mat1.nbColumns+mat2.nbColumns,mat1.real_nbCol);
  spmatrix<Tf> sub ( mat1.nbRows, mat1.real_nbCol, size);

  for (int i=1; i<=mat1.nbRows; i++)
    { r1 = mat1.nbRowsElements[i];
      r2 = mat2.nbRowsElements[i];
      if (r1<=r2)
	{ temp = mat1.mrows[i-1];
	  for (int k2=1; k2<=r2; k2++)
	    { done = true;
	      for (int k1=1; k1<=r1; k1++)
		if (mat2.columns_position(i,k2) == mat1.columns_position(i,k1))
		  { sub.mrows[i-1][sub.nbRowsElements[i]+1] = temp[k1]-mat2.mrows[i-1][k2];
		    temp[k1] = 0;
		    done = false;
		    //on vrifie que la diffrence ne donne pas un coefficient nul
		    if (abs(sub.mrows[i-1][sub.nbRowsElements[i]+1])>epsilon)
		      { sub.columns_position(i,sub.nbRowsElements[i]+1) = mat2.columns_position(i,k2);
			sub.nbRowsElements[i]+=1;
		      }
		  }
	      if (done)
		{ sub.columns_position(i,sub.nbRowsElements[i]+1) = mat2.columns_position(i,k2);
		  sub.mrows[i-1][sub.nbRowsElements[i]+1] = -1*mat2.mrows[i-1][k2];
		  sub.nbRowsElements[i]+=1;
		}
	    }

	  for(int t=1; t<=r1; t++)
	    if (temp[t])
	      { sub.columns_position(i,sub.nbRowsElements[i]+1) = mat1.columns_position(i,t);
		sub.mrows[i-1][sub.nbRowsElements[i]+1] = temp[t];
		sub.nbRowsElements[i]+=1;
	      }
	  if (sub.nbRowsElements[i]>nbEleMax)  nbEleMax = sub.nbRowsElements[i];
	}

      else
	{ temp = mat2.mrows[i-1];
	  for (int k1=1; k1<=r1; k1++)
	    { done = true;
	      for (int k2=1; k2<=r2; k2++)
		if (mat1.columns_position(i,k1) == mat2.columns_position(i,k2))
		  { sub.mrows[i-1][sub.nbRowsElements[i]+1] = mat1.mrows[i-1][k1]-temp[k2];
		    temp[k2] = 0;
		    done = false;
		    //on vrifie que la diffrence ne donne pas un coefficient nul
		    if (abs(sub.mrows[i-1][sub.nbRowsElements[i]+1])>epsilon)
		      { sub.columns_position(i,sub.nbRowsElements[i]+1) = mat1.columns_position(i,k1);
			sub.nbRowsElements[i]+=1;
		      }
		  }
	      if (done)
		{ sub.columns_position(i,sub.nbRowsElements[i]+1) = mat1.columns_position(i,k1);
		  sub.mrows[i-1][sub.nbRowsElements[i]+1] = mat1.mrows[i-1][k1];
		  sub.nbRowsElements[i]+=1;
		}
	    }
	  for(int t=1; t<=r2; t++)
	    if (temp[t])
	      { sub.columns_position(i,sub.nbRowsElements[i]+1) = mat2.columns_position(i,t);
		sub.mrows[i-1][sub.nbRowsElements[i]+1] = -1*temp[t];
		sub.nbRowsElements[i]+=1;
	      } 
	  if (sub.nbRowsElements[i]>nbEleMax)  nbEleMax = sub.nbRowsElements[i];
	}
    }

  if (nbEleMax < size) return sub.resize(nbEleMax);
  else return sub;
}


/*!
   \brief transform spmatrix object on matrix object.
   \return matrix object
*/    
   
//-----------------------------------------------
template <class T> matrix<T> spmatrix<T>::full ()
//-----------------------------------------------
{
  assert (nbRows);
  matrix<T> mat(nbRows,real_nbCol);
  int r;

  for (int i=1; i<=nbRows; i++)
    { r = nbRowsElements[i];
      for (int j=1; j<=r; j++)
	mat(i,columns_position(i,j)) = mrows[i-1][j];
    }

  return (mat);
}


/*!
   \brief operator * for  <T>*spmatrix<T>. The arguments are two : one reference on
   <T> elt and one reference on spmatrix<T> object mat.
   \param elt reference on <T>
   \param mat reference on spmatrix<T> object
   \return elt*mat spmatrix<T> object
*/

//----------------------------------------------------------------------------------
template <class Tf> spmatrix<Tf> operator * (const Tf& elt, const spmatrix<Tf>& mat)
//----------------------------------------------------------------------------------
{
  assert (mat.nbRows && mat.nbColumns);

  spmatrix<Tf> sp = mat;
  sp *= elt;

  return sp;
}


/*!
   \brief opertor * for spmatrix*vector.The arguments are two : one reference on
    spmatrix object mat and one reference on vector u. It must be possible to do 
    the product between mat and u.
   \param mat reference on spmatrix object
   \param u reference on vector object
   \return u* mat vector object
*/

//--------------------------------------------------------------------------------------
template <class Tf> vector<Tf> operator * (const spmatrix<Tf>& mat, const vector<Tf>& v)
//--------------------------------------------------------------------------------------
{
  assert (mat.real_nbCol == v.dim()); // checks validity of the operation

  vector<Tf> u(mat.nbRows);
  vector<Tf> *line;
  vector<int> *positions;
  int r;
  
  for (int i=1; i<=mat.nbRows; i++)
    { r = mat.nbRowsElements[i];
      line = &mat.mrows[i-1];
      positions = &mat.columns_position[i];

      for (int j=1; j<=r; j++)
	u[i] += (*line)[j] * v[(*positions)[j]];
    }

  return u;
}


/*!
   \brief opertor * for spmatrix*spmatrix.The arguments are two spmatrix objects
    mat1 and mat2. It must be possible to do the product between mat1 and mat2.
   \param mat1 reference on spmatrix object
   \param mat2 reference on spmatrix object
   \return matrix object

*/

//--------------------------------------------------------------------------------
template <class Tf> matrix<Tf> operator * (spmatrix<Tf>& mat1, spmatrix<Tf>& mat2)
//--------------------------------------------------------------------------------
{
  assert(mat1.nbColumns && mat2.nbColumns);
  assert(mat1.real_nbCol == mat2.nbRows);

  return mat1.full()*mat2.full();
}


/*!
   \brief operator / for  spmatrix<T>/<T>. The arguments are two :  one reference on spmatrix
   <T> object mat and one reference on <T> elt.
   \param mat reference on spmatrix<T> object
   \param elt reference on <T>
   \return mat/elt spmatrix<T> object
*/

//----------------------------------------------------------------------------------
template <class Tf> spmatrix<Tf> operator / (const spmatrix<Tf>& mat, const Tf& elt)
//----------------------------------------------------------------------------------
{
  assert (abs(elt) > epsilon);

  spmatrix<Tf> sp = mat;
  sp /= elt;

  return sp;
}


/*!
   \brief operator t of the sparse matrix transpose.
   \param mat reference on spmatrix object
   \return t(mat) spmatrix object
*/

//----------------------------------------------------------
template <class Tf> spmatrix<Tf> t (const spmatrix<Tf>& mat)
//----------------------------------------------------------
{  
  int r;

  //number of non-zero elements per Columns
  vector<int> c(mat.real_nbCol);

  for (int i=1; i<=mat.nbRows; i++)
    { r = mat.nbRowsElements[i];
      for (int j=1; j<=r; j++) 
	c[mat.columns_position(i,j)] += 1;
    }

  //number max of non-zero elements per Columns
  int maxcol = 0;
  for (int k=1; k<=mat.real_nbCol; k++)
    if (maxcol < c[k]) maxcol = c[k];

  spmatrix<Tf> tmat (mat.real_nbCol, mat.nbRows, maxcol);

  int nbEleMax = 0;

  for (int i=1; i<=mat.nbRows; i++)
    { r=mat.nbRowsElements[i];
      for (int j=1; j<=r; j++) 
	{ tmat.mrows[mat.columns_position(i,j)-1][tmat.nbRowsElements[mat.columns_position(i,j)]+1] = mat.mrows[i-1][j];
	  tmat.columns_position(mat.columns_position(i,j),tmat.nbRowsElements[mat.columns_position(i,j)]+1) = i;
	  tmat.nbRowsElements[mat.columns_position(i,j)]+=1;     
	}
    }

  return tmat;   
}


/*!
   \brief operator == for the spmatrix class. The arguments are two references
   on spmatrix object.
   \param mat1 reference on spmatrix object
   \param mat2 reference on spmatrix object
   \return bool
*/

//---------------------------------------------------------------------------------------
template <class Tf> bool operator == (const spmatrix<Tf>& mat1, const spmatrix<Tf>& mat2)
//---------------------------------------------------------------------------------------
{
  int result = (mat1.nbRows==mat2.nbRows && mat1.nbColumns==mat2.nbColumns && mat1.real_nbCol==mat2.real_nbCol);
  int r;

  for (int i=1; (i<=mat1.nbRows) && result; i++)
    if (mat1.nbRowsElements[i] == mat2.nbRowsElements[i])
      { r = mat1.nbRowsElements[i];
	for (int j=1; (j<=r) && result; j++)
	  result *= (abs(mat1.mrows[i-1][j]-mat2.output(i,mat1.columns_position(i,j)))<epsilon);  
      }
    else result = 0;

  return result; 
}


/*!
   \brief operator != for the spmatrix class. The arguments are two references
   on spmatrix object.
   \param mat1 reference on spmatrix object
   \param mat2 reference on spmatrix object
   \return bool
*/

//---------------------------------------------------------------------------------------
template <class Tf> bool operator != (const spmatrix<Tf>& mat1, const spmatrix<Tf>& mat2)
//---------------------------------------------------------------------------------------
{
  return !(mat1==mat2);
}


/*!
  \brief gives the trace of a squared spmatrix.
  \param mat reference on spmatrix<T> object
  \return <T>
*/

//----------------------------------------------------
template <class Tf> Tf trace (const spmatrix<Tf>& mat)
//----------------------------------------------------
{
  assert (mat.nbRows);
  assert (mat.nbRows==mat.real_nbCol); // matrix must be squared
  int r;
  Tf tr = 0;

  for (int i=1; i<=mat.nbRows; i++)
    { r = mat.nbRowsElements[i];
      for (int j=1; j<=r; j++)
	if (i == mat.columns_position(i,j)) tr += mat.mrows[i-1][j];
    }

  return tr;
}


/*!
   \brief gives the greatest element of a spmatrix.
   \param mat reference on spmatrix<T> object
   \return <T>
   
*/

//--------------------------------------------------
template <class Tf> Tf max (const spmatrix<Tf>& mat)
//--------------------------------------------------
{
  assert(mat.nbRows && mat.real_nbCol);
  int r;
  vector<Tf> *line;
  Tf max = mat.mrows[0][1];

  for (int i=1; i<=mat.nbRows; i++)
    { r = mat.nbRowsElements[i];
      line = &mat.mrows[i-1];
      for (int j=1; j<=r; j++)
	if (max < (*line)[j])  max =(*line)[j];
    }

  return max;
}


#endif
