/*  GNU Moe - My Own Editor
    Copyright (C) 2005 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <algorithm>
#include <cctype>
#include <string>
#include <vector>

#include "buffer.h"
#include "block.h"
#include "iso_8859.h"
#include "menu.h"
#include "rc.h"
#include "regex.h"
#include "screen.h"
#include "window.h"


namespace {

bool no_block() throw()
  {
  if( !Block::valid() ) { Screen::show_message( "No block" ); return true; }
  return false;
  }


bool no_block_in_this_file( const Buffer & buffer ) throw()
  {
  if( !Block::same_buffer( buffer ) )
    { Screen::show_message( "Block is not in this file" ); return true; }
  return false;
  }


bool read_only( const Buffer & buffer ) throw()
  {
  if( buffer.options.read_only )
    { Screen::show_message( "Read only" ); return true; }
  return false;
  }


void update_points( std::vector< Window > & data, const Buffer * bufp ) throw()
  {
  for( unsigned int i = 0; i < data.size(); ++i )
    if( !bufp || bufp == &data[i].buffer() )
      data[i].update_points( data[i].buffer_handle().pointer, false );
  }

} // end namespace


int Window_vector::init() throw()
  {
  _curwin = 0;
  return load( RC::options().orphan_extra ? 1 : bufhandle_vector.handles() );
  }


// 'new_size' == 0: reload same number of windows
// 'new_size' < 0: toggle 1 window <--> max_windows
//
int Window_vector::load( int new_size ) throw()
  {
  const int handles = bufhandle_vector.handles();
  const bool toggle = ( new_size < 0 && handles > 1 );
  const bool center = ( data.size() == 0 );		// center all on init

  if( !handles ) { _curwin = 0; data.clear(); return 0; }
  if( new_size == 0 ) new_size = data.size();
  else if( new_size < 0 )
    {
    for( int i = 0; i < _curwin; ++i ) bufhandle_vector.next();
    new_size = (data.size() != 1) ? 1 : handles;
    _curwin = 0;
    }
  new_size = std::min( new_size, handles );
  new_size = std::min( new_size, Screen::height() / Window::min_height );
  new_size = std::max( new_size, 1 );
  while( _curwin >= new_size ) { --_curwin; bufhandle_vector.next(); }
  int wheight = Screen::height() / new_size;

  data.clear();
  int i = 0, y = 0;
  while( i < new_size - 1 )
    {
    data.push_back( Window( y, wheight, bufhandle_vector[i], center ) );
    ++i; y += wheight;
    }
  data.push_back( Window( y, Screen::height() - y, bufhandle_vector[i], center ) );
  if( toggle && new_size == 1 ) data[0].center_cursor();
  for( i = 0; i < new_size; ++i ) data[i].repaint();
  return data.size();
  }


void Window_vector::load_file() throw()
  {
  std::string name;
  if( File::get_filename( "Name of file to edit (^C to abort): ", name ) < 0 )
    return;
  int handle;
  try { handle = bufhandle_vector.add_handle( RC::default_buffer_options(),
                                              &name, _curwin + 1 ); }
  catch( Buffer::Error e ) { Screen::show_message( e.s ); return; }
  if( name.size() ) RC::apply_regex_options( bufhandle_vector[handle].buffer() );
  ++_curwin;
  if( windows() > 1 ) load( windows() + 1 ); else load();
  }


void Window_vector::split_window() throw()
  {
  bufhandle_vector.duplicate_handle( _curwin++ );
  load( windows() + 1 );
  }


void Window_vector::save_file() const throw()
  {
  const Buffer & buffer = data[_curwin].buffer();

  std::string name = buffer.name();
  if( File::get_filename( "Name of file to save (^C to abort): ", name ) <= 0 )
    return;
  bool done;
  try { done = buffer.save( &name ); }
  catch( Buffer::Error e ) { Screen::show_message( e.s ); return; }
  if( done )
    {
    show_status_lines( &buffer );
    std::string s( "File '" ); s += name; s += "' saved";
    Screen::show_message( s );
    }
  else Screen::show_message( "File not saved" );
  }


void Window_vector::save_all_named() const throw()
  {
  const bool done = bufhandle_vector.save_all_named();
  show_status_lines();
  if( done ) Screen::show_message( "All named buffers saved" );
  else Screen::show_message( "Not all named buffers saved" );
  }


// Returns number of windows, 0 if none, -1 if aborted.
//
int Window_vector::close( bool abort, const bool quiet ) throw()
  {
  Window & w = data[_curwin];
  Buffer_handle & bh = w.buffer_handle();
  const bool last_handle = bufhandle_vector.handles( bh.buffer() ) <= 1;
  const bool modified = bh.buffer().modified();
  const bool exit_ask = RC::options().exit_ask && !quiet;

  if( !abort || !quiet )
    {
    if( abort && modified && last_handle )
      {
      int key = Screen::show_message( "Lose changes to this file ? (y/n/^C) ", true, true );
      if( key == 'n' || key == 'N' ) abort = false;
      else if( key != 'y' && key != 'Y' ) return -1;
      }
    if( !abort && modified )
      {
      std::string name = bh.buffer().name();
      if( exit_ask || !name.size() )
        while( !File::get_filename( "Name of file to save (^C to abort): ", name ) );
      if( !name.size() ) return -1;
      try { if( !bh.buffer().save( &name ) ) return -1; }
      catch( Buffer::Error e )
        { if( !quiet ) Screen::show_message( e.s ); return -1; }
      }
    else if( exit_ask && last_handle && !modified )
      {
      int key = Screen::show_message( "Abandon this file ? (y/N) ", true, true );
      if( key != 'y' && key != 'Y' ) return -1;
      }
    }
  bufhandle_vector.delete_handle( _curwin );
  if( _curwin > 0 ) --_curwin;
  return load();
  }


int Window_vector::close_and_exit( const bool abort ) throw()
  {
  int key;
  if( abort ) key = Screen::show_message( "Are you sure you want to abort ? (y/N) ", true, true );
  else key = Screen::show_message( "Are you sure you want to exit ? (y/N) ", true, true );
  if( key == 'y' || key == 'Y' )
    {
    prev();
    while( data.size() )
      { Screen::clock_handler( 0 ); if( close( abort, true ) < 0 ) break; }
    }
  return data.size();
  }


void Window_vector::next() throw()
  {
  const int handles = bufhandle_vector.handles();
  if( handles <= 1 ) return;
  if( _curwin + 1 < windows() ) ++_curwin;
  else
    {
    if( handles <= windows() ) _curwin = 0;
    else { bufhandle_vector.next(); load(); }
    }
  }


void Window_vector::prev() throw()
  {
  const int handles = bufhandle_vector.handles();
  if( handles <= 1 ) return;
  if( _curwin > 0 ) --_curwin;
  else
    {
    if( handles <= windows() ) _curwin = windows() - 1;
    else { bufhandle_vector.prev(); load(); }
    }
  }


Window & Window_vector::operator[]( int i ) throw()
  {
  if( i < 0 ) i = 0; else if( i >= windows() ) i = windows() - 1;
  return data[i];
  }


void Window_vector::add_char( unsigned char ch, const bool force ) throw()
  {
  Window & w = data[_curwin];
  if( read_only( w.buffer() ) ) return;

  if( ch == '\r' && !force ) ch = '\n';
  if( ( force || ch == '\t' || ch == '\n' || ISO_8859::isprint( ch ) ) &&
      w.buffer().pputc( w.buffer_handle().pointer, ch ) )
    { update_points( data, &w.buffer() ); repaint( &w.buffer() ); }
  }


void Window_vector::delete_char( const bool back ) throw()
  {
  Window & w = data[_curwin];
  Point & p = w.buffer_handle().pointer;

  if( !read_only( w.buffer() ) && w.buffer().pdelc( p, back ) )
    { update_points( data, &w.buffer() ); repaint( &w.buffer() ); }
  }


void Window_vector::delete_line() throw()
  {
  Window & w = data[_curwin];

  if( !read_only( w.buffer() ) && w.buffer().pdell( w.buffer_handle().pointer ) )
    { update_points( data, &w.buffer() ); repaint( &w.buffer() ); }
  }


void Window_vector::delete_block() throw()
  {
  Buffer & buffer = data[_curwin].buffer();

  if( !no_block() && !no_block_in_this_file( buffer ) &&
      !read_only( buffer ) && Block::delete_block() )
    { update_points( data, &buffer ); repaint( &buffer ); }
  }


void Window_vector::copy_block() throw()
  {
  Window & w = data[_curwin];

  if( !no_block() && !read_only( w.buffer() ) &&
      Block::copy_block( w.buffer(), w.buffer_handle().pointer ) )
    { update_points( data, &w.buffer() ); repaint(); }
  }


void Window_vector::move_block() throw()
  {
  Window & w = data[_curwin];
  Point & p = w.buffer_handle().pointer;

  if( no_block() || Block::in_block_or_end( w.buffer(), p ) ) return;
  const Buffer & buffer = Block::buffer();
  if( !read_only( w.buffer() ) && !read_only( buffer ) &&
      Block::move_block( w.buffer(), p ) )
    {
    for( unsigned int i = 0; i < data.size(); ++i )
      if( &w.buffer() == &data[i].buffer() || &buffer == &data[i].buffer() )
        data[i].update_points( data[i].buffer_handle().pointer, false );
    repaint();
    }
  }


void Window_vector::read_block() throw()
  {
  Window & w = data[_curwin];
  if( read_only( w.buffer() ) ) return;

  std::string name;
  if( File::get_filename( "Name of file to insert (^C to abort): ", name ) <= 0 )
    return;
  try {
    Buffer tmp( RC::default_buffer_options(), &name );
    if( tmp.empty() ) throw Buffer::Error( "File not found" );
    Point p = w.buffer_handle().pointer;
    if( w.buffer().pputb( p, tmp, tmp.bof(), tmp.eof() ) )
      { update_points( data, &w.buffer() ); repaint(); }
    }
  catch( Buffer::Error e ) { Screen::show_message( e.s ); }
  }


void Window_vector::write_block() const throw()
  {
  if( no_block() ) return;
  std::string name;
  if( File::get_filename( "Name of file to write (^C to abort): ", name ) <= 0 )
    return;
  try { Block::buffer().save( &name, false, false,
                              Block::begin(), Block::end() ); }
  catch( Buffer::Error e ) { Screen::show_message( e.s ); }
  }


void Window_vector::indent_block() throw()
  {
  Buffer & buffer = data[_curwin].buffer();

  if( no_block() || no_block_in_this_file( buffer ) || read_only( buffer ) )
    return;
  Point p = Block::begin();
  bool first_post = true;
  const bool ow = buffer.options.overwrite;
  buffer.options.overwrite = false;
  buffer.reset_appendable();
  while( buffer.bol( p ) < Block::end() )
    {
    p = buffer.bot( p );
    if( p.col > 0 || p < buffer.eol( p ) )
      {
      if( first_post ) first_post = false; else buffer.force_append();
      buffer.pputc( p, ' ' ); buffer.pputc( p, ' ' );
      }
    ++p.line;
    }
  buffer.reset_appendable();
  buffer.options.overwrite = ow;
  update_points( data, &buffer ); repaint( &buffer );
  }


void Window_vector::unindent_block() throw()
  {
  Buffer & buffer = data[_curwin].buffer();

  if( no_block() || no_block_in_this_file( buffer ) || read_only( buffer ) )
    return;
  Point p = Block::begin();
  bool fail = false;
  while( buffer.bol( p ) < Block::end() && !fail )
    {
    if( buffer.eol( p ).col > 0 && buffer.to_cursor( buffer.bot( p ) ).col < 2 )
      fail = true;
    ++p.line;
    }
  if( fail ) { Screen::show_message( "Can't unindent more" ); return; }

  bool first_post = true;
  const bool ow = buffer.options.overwrite;
  buffer.options.overwrite = false;
  buffer.reset_appendable();
  p = Block::begin();
  while( buffer.bol( p ) < Block::end() )
    {
    const int target_col = buffer.to_cursor( buffer.bot( p ) ).col - 2;
    if( buffer.eol( p ).col > 0 )
      while( true )
        {
        p = buffer.bot( p );
        const int col = buffer.to_cursor( p ).col;
        if( col > target_col )
          {
          if( first_post ) first_post = false; else buffer.force_append();
          buffer.pdelc( p, true );
          }
        else if( col < target_col )
          {
          if( first_post ) first_post = false; else buffer.force_append();
          buffer.pputc( p, ' ' );
          }
        else break;
        }
    ++p.line;
    }
  buffer.reset_appendable();
  buffer.options.overwrite = ow;
  update_points( data, &buffer ); repaint( &buffer );
  }


void Window_vector::undo() throw()
  {
  Window & w = data[_curwin];

  if( !read_only( w.buffer() ) )
    {
    if( w.buffer().undo( w.buffer_handle().pointer ) )
      { update_points( data, &w.buffer() ); repaint( &w.buffer() ); }
    else Screen::show_message( "Nothing to undo" );
    }
  }


void Window_vector::redo() throw()
  {
  Window & w = data[_curwin];

  if( !read_only( w.buffer() ) )
    {
    if( w.buffer().redo( w.buffer_handle().pointer ) )
      { update_points( data, &w.buffer() ); repaint( &w.buffer() ); }
    else Screen::show_message( "Nothing to redo" );
    }
  }


void Window_vector::center_line() throw()
  {
  Window & w = data[_curwin];
  Buffer & buffer = w.buffer();
  const Point bol = buffer.bol( w.buffer_handle().pointer );
  const Point bot = buffer.bot( bol );
  const Point cbot = buffer.to_cursor( bot );
  const Point ceot = buffer.to_cursor( buffer.eot( bol ) );
  const int l = cbot.col - buffer.options.lmargin();
  const int r = buffer.options.rmargin() - ceot.col;
  int dif = std::max( -bot.col, ( r - l ) / 2 );

  if( !read_only( buffer ) && cbot < ceot && dif )
    {
    buffer.reset_appendable();
    if( dif > 0 || bot != cbot )
      {
      if( bot != cbot )
        { buffer.pdelb( bol, bot ); dif = ( l + r ) / 2; buffer.force_append(); }
      Point p = bol; while( --dif >= 0 ) buffer.pputc( p, ' ' );
      }
    else buffer.pdelb( bol, Point( bot.line, -dif ) );
    buffer.reset_appendable();
    update_points( data, &buffer ); repaint( &buffer );
    }
  }


void Window_vector::reformat() throw()
  {
  Window & w = data[_curwin];
  Buffer & buffer = w.buffer();
  const Point bop = buffer.bop( w.buffer_handle().pointer );
  const Point eop = buffer.eop( w.buffer_handle().pointer );
  const Point bot = buffer.bot( bop );

  if( !read_only( buffer ) && bop < eop )
    {
    Buffer tmp( buffer.options );
    tmp.options.auto_indent = false;
    tmp.options.overwrite = false;
    tmp.options.word_wrap = true;
    Point dummy = tmp.bof();
    if( bop < bot && buffer.to_cursor( bot ).col > buffer.options.lmargin() )
      tmp.Basic_buffer::pputb( dummy, buffer, bop, bot );
    bool space = false;
    for( Point p = bot; p < eop; )
      {
      unsigned char ch = buffer.pgetc( p );
      if( !std::isspace( ch ) )
        { tmp.pputc( dummy, ch ); dummy = tmp.eof(); space = false; }
      else if( !space ) { tmp.pputc( dummy, ' ' ); space = true; }
      }
    if( space ) tmp.pdelc( dummy, true );
    if( dummy != tmp.bol( dummy ) ) tmp.pputc( dummy, '\n' );
    bool change = ( tmp.eof().line != ( eop.line - bop.line ) );
    if( !change )
      for( int line = 0; line < tmp.lines(); ++line )
        if( tmp.characters( line ) != buffer.characters( bop.line + line ) &&
            ( !tmp.blank( line ) || !buffer.blank( bop.line + line ) ) )
          { change = true; break; }
    if( change )
      {
      buffer.reset_appendable();
      buffer.pdelb( bop, eop );
      buffer.force_append();
      Point p = bop; buffer.pputb( p, tmp, tmp.bof(), tmp.eof() );
      buffer.reset_appendable();
      update_points( data, &buffer ); repaint( &buffer );
      }
    }
  }


void Window_vector::toggle_marking() const throw()
  {
  const Window & w = data[_curwin];
  const char * msg = Block::toggle_marking( w.buffer(), w.buffer_handle().pointer );
  if( msg ) { repaint(); Screen::show_message( msg ); }
  }


void Window_vector::repaint( const Buffer * bufp ) const throw()
  {
  for( int i = 0; i < windows(); ++i )
    if( !bufp || bufp == &data[i].buffer() )
      data[i].repaint();
  }


void Window_vector::show_status_lines( const Buffer * bufp, const char * prefix ) const throw()
  {
  for( int i = 0; i < windows(); ++i )
    if( !bufp || bufp == &data[i].buffer() )
      data[i].show_status_line( prefix );
  }


void Window_vector::search( const bool again ) throw()
  {
  static std::vector< std::string > search_history;
  static std::vector< std::string > replace_history;
  static std::string search_string;
  static std::string replace_string;
  static bool backward = false, block = false, global = false, icase = false;
  static bool replace = false;

  if( !again || search_history.size() == 0 )
    {
    search_string.clear();
    search_history.push_back( std::string() );
    if( Screen::get_string( "Find (^C to abort): ", search_history, true ) <= 0 ) return;
    search_string = search_history.back(); ISO_8859::deescapize( search_string );
    std::string prompt;
    if( RC::options().ignore_case )
      prompt = "(N)o_ignore (R)eplace (B)ackwards (G)lobal Bloc(K) (^C to abort): ";
    else prompt = "(I)gnore (R)eplace (B)ackwards (G)lobal Bloc(K) (^C to abort): ";
    std::vector< std::string > mode;
    mode.push_back( std::string() );
    const int result = Screen::get_string( prompt, mode );
    if( result < 0 ) return;
    backward = false; block = false; global = false; replace = false;
    icase = RC::options().ignore_case;
    if( result > 0 )
      {
      std::string & m = mode.back();
      for( unsigned int i = 0; i < m.size(); ++i )
        switch( ISO_8859::toupper( m[i] ) )
          {
          case 'B': backward = true; break;
          case 'G': global = true; break;
          case 'I': icase = true; break;
          case 'K': block = true; break;
          case 'N': icase = false; break;
          case 'R': replace = true; break;
          }
      }
    if( replace )
      {
      replace_string.clear();
      replace_history.push_back( std::string() );
      const int len = Screen::get_string( "Replace with (^C to abort): ", replace_history, true );
      if( len < 0 ) { replace = false; return; }
      if( len > 0 )
        { replace_string = replace_history.back();
          ISO_8859::deescapize( replace_string ); }
      }
    }

  Point p = curwin().buffer_handle().pointer;
  if( block )
    {
    if( no_block() || no_block_in_this_file( curwin().buffer() ) ) return;
//    if( RC::options().rectangle_mode )
//      { Screen::show_message( "Can't search block in rectangle mode" ); return; }
    if( backward ) { if( p > Block::end() ) p = Block::end(); }
    else { if( p < Block::begin() ) p = Block::begin(); }
    }
  bool found = false, ask = true, found_any = false, wrapped = false;
  for( int i = (global ? bufhandle_vector.handles() - 1 : 0); i >= 0 && !found; --i )
    {
    Window & w = data[_curwin];
    std::vector< std::string > regex_pieces;
    Point p1 = p, p2 = p;
    found = Regex::find( w.buffer(), p1, p2, search_string, regex_pieces, icase, backward );
    if( backward ) p = p1; else p = p2;

    if( found )
      {
      if( block && ( p1 < Block::begin() || p2 > Block::end() ) ) break;
      found_any = true;
      if( !replace ) w.update_points( p, true, true );
      else
        {
        std::string error;
        Block::save_block_position(); w.update_points( p, false, true );
        if( ask ) { Block::set_block( w.buffer(), p1, p2 ); w.repaint(); }
        while( true )
          {
          int key = ask ? Screen::show_message( "Replace (Y)es (N)o (R)est (^C to abort): ", true ) : 'y';
          if( key == 3 ) break;
          if( key == 'n' || key == 'N' ) { found = false; break; }
          if( key == 'r' || key == 'R' ) { key = 'y'; ask = false; }
          if( key == 'y' || key == 'Y' )
            {
            if( Regex::replace( w.buffer(), p1, p2, replace_string, regex_pieces ) )
              {
              found = false; if( backward ) p = p1; else p = p2;
              w.update_points( p, false, true );
              }
            else if( w.buffer().options.read_only ) error = "Read only";
            else if( p2 < w.buffer().eof() ) error = "Replace error: check syntax";
            break;
            }
          }
        ++i; Block::restore_block_position(); if( ask || found ) w.repaint();
        if( error.size() ) Screen::show_message( error );
        }
      continue;
      }
    if( !ask ) w.repaint();
    if( block ) break;
    if( global && !wrapped )
      {
      if( backward ) { prev(); p = curwin().buffer().eof(); }
      else { next(); p = curwin().buffer().bof(); }
      }
    if( ( ( global && i == 0 ) || RC::options().search_wrap ) && !wrapped )
      {
      if( backward ) p = curwin().buffer().eof();
      else p = w.buffer().bof();
      wrapped = true; ++i;
      }
    }
  if( !found_any )
    {
    if( block ) Screen::show_message( "Not found (search restricted to marked block)" );
    else Screen::show_message( "Not found" );
    }
  else if( wrapped ) Screen::show_message( "Wrapped" );
  }
