/*  Moe - My Own Editor
    Copyright (C) 2005 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <algorithm>
#include <cctype>
#include <cerrno>
#include <cstdlib>
#include <string>
#include <vector>
#include <unistd.h>
#include <ncurses.h>

#include "buffer.h"
#include "block.h"
#include "iso_8859.h"
#include "menu.h"
#include "rc.h"
#include "screen.h"
#include "window.h"


namespace Screen {

struct Lines_and_cursor
  {
  int line, lines;
  std::vector< chtype > buf;
  Point cursor;

  Lines_and_cursor( const int ln, const int lns ) throw()
    : line( ln ), lines( lns ) {}
  };

int _height, _width;
int clock_limit;		//   0: disable clock
				//   1: showing help
				// > 1: first line used by menu
Window_vector window_vector;	// Vector of windows shown on screen
std::vector< Lines_and_cursor > lines_and_cursor_vector;


void dummy_endwin() throw() { endwin(); }

void initialize_ncurses() throw()
  {
  atexit( dummy_endwin );
  initscr();			// initializes curses data structures
  raw();			// read a char at a time and disable signals
  keypad( stdscr, true );	// enables single value for function keys
  nodelay( stdscr, false );	// forces getch() to wait for key
  noecho();			// disables echoing of getch()
  nonl();			// disables CR LF translation
  scrollok( stdscr, false );	// Disables automatic scrolling
  }


void rep_addch( int n, const chtype ch )
  { for( ; n > 0; --n ) waddch( stdscr, ch ); }


// '@b' sets-resets bold mode
// '@i' sets-resets inverse video mode
// '@u' sets-resets underline mode
// '@@' shows a '@'
//
void out_string( const std::string & s, int len, int pos = 0 ) throw()
  {
  if( len > _width ) len = _width;
  chtype atr = A_NORMAL;
  int x;
  for( x = 0; x < len && pos < (int)s.size(); ++pos )
    {
    unsigned char ch = s[pos];
    if( ch == '@' && pos + 1 < (int)s.size() )
      switch( s[pos+1] )
        {
        case '@': ++pos; break;
        case 'b': atr ^= A_BOLD; ++pos; ch = 0; break;
        case 'i': atr ^= A_REVERSE; ++pos; ch = 0; break;
        case 'u': atr ^= A_UNDERLINE; ++pos; ch = 0; break;
        }
    if( ch ) { waddch( stdscr, atr | ch ); ++x; }
    }
  if( x < len ) rep_addch( len - x, atr | ' ' );
  }


void remove_duplicates( std::vector< std::string > & history ) throw()
  {
  for( unsigned int i = 0; i + 1 < history.size(); )
    {
    if( history[i].size() && history[i] != history.back() ) ++i;
    else history.erase( history.begin() + i );
    }
  }


void alt_key_loop( int key ) throw()
  {
  Window & w = window_vector.curwin();
  w.show_status_line( ISO_8859::control_name( key ) );
  do key = wait_kbhit(); while( key < 0 );
  w.show_status_line();
  key = ISO_8859::decontrolize( key );
  if( std::isdigit( key ) ) w.goto_mark( key - '0' );
  else switch( key )
    {
    case 'A': w.scroll_horizontal( -8 ); break;
    case 'B': w.goto_begin_of_block(); break;
    case 'C': w.goto_column(); break;
    case 'D': w.center_cursor(); break;
    case 'G': w.goto_matching_delimiter(); break;
    case 'H': w.goto_matching_delimiter( false ); break;
    case 'K': w.goto_end_of_block(); break;
    case 'L': w.goto_line(); break;
    case 'S': w.scroll_horizontal( 8 ); break;
    case 'U': w.goto_bof(); break;
    case 'V': w.goto_eof(); break;
    case 'W': w.scroll_page( false ); break;
    case 'Z': w.scroll_page( true ); break;
    }
  }


void control_k_loop( int key ) throw()
  {
  Window & w = window_vector.curwin();
  w.show_status_line( ISO_8859::control_name( key ) );
  do key = wait_kbhit(); while( key < 0 );
  w.show_status_line();
  key = ISO_8859::decontrolize( key );
  if( std::isdigit( key ) ) w.set_mark( key - '0' );
  else switch( key )
    {
    case 'B': Block::set_begin( w.buffer(), w.buffer_handle().pointer );
              window_vector.repaint(); break;
    case 'C': window_vector.copy_block(); break;
    case 'I': window_vector.indent_block(); break;
    case 'K': Block::set_end( w.buffer(), w.buffer_handle().pointer );
              window_vector.repaint(); break;
    case 'M': window_vector.move_block(); break;
    case 'R': window_vector.read_block(); break;
    case 'U': window_vector.unindent_block(); break;
    case 'W': window_vector.write_block(); break;
    case 'Y': window_vector.delete_block(); break;
    }
  }


void control_o_loop( int key ) throw()
  {
  Window & w = window_vector.curwin();
  w.show_status_line( ISO_8859::control_name( key ) );
  do key = wait_kbhit(); while( key < 0 );
  w.show_status_line();
  key = ISO_8859::decontrolize( key );
  switch( key )
    {
    case 'C': window_vector.center_line(); break;
    case 'R': window_vector.repaint(); break;
    case 'S': window_vector.split_window(); break;
    }
  }


void control_q_loop( int key, int & retval ) throw()
  {
  Window & w = window_vector.curwin();
  w.show_status_line( ISO_8859::control_name( key ) );
  do key = wait_kbhit(); while( key < 0 );
  w.show_status_line();
  key = ISO_8859::decontrolize( key );
  switch( key )
    {
    case 'C': if( window_vector.close_and_exit( true ) == 0 ) retval = 0; break;
    case 'X': if( window_vector.close_and_exit( false ) == 0 ) retval = 0; break;
    }
  }


void user_control_char() throw()
  {
  const std::string prompt = "Ctrl(A-Z) or 0-255 or 0x00-0xFF or 0-0377 or Escape sequence: ";
  int key;
  while( true )
    { key = show_message( prompt, true, true );
      if( key == KEY_F(1) ) Help::help_menu( key ); else break; }
  if( key < 0 || key >= 256 || key == 3 ) return;
  if( key != '\\' && !std::isdigit( key ) )
    { key = ISO_8859::controlize( key ); if( key >= 32 ) key = -1; }
  else
    {
    std::vector< std::string > dummy;		// dummy history
    dummy.push_back( std::string() );
    dummy.back() += key;
    if( get_string( prompt, dummy ) <= 0 ) return;
    if( dummy.back()[0] == '\\' )
      key = ISO_8859::escape( std::string( dummy.back(), 1 ) );
    else if( !RC::parse_int( dummy.back(), key ) ) key = -1;
    }
  if( key < 0 || key >= 256 ) show_message( "Invalid code" );
  else window_vector.add_char( key, true );
  }

} // end namespace Screen


bool Screen::init() throw()
  {
  initialize_ncurses();
  getmaxyx( stdscr, _height, _width );
  if( _height < 24 || _width < 80 ) return false;
  clock_limit = 0;
  window_vector.init();
  return true;
  }


//void Screen::beep() throw() { ::beep(); wrefresh( stdscr ); }


// Wait for a keystroke.
// The clock only blinks when waiting here for a keystroke.
// The screen is also only refreshed here (and in clock_handler).
// This function is the real "main loop" of the program.
//
int Screen::wait_kbhit( const int new_clock_limit, const bool get_key ) throw()
  {
  if( new_clock_limit < 0 )
    { move_to( window_vector.curwin().absolute_cursor() ); }

  clock_limit = ( new_clock_limit >= 0 ) ? new_clock_limit : _height;
  if( clock_limit > 0 ) { clock_handler( 0 ); alarm( 1 ); }
  else wrefresh( stdscr );

  int key = wgetch( stdscr );
  alarm( 0 );
  if( key == ERR ) return -1;
  if( !get_key ) ungetch( key );
  return key;
  }


void Screen::clock_handler( const int sig ) throw()	// updates the clock
  {
  static char c = ':';

  if( sig ) { alarm( 1 ); c = (c != ':')? ':' : ' '; } else c = ':';
  time_t dummy = time(0); struct tm *t = localtime( &dummy );
  if( t )
    {
    int h = t->tm_hour%12; if( h == 0 ) h = 12;
    char buf[16];
    std::snprintf( buf, sizeof( buf ), "@i%2d%c%02d", h, c, t->tm_min );
    std::string s( buf );
    Point old_cursor; save_cursor( old_cursor );
    if( clock_limit == 1 ) { wmove( stdscr, 0, 60 ); out_string( s, 5 ); }
    else for( int i = 0; i < window_vector.windows(); ++i )
      {
      Point cursor = window_vector[i].clock_position();
      if( cursor.line >= clock_limit ) break;
      move_to( cursor ); out_string( s, 5 );
      }
    move_to( old_cursor );
    if( sig ) wrefresh( stdscr );
    }
  }


// 'history.back()' contains the default result (may be empty), and
// stores the result (if any).
// Returns the size of the result string, 0 if none, -1 if aborted.
//
int Screen::get_string( const std::string & prompt,
                        std::vector< std::string > & history, const bool usetab ) throw()
  {
  const int line = _height - 1, lines = 1;
  save_lines_and_cursor( line, lines );

  const std::string original = history.back();
  remove_duplicates( history );
  std::string s( prompt );
  s += history.back();
  unsigned int scol = s.size(), idx = history.size() - 1;
  int key = 0;
  bool modified = false;
  while( key >= 0 )
    {
    int pos = scol; pos -= ( _width - 1 ); if( pos < 0 ) pos = 0;
    wmove( stdscr, line, 0 );
    out_string( s, _width, pos );
    wmove( stdscr, line, scol - pos );
    do key = wait_kbhit( line ); while( key < 0 );
    switch( key )
      {
      case   3 : key = -1; break;		// ^C
      case '\t': if( usetab )
                   {
                   std::string name( s, prompt.size(), s.size() );
                   if( !File::file_menu( prompt, name ) ) { key = -1; break; }
                   s.erase( prompt.size(), s.size() ); s += name;
                   scol = s.size(); modified = true;
                   }
                 break;
      case '\r': { std::string name( s, prompt.size(), s.size() );
                   if( !usetab || !name.size() ||
                       ( File::test_slash_filename( name ) && File::is_regular( name ) ) )
                     { key = -2; history.back() = name; } }
                 break;
      case KEY_F(1) : Help::help_menu( key ); break;
      case KEY_DC   : if( scol < s.size() )
                        { s.erase( scol, 1 ); modified = true; } break;
      case KEY_BACKSPACE: if( scol > prompt.size() )
                            { s.erase( --scol, 1 ); modified = true; } break;
      case KEY_HOME : scol = prompt.size(); break;
      case KEY_END  : scol = s.size(); break;
      case KEY_LEFT : if( scol > prompt.size() ) { --scol; } break;
      case KEY_RIGHT: if( scol < s.size() ) { ++scol; } break;
      case KEY_UP   : if( idx > 0 )
        {
        if( modified )
          {
          modified = false;
          if( s.size() > prompt.size() )
            history[idx].assign( s, prompt.size(), s.size() );
          else if( idx == history.size() - 1 ) history[idx].clear();
          else history.erase( history.begin() + idx );
          }
        s.erase( prompt.size(), s.size() ); s += history[--idx]; scol = s.size();
        } break;
      case KEY_DOWN : if( idx + 1 < history.size() )
        {
        if( modified )
          {
          modified = false;
          if( s.size() > prompt.size() )
            history[idx].assign( s, prompt.size(), s.size() );
          else { history.erase( history.begin() + idx ); --idx; }
          }
        s.erase( prompt.size(), s.size() ); s += history[++idx]; scol = s.size();
        } break;
      default: if( key < 256 && ISO_8859::isprint( key ) )
                 { s.insert( scol++, 1, key ); modified = true; }
      }
    }

  restore_lines_and_cursor();
  if( original.size() && original != history.back() )
    history.insert( history.end() - 1, original );
  if( key != -1 ) remove_duplicates( history );
  if( ( key == -1 && !original.size() ) || history.back().size() == 0 )
    { history.pop_back(); if( key != -1 ) return 0; }
  if( key == -1 ) return -1;
  return history.back().size();
  }


int Screen::height() throw() { return _height; }

int Screen::width() throw() { return _width; }


void Screen::move_to( const Point & cursor ) throw()
  { wmove( stdscr, cursor.line, cursor.col ); }


void Screen::save_cursor( Point & cursor ) throw()
  { getyx( stdscr, cursor.line, cursor.col ); }


void Screen::save_lines_and_cursor( const int line, const int lines ) throw()
  {
  lines_and_cursor_vector.push_back( Lines_and_cursor( line, lines ) );
  Lines_and_cursor & lc = lines_and_cursor_vector.back();

  save_cursor( lc.cursor );
  for( int y = line; y < line + lines; ++y )
    for( int x = 0; x < _width; ++x )
      lc.buf.push_back( mvinch( y, x ) );
  }


void Screen::restore_lines_and_cursor() throw()
  {
  const Lines_and_cursor & lc = lines_and_cursor_vector.back();

  for( int y = lc.line, i = 0; y < lc.line + lc.lines; ++y )
    for( int x = 0; x < _width; ++x )
      { wmove( stdscr, y, x ); waddch( stdscr, lc.buf[i++] ); }
  move_to( lc.cursor );
  lines_and_cursor_vector.pop_back();
  }


void Screen::out_buffer_line( const Buffer & buffer, Point c, const int line ) throw()
  {
  wmove( stdscr, line, 0 );
  if( c.line < 0 || c.line >= buffer.lines() )
    { rep_addch( _width, ' ' ); return; }
  Point pc, p = buffer.to_pointer( c, pc );
  if( pc.col < c.col ) pc.col = c.col;		// landed in a tab

  for( int i = 0; i < _width; ++i, ++c.col, ++p.col )
    {
    int ch = buffer[p];
    if( ch < 0 || ( p == buffer.eol( p ) && ch == '\n' ) )
      { rep_addch( _width - i, ' ' ); break; }
    int block_attr = Block::in_block( buffer, p ) ? A_REVERSE : 0;
    if( ch == '\t' )
      {
      ch = ' ';
      while( i + 1 < _width && c.col % 8 != 7 )
        { waddch( stdscr, ch | block_attr ); ++i; ++c.col; }
      }
    else if( ch < 32 ) ch = ( ch + '@' ) | A_UNDERLINE;
    else if( ch >= 127 && ch < 160 )
      ch = ( ch + '@' - 128 ) | A_REVERSE | A_UNDERLINE;
    waddch( stdscr, ch | block_attr );
    }
  }


bool Screen::out_line( const std::string & s, const int line, const bool shift ) throw()
  {
  if( line < 0 || line >= _height ) return false;
  int pos = s.size(); pos -= ( _width - 1 );
  if( !shift || pos < 0 ) pos = 0;
  wmove( stdscr, line, 0 );
  out_string( s, _width, pos );
  return true;
  }

/*
void Screen::repaint( const Buffer * bufp ) throw()
  { window_vector.repaint( bufp ); }
*/

int Screen::show_message( const std::string & s, const bool get_key,
                          const bool take_cursor ) throw()
  {
  const int line = _height - 1;

  save_lines_and_cursor( line, 1 );

  wmove( stdscr, line, 0 ); out_string( s, _width );
  wmove( stdscr, line, std::min( (int)s.size(), _width - 1 ) );
  int key;
  do key = wait_kbhit( take_cursor ? line : -1, get_key ); while( key < 0 );

  restore_lines_and_cursor();
  return key;
  }


void Screen::show_status_lines() throw()
  { window_vector.show_status_lines(); }


Screen::Keys Screen::convert_key( const int key ) throw()
  {
  switch( key )
    {
    case KEY_UP   : return key_up;
    case KEY_DOWN : return key_down;
    case KEY_LEFT : return key_left;
    case KEY_RIGHT: return key_right;
    case KEY_NPAGE: return key_npage;
    case KEY_PPAGE: return key_ppage;
    case KEY_HOME : return key_home;
    case KEY_END  : return key_end;
    default: return key_invalid;
    }
  }


int Screen::user_loop() throw()
  {
  int retval = -1;
  while( retval < 0 )
    {
    int key;
    do key = wait_kbhit(); while( key < 0 );
    Window & w = window_vector.curwin();
    switch( key )
      {
      case  0: window_vector.toggle_marking(); break;	// ^SPACE
      case  2: window_vector.reformat(); break;		// ^B
      case  3: if( !window_vector.close( true ) ) retval = 0; break;  // ^C
      case  6: window_vector.search(); break;		// ^F
      case 11: control_k_loop( key ); break;		// ^K
      case 12: window_vector.search( true ); break;	// ^L
      case 15: control_o_loop( key ); break;		// ^O
      case 16: user_control_char(); break;		// ^P
      case 17: control_q_loop( key, retval ); break;	// ^Q
      case 19: w.show_character_info(); break;		// ^S
      case 24: if( !window_vector.close( false ) ) retval = 0; break; // ^X
      case 25: window_vector.delete_line(); break;	// ^Y
      case 27: alt_key_loop( key ); break;		// Alt or Esc
      case KEY_F(1) : Help::help_menu( key ); break;
      case KEY_F(2) : window_vector.save_file(); break;
      case KEY_F(3) : window_vector.load_file(); break;
      case KEY_F(4) : window_vector.load( -1 ); break;
      case KEY_F(5) : window_vector.prev(); break;
      case KEY_F(6) : window_vector.next(); break;
      case KEY_F(7) : window_vector.undo(); break;
      case KEY_F(8) : window_vector.redo(); break;
      case KEY_F(9) : window_vector.copy_block(); break;
      case KEY_F(10): Menu::options_menu( w.buffer(), key ); break;
      case KEY_BACKSPACE: window_vector.backspace(); break;
      case KEY_DC   : window_vector.delete_char(); break;
      case KEY_IC   : w.buffer().options.overwrite = !w.buffer().options.overwrite;
                      window_vector.show_status_lines( &w.buffer() ); break;
      case KEY_HOME : w.goto_home(); break;
      case KEY_END  : w.goto_eol(); break;
      case KEY_PPAGE: w.move_page( false ); break;
      case KEY_NPAGE: w.move_page( true ); break;
      case KEY_UP   : w.move_vertical( -1 ); break;
      case KEY_DOWN : w.move_vertical( +1 ); break;
      case KEY_LEFT : w.goto_pprev(); break;
      case KEY_RIGHT: w.goto_pnext(); break;
      default       : if( key < 256 ) { window_vector.add_char( key ); break; }
      }
    if( Block::follow_marking( w.buffer(), w.buffer_handle().pointer ) )
      window_vector.repaint( &w.buffer() );
    }
  wmove( stdscr, _height - 1, 0 );	// clear bottom line on exit
  wclrtoeol( stdscr ); wrefresh( stdscr );
  return retval;
  }
