/*  Moe - My Own Editor
    Copyright (C) 2005 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <cstdio>
#include <cstring>
#include <string>
#include <vector>

#include "buffer.h"
#include "menu.h"
#include "rc.h"
#include "regex.h"


namespace RC {

// priority of _default_buffer_options by place of specification:
// ( 0 = lowest, 3 = highest )
// 0 default priority
// 1 priority of option specified in rc file as 'default file option'
// 2 priority of option specified in rc file as 'file name dependent option'
// 3 priority of option specified in command line as 'default file option'
//
struct Buffer_options_priorities
  {
  int lmargin, rmargin;
  int auto_indent, overwrite, read_only, word_wrap;

  Buffer_options_priorities() throw()
    : lmargin( 0 ), rmargin( 0 ),
    auto_indent( 0 ), overwrite( 0 ), read_only( 0 ), word_wrap( 0 ) {}

  void reset() throw() { *this = Buffer_options_priorities(); }
  };


struct Regex_options
  {
  std::vector< std::string > regex_vector;
  Buffer::Options buffer_options;
  };


Options _editor_options;
Buffer::Options _default_buffer_options;
Buffer_options_priorities priorities;
std::vector< Regex_options > regex_options_vector;


enum Optcode
  {
  auto_unmark, no_auto_unmark, backup, no_backup, /*beep, no_beep,*/
  exit_ask, no_exit_ask, ignore_case, no_ignore_case, keep_lines,
  orphan_extra, /*rectangle_mode, no_rectangle_mode,*/ search_wrap,
  no_search_wrap, smart_home, no_smart_home, auto_indent,
  no_auto_indent, lmargin, rmargin, overwrite, no_overwrite, read_only,
  no_read_only, word_wrap, no_word_wrap,
  END_OPTS
  };


struct Entry
  {
  Optcode optcode;		// Option code
  const char * name;		// Option name
  };

Entry table[] = {
    { auto_unmark,       "-u" },
    { auto_unmark,       "--auto-unmark" },
    { no_auto_unmark,    "--no-auto-unmark" },
    { backup,            "-b" },
    { backup,            "--backup" },
    { no_backup,         "--no-backup" },
//    { beep,              "-p" },
//    { beep,              "--beep" },
//    { no_beep,           "--no-beep" },
    { exit_ask,          "-e" },
    { exit_ask,          "--exit-ask" },
    { no_exit_ask,       "--no-exit-ask" },
    { ignore_case,       "-i" },
    { ignore_case,       "--ignore-case" },
    { no_ignore_case,    "--no-ignore-case" },
    { keep_lines,        "-k" },
    { keep_lines,        "--keep-lines" },
    { orphan_extra,      "--orphan" },
//    { rectangle_mode,    "-x" },
//    { rectangle_mode,    "--rectangle" },
//    { no_rectangle_mode, "--no-rectangle" },
    { search_wrap,       "-s" },
    { search_wrap,       "--search-wrap" },
    { no_search_wrap,    "--no-search-wrap" },
    { smart_home,        "-H" },
    { smart_home,        "--smart-home" },
    { no_smart_home,     "--no-smart-home" },
    { auto_indent,       "-a" },
    { auto_indent,       "--auto-indent" },
    { no_auto_indent,    "--no-auto-indent" },
    { lmargin,           "-l" },
    { lmargin,           "--lmargin" },
    { rmargin,           "-r" },
    { rmargin,           "--rmargin" },
    { overwrite,         "-O" },
    { overwrite,         "--overwrite" },
    { no_overwrite,      "--no-overwrite" },
    { read_only,         "-o" },
    { read_only,         "--read-only" },
    { no_read_only,      "--no-read-only" },
    { word_wrap,         "-w" },
    { word_wrap,         "--word-wrap" },
    { no_word_wrap,      "--no-word-wrap" },
    { END_OPTS, 0 } };


Optcode to_optcode( const char *optname ) throw()
  {
  for( int i = 0; table[i].name; ++i )
    if( strcmp( optname, table[i].name ) == 0 )
      return table[i].optcode;
  return END_OPTS;
  }


bool is_editor_option( Optcode optcode ) throw()
  {
  switch( optcode )
    {
    case auto_unmark:
    case no_auto_unmark:
    case backup:
    case no_backup:
//    case beep:
//    case no_beep:
    case exit_ask:
    case no_exit_ask:
    case ignore_case:
    case no_ignore_case:
    case keep_lines:
    case orphan_extra:
//    case rectangle_mode:
//    case no_rectangle_mode:
    case smart_home:
    case no_smart_home:
    case search_wrap:
    case no_search_wrap:
         return true;
    case auto_indent:
    case no_auto_indent:
    case lmargin:
    case rmargin:
    case overwrite:
    case no_overwrite:
    case read_only:
    case no_read_only:
    case word_wrap:
    case no_word_wrap:
    case END_OPTS:
         return false;
    }
  return false;		// keep compiler happy
  }


bool is_buffer_option( Optcode optcode ) throw()
  {
  return ( !is_editor_option( optcode ) && optcode != END_OPTS );
  }


bool optarg_included( char **optname, char **optarg ) throw()
  {
  static char buf[3];
  char *name = *optname;
  if( name[0] == '-' )
    {
    if( name[1] == '-' )
      {
      for( int i = 2; name[i]; ++i )
        if( name[i] == '=' )
          { name[i] = 0; *optarg = name + i + 1; return true; }
      }
    else if( name[1] != 0 && name[2] != 0 )
      {
      buf[0] = '-'; buf[1] = name[1]; buf[2] = 0;
      *optname = buf; *optarg = name + 2; return true;
      }
    }
  return false;
  }


void set_bool_option( bool & option, bool value,
                      int & option_priority, int priority = 0 ) throw()
  {
  if( !priority || priority >= option_priority ) option = value;
  if( priority > option_priority ) option_priority = priority;
  }


bool set_int_option( int & option, const char *optarg,
                      int & option_priority, int priority = 0 ) throw()
  {
  bool good = true;
  if( ( !priority || priority >= option_priority ) && optarg )
    good = parse_int( optarg, option );
  if( priority > option_priority ) option_priority = priority;
  return good;
  }


// return value: no. of fields taken (1 or 2), or 0 if option not found.
//
int set_option_from_name( char *optname, char *optarg,
                          Options *edop, Buffer::Options *bufop,
                          int priority = 0, char *msg = "" )
  {
  bool arg_included = optarg_included( &optname, &optarg );
  bool arg_needed = false;
  Optcode optcode = to_optcode( optname );

  int errcode = 0;
  if( ( is_editor_option( optcode ) && !edop ) ||
      ( is_buffer_option( optcode ) && !bufop ) ) errcode = 2;
  else switch( optcode )
    {
    case auto_unmark:       edop->auto_unmark = true; break;
    case no_auto_unmark:    edop->auto_unmark = false; break;
    case backup:            edop->backup = true; break;
    case no_backup:         edop->backup = false; break;
//    case beep:              edop->beep = true; break;
//    case no_beep:           edop->beep = false; break;
    case exit_ask:          edop->exit_ask = true; break;
    case no_exit_ask:       edop->exit_ask = false; break;
    case ignore_case:       edop->ignore_case = true; break;
    case no_ignore_case:    edop->ignore_case = false; break;
    case keep_lines:        if( !parse_int( optarg, edop->keep_lines ) ) errcode = 5;
                            arg_needed = true; break;
    case orphan_extra:      edop->orphan_extra = true; break;
//    case rectangle_mode:    edop->rectangle_mode = true; break;
//    case no_rectangle_mode: edop->rectangle_mode = false; break;
    case smart_home:        edop->smart_home = true; break;
    case no_smart_home:     edop->smart_home = false; break;
    case search_wrap:       edop->search_wrap = true; break;
    case no_search_wrap:    edop->search_wrap = false; break;
    case auto_indent:
      set_bool_option( bufop->auto_indent, true, priorities.auto_indent, priority ); break;
    case no_auto_indent:
      set_bool_option( bufop->auto_indent, false, priorities.auto_indent, priority ); break;
    case lmargin:
      {
      int i;
      if( !set_int_option( i, optarg, priorities.lmargin, priority ) ) errcode = 5;
      else if( !bufop->set_lmargin( i - 1 ) ) errcode = 6;
      arg_needed = true;
      } break;
    case rmargin:
      {
      int i;
      if( !set_int_option( i, optarg, priorities.rmargin, priority ) ) errcode = 5;
      else if( !bufop->set_rmargin( i - 1 ) ) errcode = 6;
      arg_needed = true;
      } break;
    case overwrite:
      set_bool_option( bufop->overwrite, true, priorities.overwrite, priority ); break;
    case no_overwrite:
      set_bool_option( bufop->overwrite, false, priorities.overwrite, priority ); break;
    case read_only:
      set_bool_option( bufop->read_only, true, priorities.read_only, priority ); break;
    case no_read_only:
      set_bool_option( bufop->read_only, false, priorities.read_only, priority ); break;
    case word_wrap:
      set_bool_option( bufop->word_wrap, true, priorities.word_wrap, priority ); break;
    case no_word_wrap:
      set_bool_option( bufop->word_wrap, false, priorities.word_wrap, priority ); break;
    case END_OPTS: errcode = 1; break;
    }

  if( !errcode )
    {
    if( arg_needed ) { if( !optarg || !*optarg ) errcode = 3; }
    else if( arg_included ) errcode = 4;
    }
  switch( errcode )
    {
    case 0: if( !arg_needed || arg_included ) return 1; else return 2;
    case 1: std::fprintf( stderr, "%sUnknown option %s\n", msg, optname );
            break;
    case 2: std::fprintf( stderr, "%sOption %s not allowed here\n", msg, optname );
            break;
    case 3: std::fprintf( stderr, "%sOption %s needs an argument\n", msg, optname );
            break;
    case 4: std::fprintf( stderr, "%sOption %s doesn't admit an argument\n", msg, optname );
            break;
    case 5: std::fprintf( stderr, "%sBad argument for option %s\n", msg, optname );
            break;
    case 6: std::fprintf( stderr, "%sValue out of range for option %s\n", msg, optname );
    }
  return 0;
  }

} // end namespace RC


RC::Options & RC::options() throw() { return _editor_options; }


const Buffer::Options & RC::default_buffer_options() throw()
  { return _default_buffer_options; }


void RC::apply_regex_options( Buffer & buffer ) throw()
  {
  for( unsigned int i = 0; i < regex_options_vector.size(); ++i )
    {
    const Regex_options & ro = regex_options_vector[i];
    unsigned int j = 0;
    for( ; j < ro.regex_vector.size(); ++j )
      if( Regex::match_filename( ro.regex_vector[j], File::my_basename( buffer.name() ) ) )
        break;
    if( j >= ro.regex_vector.size() ) continue;
    if( priorities.lmargin <= 2 )
      buffer.options.set_lmargin( ro.buffer_options.lmargin() );
    if( priorities.rmargin <= 2 )
      buffer.options.set_rmargin( ro.buffer_options.rmargin() );
    if( priorities.auto_indent <= 2 )
      buffer.options.auto_indent = ro.buffer_options.auto_indent;
    if( priorities.overwrite <= 2 )
      buffer.options.overwrite = ro.buffer_options.overwrite;
    if( priorities.read_only <= 2 )
      buffer.options.read_only |= ro.buffer_options.read_only;
    if( priorities.word_wrap <= 2 )
      buffer.options.word_wrap = ro.buffer_options.word_wrap;
    break;
    }
  }


bool RC::parse_int( const std::string & s, int & result ) throw()
  {
  if( s.size() == 0 ) return false;
  const char *str = s.c_str();
  char *tail;
  result = strtol( str, &tail, 0 );
  return ( tail != str );
  }


// Returns 0 for success, 1 for file not found, 2 for syntax error.
//
int RC::process_rcfile( const std::string & name ) throw()
  {
  FILE *f = std::fopen( name.c_str(), "r" );
  if( !f ) return 1;

  std::fprintf( stderr, "Processing '%s'... ", name.c_str() );
  std::fflush( stderr );

  const char delimiters[] = " \t\r\n";
  int line = 0, retval = 0;
  int regex_status = 0;		// 1 = got regex name, 2 got regex option
  char *buf = 0;
  size_t size = 0;
  while( true )
    {
    int len = getline( &buf, &size, f );
    if( len <= 0 ) break;
    ++line;
    if( len <= 1 || std::isspace( buf[0] ) ) continue;
    if( buf[0] != '-' )		// Create new file name dependent options
      {
      std::string regex = strtok( buf, delimiters );
      if( regex_status != 1 )
        regex_options_vector.push_back( Regex_options() );
      regex_options_vector.back().regex_vector.push_back( regex );
      regex_status = 1;
      }
    else			// Set an option
      {
      char *optname = strtok( buf, delimiters );
      char *optarg = strtok( 0, delimiters );
      char msg[80];
      std::snprintf( msg, sizeof(msg), "\n%s %d: ", name.c_str(), line );
      if( regex_status == 0 )
        { if( !set_option_from_name( optname, optarg, &_editor_options,
                                     &_default_buffer_options, 1, msg ) ) retval = 2; }
      else
        {
        if( !set_option_from_name( optname, optarg, 0,
                                   &regex_options_vector.back().buffer_options,
                                   0, msg ) ) retval = 2;
        regex_status = 2;
        }
      }
    }
  if( buf ) std::free( buf );
  std::fclose( f );
  if( !retval ) std::fprintf( stderr, "done\n" );
  return retval;
  }


// Returns 0 for success, 1 for invalid option, 2 for a no regular file.
//
int RC::process_options( int argc, char *argv[] ) throw()
  {
  int line = 0, optind;

  std::fprintf( stderr, "Processing options... " );
  std::fflush( stderr );

  for( optind = 1; optind < argc; ++optind )	// Process "global" options
    {
    if( argv[optind][0] == '+' )
      {
      if( parse_int( argv[optind] + 1, line ) && --line >= 0 ) continue;
      else { std::fprintf( stderr, "Bad line number\n" ); return 1; }
      }
    if( argv[optind][0] != '-' || !argv[optind][1] ) break;
    const int n = set_option_from_name( argv[optind], argv[optind+1],
                                        &_editor_options,
                                        &_default_buffer_options, 3 );
    if( n == 0 ) return 1; else if( n == 2 ) ++optind;
    }

  try
    {
    while( optind < argc )	// Process input files and file options
      {
      const std::string name = argv[optind++];
      if( !File::is_regular( name, true ) ) return 2;
      const int i = bufhandle_vector.add_handle( _default_buffer_options, &name, -1, line );
      line = 0;
      Buffer & buffer = bufhandle_vector[i].buffer();
      apply_regex_options( buffer );
      for( ; optind < argc; ++optind )		// Process file options
        {
        if( argv[optind][0] == '+' )
          {
          if( parse_int( argv[optind] + 1, line ) && --line >= 0 ) continue;
          else { std::fprintf( stderr, "Bad line number\n" ); return 1; }
          }
        if( argv[optind][0] != '-' || !argv[optind][1] ) break;
        const int n = set_option_from_name( argv[optind], argv[optind+1],
                                            0, &buffer.options );
        if( n == 0 ) return 1; else if( n == 2 ) ++optind;
        }
      }
    }
  catch( Buffer::Error e ) { std::fprintf( stderr, "%s\n", e.s ); return 1; }

  // Add empty buffer if no files given or stdin is not a tty
  try { bufhandle_vector.add_handle_if_pending_input_or_empty( line ); }
  catch( Buffer::Error e ) { std::fprintf( stderr, "%s\n", e.s ); return 1; }

  std::fprintf( stderr, "done\n" );
  return 0;
  }


void RC::reset() throw()
  {
  _editor_options.reset();
  _default_buffer_options.reset();
  priorities.reset();
  regex_options_vector.clear();
  }
