(*
 * Library for writing templates.
 * Copyright (C) 2003-2004 Merjis Ltd. (http://www.merjis.com/)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: template.ml,v 1.7 2005/01/05 19:26:43 ChriS Exp $
 *)

open Cgi_escape

(* Find content in str, starting at index i. *)
let string_find_from str i content =
  let first = content.[0] in
  let len = String.length str in
  let clen = String.length content in
  let rec loop i =
    let r = try String.index_from str i first with Not_found -> -1 in
    if r >= i && r <= len-clen then (
      let sub = String.sub str r clen in
      if sub = content then
	r				(* Found at index r. *)
      else
	loop (r+1)			(* Start looking again at r+1. *)
    ) else
      -1				(* Not found. *)
  in
  loop i

(* Find content in str, starting at beginning. *)
let string_find str content =
  string_find_from str 0 content

(* Return two strings by splitting str at character offset i. *)
let string_partition str i =
  String.sub str 0 i, String.sub str i ((String.length str) - i)

(* Return true if the string starts with the prefix. *)
let string_starts_with s pref =
  String.length s >= String.length pref
  && String.sub s 0 (String.length pref) = pref

(* Return true if the string ends with the suffix. *)
let string_ends_with s sfx =
  let ls = String.length s
  and lsfx = String.length sfx in
  ls >= lsfx && String.sub s (ls - lsfx) lsfx = sfx

let read_whole_chan chan =
  let buf = Buffer.create 4096 in
  let rec loop () =
    let line = input_line chan in
    Buffer.add_string buf line;
    Buffer.add_char buf '\n';
    loop ()
  in
  try  loop ()
  with End_of_file -> Buffer.contents buf


(*
 * Compile templates
 ***********************************************************************)

(* This is the type used to store the compiled template. *)
type node_t =
  | Plain of string 			(* Just some text. *)
  | Tag of string * escape_t		(* ::tag:: *)
  | If of string * compiled_t * compiled_t
					(* ::if(..):: .. ::else:: .. ::end:: *)
  | Table of string * compiled_t	(* ::table(..):: .. ::end:: *)
  | Call of string * string list * escape_t
					(* ::call(fname,arg,arg...):: *)
and compiled_t = node_t list

let tag_re = Pcre.regexp "^::(\\w+)::$"
let if_re = Pcre.regexp "^::if\\((.*)\\)::$"
let table_re = Pcre.regexp "^::table\\((.*)\\)::$"
let call_re = Pcre.regexp "^::call\\((.*)\\)::$"
let comma_split_re = Pcre.regexp ","

(* If name has the form "<name>_html" (or one of the similar cases),
 * return bare name and the form of the escaping that needs to be
 * applied.  *)
let get_escaping_from_name =
  let suppress s sfx = String.sub s 0 (String.length s - String.length sfx) in
  fun name ->
    if string_ends_with name "_url" then
      suppress name "_url", EscapeUrl
    else if string_ends_with name "_html" then
      suppress name "_html", EscapeHtml
    else if string_ends_with name "_html_tag" then
      suppress name "_html_tag", EscapeHtmlTag
    else if string_ends_with name "_html_textarea" then
      suppress name "_html_textarea", EscapeHtmlTextarea
    else name, EscapeNone


(* Return Some $1 if string matches regular expression. *)
let matches str rex =
  try
    let subs = Pcre.exec ~rex str in
    Some (Pcre.get_substring subs 1)
  with
    Not_found -> None

let compile_template ?filename source =
  (* Work out the base for file includes. *)
  let base =
    match filename with
      | None -> Filename.current_dir_name
      | Some filename -> Filename.dirname filename in

  (* This function splits up source into a list like this:
   * source = "some text ::tag:: more text ::end::" would become:
   * [ "some text "; "::tag::"; " more text "; "::end::" ]
   *)
  let split_up source =
    let rec loop source list =
      let i = string_find source "::" in
      match i with
	  -1 -> (* No "::" in the whole string. *)
	    source :: list
	| 0 ->  (* String starts with "::". *)
	    let i = string_find_from source 2 "::" in
	    if i > 2 then (
	      let tag, remainder = string_partition source (i+2) in
	      loop remainder (tag :: list)
	    ) else
	      source :: list
	| i -> (* String followed by possible tag start. *)
	    let str, remainder = string_partition source i in
	    loop remainder (str :: list)
    in
    List.rev (loop source [])
  in

  (* Return Some filename if the string has the form ::include(filename):: *)
  let is_include str =
    let len = String.length str in
    if string_starts_with str "::include(" &&
      str.[len-3] = ')' && str.[len-2] = ':' && str.[len-1] = ':' then
	Some (String.sub str 10 (len - 13))
    else
      None
  in

  (* Load included files. *)
  let rec load_includes = function
    | [] -> []
    | x :: xs ->
	(match is_include x with
	 | None -> [x]
	 | Some filename ->
	     let filename =
	       if Filename.is_relative filename then
		 Filename.concat base filename
	       else
		 filename in
	     let chan = open_in filename in
	     let source = read_whole_chan chan in
	     close_in chan;
	     load_includes (split_up source)) @ load_includes xs
  in

  (* Convert the flat list of strings into a list of (type, string)
   * tuples. So for example "::table(foo)::" would become ("table",
   * "foo").
   *)
  let typify =
    (* Return true if string is not a ::tag::. *)
    let is_plain str =
      let len = String.length str in
      len < 2 || str.[0] != ':' || str.[1] != ':'
    in

    List.map (fun str ->
		if is_plain str then
		  ("plain", str)
		else if str = "::else::" then
		  ("else", "")
		else if str = "::end::" then
		  ("end", "")
		else match matches str tag_re with
		    Some tag ->
		      ("tag", tag)
		  | None ->
		      match matches str if_re with
			  Some cond ->
			    ("if", cond)
			| None ->
			    match matches str table_re with
				Some name ->
				  ("table", name)
			      | None ->
				  match matches str call_re with
				      Some call ->
					("call", call)
				    | None ->
					failwith ("Template: " ^
						  "unknown tag in template: " ^
						  str))
  in

  (* Combine plain text parts of the list (these might have been split
   * naturally or across include files). This is for efficiency.
   *)
  let rec combine_plains = function
      [] -> []
    | [x] -> [x]
    | ("plain", x) :: ("plain", y) :: xs ->
	combine_plains (("plain", x ^ y) :: xs)
    | x :: xs ->
	x :: combine_plains xs
  in

  (* Split up the original source and load all included files to produce
   * a big flat list of (type, string) pairs.
   *)
  let flat = combine_plains (typify (load_includes (split_up source))) in

  (* This creates the final structure - a compiled tree (ct) - from the
   * flat list of (type, string) pairs.
   *)
  let rec build_tree = function
    | [] -> []
    | ("plain", text) :: xs ->
	Plain text :: build_tree xs
    | ("tag", tag) :: xs ->
	let name, esc = get_escaping_from_name tag in
	Tag (name, esc) :: build_tree xs
    | ("if", cond) :: xs ->
	let then_clause, else_clause, remainder = build_if_stmt xs in
	If (cond, then_clause, else_clause) :: build_tree remainder
    | ("table", name) :: xs ->
	let body, remainder = build_table_stmt xs in
	Table (name, body) :: build_tree remainder
    | ("call", call) :: xs ->
	let call, esc = get_escaping_from_name call in
	let fname, args = split_call_args call in
	Call (fname, args, esc) :: build_tree xs
    | (typ, _) :: xs ->
	failwith ("Template: unexpected tag ::" ^ typ ^ "::")
  and build_if_stmt xs =
    let rec then_part = function
	_, _, [] ->
	  failwith "Template: missing ::end:: in ::if:: statement"
      | 0, part, ("else", _) :: xs ->
	  List.rev part, xs, true
      | 0, part, ("end", _) :: xs ->
	  List.rev part, xs, false
      | lvl, part, ("if", cond) :: xs ->
	  then_part (lvl+1, (("if", cond) :: part), xs)
      | lvl, part, ("table", name) :: xs ->
	  then_part (lvl+1, (("table", name) :: part), xs)
      | lvl, part, ("end", _) :: xs ->
	  then_part (lvl-1, (("end", "") :: part), xs)
      | lvl, part, x :: xs ->
	  then_part (lvl, (x :: part), xs)
    in
    let rec else_part = function
	_, _, [] ->
	  failwith "Template: missing ::end:: in ::if:: statement"
      | 0, part, ("else", _) :: xs ->
	  failwith "Template: multiple ::else:: in ::if:: statement"
      | 0, part, ("end", _) :: xs ->
	  List.rev part, xs
      | lvl, part, ("if", cond) :: xs ->
	  else_part (lvl+1, (("if", cond) :: part), xs)
      | lvl, part, ("table", name) :: xs ->
	  else_part (lvl+1, (("table", name) :: part), xs)
      | lvl, part, ("end", _) :: xs ->
	  else_part (lvl-1, (("end", "") :: part), xs)
      | lvl, part, x :: xs ->
	  else_part (lvl, (x :: part), xs)
    in
    let then_part, remainder, has_else_clause = then_part (0, [], xs) in
    let then_clause = build_tree then_part in
    let else_part, remainder =
      if has_else_clause then else_part (0, [], remainder)
      else [], remainder in
    let else_clause = build_tree else_part in
    then_clause, else_clause, remainder
  and build_table_stmt xs =
    let rec body_part = function
	_, _, [] ->
	  failwith "Template: missing ::end:: in ::table:: statement"
      | 0, part, ("end", _) :: xs ->
	  List.rev part, xs
      | lvl, part, ("if", cond) :: xs ->
	  body_part (lvl+1, (("if", cond) :: part), xs)
      | lvl, part, ("table", name) :: xs ->
	  body_part (lvl+1, (("table", name) :: part), xs)
      | lvl, part, ("end", _) :: xs ->
	  body_part (lvl-1, (("end", "") :: part), xs)
      | lvl, part, x :: xs ->
	  body_part (lvl, (x :: part), xs)
    in
    let body_part, remainder = body_part (0, [], xs) in
    let body_clause = build_tree body_part in
    body_clause, remainder
  and split_call_args call =
    (* Split string on commas. *)
    let args = Pcre.split ~rex:comma_split_re call in
    List.hd args, List.tl args
  in

  (* Build the tree from the flat list. *)
  build_tree flat

(* Type of variables, either a simple ::tag:: or a set of row
 * definitions in a table.  *)
type var_t =
  | VarString of string
  | VarTable of table_row_t list
  | VarConditional of bool
  | VarCallback of (string list -> string)
and table_row_t = (string * var_t) list

module M = Map.Make(String)

let find k m =
  try  M.find k m
  with Not_found ->
    failwith ("Template: tag/table ::" ^ k ^ ":: was not assigned any value.")


class template ?filename source =
  (* This will store the values of set variables/tables.  *)
  let bindings = ref M.empty in
  (* Compiled template source. *)
  let ct = compile_template ?filename source in
object (self)
  method set name value =
    bindings := M.add name (VarString value) !bindings

  method table name tbl =
    bindings := M.add name (VarTable tbl) !bindings

  method conditional name cond =
    bindings := M.add name (VarConditional cond) !bindings

  method callback name f =
    bindings := M.add name (VarCallback f) !bindings

  method to_string =
    let buffer = Buffer.create 4096 in
    self#output (Buffer.add_string buffer);
    Buffer.contents buffer

  method to_channel chan =
    self#output (output_string chan)

  method source =
    source

  method output (out:string -> unit) =
    let rec substitute bindings ct =
      List.iter (function
		 | Plain text -> out text
		 | Tag(name, esc) ->
		     out(escape esc (resolve_variable bindings name))
		 | If(cond, then_clause, else_clause) ->
		     substitute bindings (if eval_condition bindings cond
					  then then_clause else else_clause)
		 | Table (name, body) ->
		     (* For all table rows, add the corresponding
			bindings and process the table body. *)
		     let process_row row =
		       let newb = List.fold_left
			 (fun b (n, v) -> M.add n v b) bindings row in
		       substitute newb body in
		     List.iter process_row (resolve_table bindings name);
		 | Call (fname, args, esc) ->
		     out(escape esc (resolve_callback bindings fname args))
		) ct
    and resolve_variable b name =
      match find name b with
      | VarString str -> str
      | _ -> failwith ("Template: ::" ^ name
		       ^ ":: should be a simple string tag.")
    and eval_condition b name =
      match find name b with
      | VarConditional b -> b
      | _ -> failwith ("Template: ::if(" ^ name
		       ^ "):: should be a conditional tag.")
    and resolve_table b name =
      match find name b with
      | VarTable tbl -> tbl
      | _ -> failwith ("Template: ::table(" ^ name
		       ^ "):: should be a table tag.")
    and resolve_callback b name =
      match find name b with
      | VarCallback f -> f
      | _ -> failwith ("Template: ::call(" ^ name
		     ^ "[,...]):: should be a callback function.")
    in
    substitute !bindings ct

end

let template_from_string ?filename source =
  new template ?filename source

let template_from_channel ?filename chan =
  template_from_string ?filename (read_whole_chan chan)

let template filename =
  let chan = open_in filename in
  let template = template_from_channel ~filename chan in
  close_in chan;
  template



let may f v = match v with None -> () | Some v -> f v

module StdPages =
struct
  type button = {
    label : string;
    link : string;
    method_ : string option;
    params : (string * string) list;
  }

  let default_template = template_from_string
    "<html>
<head>
::if(has_title)::<title>::title_html::</title>::end::
::if(has_css)::
<link rel=\"stylesheet\" href=\"::css_url::\" type=\"text/css\">
::else::
<style TYPE=\"text/css\" MEDIA=screen>
<!--
stdpages.body {
  background: #ffffff;
  color: black;
}
stdpages.img {
  float: left;
  clear:both;
  margin:10px;
}
-->
</style>
::end::
</head>
<body>
::if(has_title)::<h1>::title_html::</h1>::end::
::if(has_icon)::
<img class=\"stdpages\"
     alt=\"::icon_alt_html_tag::\" src=\"::icon_html_tag::\">
::end::
<p class=\"stdpages\">::message_html::</p>
<hr>
<table class=\"stdpages\"><tr class=\"stdpages\">
::if(has_back_button)::
<td class=\"stdpages\"><form class=\"stdpages\">
  <input type=\"button\" value=\"&lt;&lt; Go Back\"
         onclick=\"history.go (-1)\"></form></td>
::end::
::if(has_close_button)::
<td class=\"stdpages\"><form class=\"stdpages\">
  <input type=\"button\" value=\"Close Window\"
         onclick=\"top.close ()\"></form></td>
::end::
::table(buttons)::
<td class=\"stdpages\"><form class=\"stdpages\"
  method=\"::method_html_tag::\" action=\"::action_html_tag::\">
::table(params)::
<input type=\"hidden\" name=\"::name_html_tag::\" value=\"::value_html_tag::\">
::end::
<input type=\"submit\" value=\"::name_html_tag::\"></form></td>
::end::
</body>
</html>"

  let dialog ?cookie ?cookies ?css_url
      ?(template = default_template) ?title ?icon ?icon_alt
      ?(back_button = true) ?(close_button = false) ?buttons
      (q : Cgi.cgi) message =
    template#conditional "has_title" (title <> None);
    may (fun title -> template#set "title" title) title;
    template#conditional "has_css" (css_url <> None);
    may (fun css -> template#set "css" css) css_url;
    template#conditional "has_icon" (icon <> None);
    may (fun icon -> template#set "icon" icon) icon;
    template#set "icon_alt" " ";
    may (fun icon_alt -> template#set "icon_alt" icon_alt) icon_alt;
    template#set "message" message;
    template#conditional "has_back_button" back_button;
    template#conditional "has_close_button" close_button;
    template#table "buttons" [];
    may
      (fun buttons ->
	 let buttons =
	   List.map
	     (fun button ->
		let params =
		  List.map
		    (fun (name, value) ->
		       [ "name", VarString name;
			 "value", VarString value;
		       ]) button.params in
		[ "method", VarString (match button.method_ with
						    None -> "GET"
						  | Some m -> m);
		  "action", VarString button.link;
		  "params", VarTable params;
		  "name", VarString button.label;
		]
	     ) buttons in
	 template#table "buttons" buttons
      ) buttons;
    q#template ?cookie ?cookies template

  let error ?cookie ?cookies ?css_url ?template
      ?(title = "There was an error")
      ?(icon = "/caml-icons/error.png")
      ?(icon_alt = "Error")
      ?back_button ?close_button q message =
    dialog ?cookie ?cookies ?css_url ?template ~title ~icon
      ?back_button ?close_button  q message

  let ok ?cookie ?cookies ?css_url ?template
      ?(title = "That operation was carried out successfully")
      ?(icon = "/caml-icons/ok.png")
      ?(icon_alt = "OK")
      ?(back_button = false)
      ?close_button ?buttons q message =
    dialog ?cookie ?cookies ?css_url ?template ~title ~icon
      ~back_button ?close_button ?buttons
      q message

end
