(* mod_caml module for executing CGI scripts written in OCaml.
 * Copyright (C) 2003 Merjis Ltd.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: registry.ml,v 1.10 2005/01/05 16:13:28 ChriS Exp $
 *
 * CGI scripts are loaded dynamically into memory, and stay there unless
 * you restart the server or the file changes on disk.
 *
 * To use:
 *
 * CamlLoad /usr/share/mod_caml/registry.cmo
 * Alias /caml-bin/ /path/to/your/scripts/
 * <Location /caml-bin>
 *   SetHandler ocaml-bytecode
 *   CamlHandler Registry.handler
 *   Options ExecCGI
 *   Allow from all
 * </Location>
 *
 * This is called the "registry" in honour of mod_perl's Apache::Registry
 * which does much the same thing for Perl programs.
 *)

open Unix
open Apache

(* Table of scripts we have already loaded.  Maps filename ->
   (handler, mtime) so we can reload the file if the mtime has
   changed.  *)
let registry = Hashtbl.create 32

let reg_hand = ref None

(* Callback from the loaded script; tells which function to run. *)
let register_script handler =
  if !reg_hand <> None then
    failwith ("Registry.register_script should only be called once during script initialization");
  reg_hand := Some handler

(* Load a file and return the function to run. *)
let load_file filename =
  reg_hand := None;
  (try  Dynlink.loadfile_private filename;
   with Dynlink.Error e -> failwith (Dynlink.error_message e));
  (* If the file was loaded and thus the toplevel code executed,
     [reg_hand] is now [Some handler]. *)
  match !reg_hand with
  | None -> failwith (filename ^ ": forgot to call Registry.register_script!")
  | Some handler -> handler

(* Handler for requests. *)
let handler r =
  let filename = Request.filename r in
  let mtime =
    match Request.finfo r with
    | None -> prerr_endline
	(filename
	 ^ ": no request->finfo: probably means the file does not exist.");
	raise (HttpError cHTTP_NOT_FOUND)
    | Some finfo -> finfo.st_mtime  in

  (* Get the handler for this file. *)
  let handler =
    try
      let (old_handler, old_mtime) = Hashtbl.find registry filename in
      if old_mtime < mtime then (
        (* Reload the file. *)
	let handler = load_file filename in
	Hashtbl.replace registry filename (handler, mtime);
	handler
      ) else
	old_handler
    with Not_found ->
      (* Load the file for the first time. *)
      let handler = load_file filename in
      Hashtbl.add registry filename (handler, mtime);
      handler in

  (* Run the handler.  Exceptions are passed upwards and handled
     correctly in mod_caml_c.c:exception_in_handler, except for [Exit]
     which we catch and ignore.  *)
  (try handler r
   with Exit -> ());
  OK (* Return OK (actually the return value is ignored, I think). *)

let return () = raise Exit

let () =
  Mod_caml.register_handler handler "handler"
