(* Breadcrumb trails for SSI pages.
 * Copyright (C) 2003 Merjis Ltd.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: crumbs.ml,v 1.7 2004/02/26 12:10:51 rwmj Exp $
 *
 * To get this working you need to locate registry.cmo and create a caml-bin
 * directory containing crumbs.cmo. Then set up Apache like this:
 *
 * CamlLoad /the/path/to/registry.cmo
 * Alias /caml-bin/ /path/to/your/caml-bin/
 * <Location /caml-bin>
 *   SetHandler ocaml-bytecode
 *   CamlHandler Registry.handler
 *   Options ExecCGI
 *   Allow from all
 * </Location>
 *
 * Now you will need to use crumbs.cmo in your SSI, like this (suggest you
 * put it somewhere in the common header of your pages):
 *
 * <!--#include virtual="/caml-bin/crumbs.cmo" -->
 *
 * By default the crumbtrail is generated from the URL using a trivial
 * algorithm. In future we may add a way to customise the names of
 * elements, probably by adding ".crumb" files in each directory.
 * (This is quite complex because you need to issue subrequests to
 * fetch those files).
 *
 * You may also name the current (leaf) node explicitly by passing in
 * a node name, eg:
 *
 * <!--#include virtual="/caml-bin/crumbs.cmo?node=name" -->
 *
 * Don't forget to use proper URL-escaping on the name.
 *)

open Apache
open Registry
open Cgi

let template = "::table(elements)::" ^
	       "<a href=\"::url_html_tag::\">::name_html::</a>" ^
	       "&nbsp;<b>&gt;</b>&nbsp;" ^
	       "::end::&nbsp;" ^
	       "::last_html::"

(* The template is compiled once when the script is loaded. *)
let template = Template.template_from_string template;;

let split_re = Str.regexp "/+"
let index_re = Str.regexp "^index\\."

let run r =
  let q = new cgi r in

  (* If we were correctly embedded into a SSI page using something like
   * <!--#include virtual="/caml-bin/crumbs.cmo" --> then mod_include
   * is running this script as a subrequest. We navigate through r->main
   * to get back to the main request, and thence to the original request URI
   * (so we know the original page, of course). We may have to go "up"
   * through several levels of subrequest.
   *)
  let main_r =
    let rec get_main_req r =
      try
	let r = Request.main r in
	get_main_req r
      with
	  Not_found -> r
    in
    get_main_req r
  in
  let uri = Request.uri main_r in

  (* Split up the URI into sections.
   * + "/" -> []
   *   "/index.shtml" -> ["index.shtml"]
   * + "/caml-bin/" -> ["caml-bin"]
   *   "/caml-bin/index.shtml" -> ["caml-bin"; "index.shtml"]
   *   "/caml-bin//index.shtml" -> ["caml-bin"; "index.shtml"]
   * + = can never happen in practice
   *)
  let elements = Str.split split_re uri in

  (* If the last element in the list is the index file, then we discard
   * it. We use a heuristic for this: 'index.*' is an index file.
   *)
  let rec loop = function
      [] -> []
    | [x] -> if Str.string_match index_re x 0 then [] else [x]
    | x :: xs -> x :: loop xs
  in
  let elements = loop elements in

  (* Prepend 'Home' to the list. *)
  let elements = "Home" :: elements in

  (* The last element in the list is split out into a separate variable. *)
  let elements, last =
    let relements = List.rev elements in
    List.rev (List.tl relements), List.hd relements in

  (* Map each element to the printable name. *)
  let rec loop prefix = function
      [] -> []
    | x :: xs ->
	let prefix = if prefix = "" then "/" else prefix ^ x ^ "/" in
	let url = Template.VarString prefix in
	let name = Template.VarString (String.capitalize x) in
	["url", url; "name", name] :: loop prefix xs
  in
  let elements = loop "" elements in

  (* Map the final element to the printable name. *)
  let last =
    try
      let node = q#param "node" in
      if node = "" then raise Not_found;
      node
    with
	Not_found ->
	  try
	    let i = String.index last '.' in
	    String.capitalize (String.sub last 0 i)
	  with
	      Not_found ->
		String.capitalize last in

  (* Substitute the template variables. *)
  template#table "elements" elements;
  template#set "last" last;

  (* Display the page. *)
  q#template template

let () =
  register_script run
