(* Example authentication handler. This just authenticates against a built-in
 * table of users, but you could easily extend this to support files, database
 * access or whatever.
 * Copyright (C) 2003 Merjis Ltd.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: auth_simple.ml,v 1.2 2003/09/27 15:52:31 rwmj Exp $
 *
 * To use this module, add this to your Apache configuration:
 *
 * CamlLoad /path/to/auth_simple.cmo
 * <Location /private>                     # The protected content
 *   CamlCheckUserIDHandler Auth_simple.check_user_id
 *   CamlAuthCheckerHandler Auth_simple.auth_checker
 *   AuthType Basic
 *   AuthName Members-only
 *   Require valid-user
 * </Location>
 *
 * (Note that the "AuthType", "AuthName" and "Require" directives are all
 * essential, otherwise Apache may crash!)
 *)

open Apache
open Mod_caml

(* Our fixed table of user/password pairs. *)
let users = [ "rich", "furby";
	      "guest", "guest" ]

(* If false then allow access control to be passed along to
 * other modules if the user ID is not known to this module.
 *)
let authoritative = true

(* This handler just checks the user's name and password. *)
let check_user_id r =
  (* Get the password sent. You must call get_basic_auth_pw before trying
   * to read the username!
   *)
  match Request.get_basic_auth_pw r with
    None ->
      prerr_endline "Auth_simple: no username/password supplied";
      Request.note_basic_auth_failure r;
      raise (HttpError cAUTH_REQUIRED)	(* No username/password supplied. *)
  | Some sent_password ->
      (* Get the username sent. *)
      let user = Request.user r in

      (* Check against our table. *)
      let valid = List.mem (user, sent_password) users in

      if not valid then
	if not authoritative then
	  DECLINED			(* Pass to a lower module. *)
	else (
	  (* Authentication failed. *)
	  prerr_endline ("Auth_simple: user authentication failed for " ^
			 user);
	  Request.note_basic_auth_failure r;
	  raise (HttpError cAUTH_REQUIRED)
	)
      else (
	(* Authentication succeeded. *)
	prerr_endline ("Auth_simple: successful authentication for " ^
		       user);
	OK
      )

(* This handler verifies if the user is permitted to enter this area. This
 * simple module assumes that if you supplied a valid username/password
 * combination then you can access all the protected content. If you are
 * anonymous, or didn't supply a valid username/password combination, then
 * the code won't get this far, because the handler above will have thrown
 * an 'AUTH_REQUIRED' error back at you.
 *)
let auth_checker r =
  OK

(* Register our handlers. *)
let () =
  register_handler check_user_id "check_user_id";
  register_handler auth_checker "auth_checker"
