// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \file hash.cc C++ API <i>mobius.forensics.turing.hash</i> class wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "hash.h"
#include "module.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create <i>hash</i> Python object from C++ object
//! \param obj C++ object
//! \return new hash object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject *
PyMobius_Forensics_Turing_Hash_from_cpp (mobius::forensics::turing::hash obj)
{
  PyObject *ret = _PyObject_New (&forensics_turing_hash_t);

  if (ret)
    ((forensics_turing_hash_o *) ret)->obj = new mobius::forensics::turing::hash (obj);

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>type</i> attribute getter
//! \param self object
//! \return <i>type</i> attribute
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_type (forensics_turing_hash_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_FromString (self->obj->get_type ().c_str ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>value</i> attribute getter
//! \param self object
//! \return <i>value</i> attribute
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_value (forensics_turing_hash_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_FromString (self->obj->get_value ().c_str ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>password</i> attribute getter
//! \param self object
//! \return <i>password</i> attribute
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_password (forensics_turing_hash_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_FromString (self->obj->get_password ().c_str ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>password</i> attribute setter
//! \param self object
//! \param value new value
//! \return 0 on success, -1 on error
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static int
tp_setter_password (forensics_turing_hash_o *self, PyObject *value, void *)
{
  // if it is a delete operation, then returns with error
  if (value == nullptr)
    {
      PyErr_SetString (PyExc_TypeError, "cannot delete 'password' attribute");
      return -1;
    }

  // check argument type
  if (!PyString_Check (value))
    {
      PyErr_SetString (PyExc_TypeError, "invalid type for 'password' attribute");
      return -1;
    }

  // set password attribute
  try
    {
      self->obj->set_password (PyString_AsString (value));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return -1;
    }

  // return success
  return 0;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>password_status</i> attribute getter
//! \param self object
//! \return <i>password_status</i> attribute
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_getter_password_status (forensics_turing_hash_o *self)
{
  PyObject *ret = nullptr;

  try
    {
      ret = PyInt_FromLong (static_cast <int> (self->obj->get_password_status ()));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef tp_getset[] =
{
  {
    (char *) "type",
    (getter) tp_getter_type,
    (setter) 0,
    (char *) "hash type",
    NULL
  },
  {
    (char *) "value",
    (getter) tp_getter_value,
    (setter) 0,
    (char *) "hash value",
    NULL
  },
  {
    (char *) "password",
    (getter) tp_getter_password,
    (setter) tp_setter_password,
    (char *) "password",
    NULL
  },
  {
    (char *) "password_status",
    (getter) tp_getter_password_status,
    (setter) 0,
    (char *) "password status",
    NULL
  },
  {NULL, NULL, NULL, NULL, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>has_argument</i> method implementation
//! \param self object
//! \param args argument list
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_has_argument (forensics_turing_hash_o *self, PyObject *args)
{
  // parse input args
  const char * arg_name;

  if (!PyArg_ParseTuple (args, "s", &arg_name))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyBool_FromLong (self->obj->has_argument (arg_name));
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get_argument</i> method implementation
//! \param self object
//! \param args argument list
//! \return argument value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_argument (forensics_turing_hash_o *self, PyObject *args)
{
  // parse input args
  const char * arg_name;

  if (!PyArg_ParseTuple (args, "s", &arg_name))
    return nullptr;

  // execute C++ function
  PyObject *ret = nullptr;

  try
    {
      ret = PyString_FromString (self->obj->get_argument (arg_name).c_str ());
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  // return value
  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>set_argument</i> method implementation
//! \param self object
//! \param args argument list
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_set_argument (forensics_turing_hash_o *self, PyObject *args)
{
  // parse input args
  const char * arg_name;
  const char * arg_value;

  if (!PyArg_ParseTuple (args, "ss", &arg_name, &arg_value))
    return nullptr;

  // execute C++ function
  try
    {
      self->obj->set_argument (arg_name, arg_value);
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>remove_argument</i> method implementation
//! \param self object
//! \param args argument list
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_remove_argument (forensics_turing_hash_o *self, PyObject *args)
{
  // parse input args
  const char * arg_name;

  if (!PyArg_ParseTuple (args, "s", &arg_name))
    return nullptr;

  // execute C++ function
  try
    {
      self->obj->remove_argument (arg_name);
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      return nullptr;
    }

  // return None
  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>get_arguments</i> method implementation
//! \param self object
//! \param args argument list
//! \return arguments
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_f_get_arguments (forensics_turing_hash_o *self, PyObject *)
{
  PyObject *ret = PyDict_New ();

  if (!ret)
    return nullptr;

  try
    {
      for (const auto& p : self->obj->get_arguments ())
        {
          PyObject *py_key = PyString_FromString (p.first.c_str ());

          if (!py_key)
            {
              Py_CLEAR (ret);
              return nullptr;
            }

          PyObject *py_value = PyString_FromString (p.second.c_str ());

          if (!py_value)
            {
              Py_DECREF (py_key);
              Py_CLEAR (ret);
              return nullptr;
            }

          PyDict_SetItem (ret, py_key, py_value);
          Py_DECREF (py_key);
          Py_DECREF (py_value);
        }
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
      Py_CLEAR (ret);
      ret = nullptr;
    }

  return ret;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef tp_methods[] =
{
  {
    (char *) "has_argument",
    (PyCFunction) tp_f_has_argument,
    METH_VARARGS,
    "check if hash has a given argument"
  },
  {
    (char *) "get_argument",
    (PyCFunction) tp_f_get_argument,
    METH_VARARGS,
    "get argument"
  },
  {
    (char *) "set_argument",
    (PyCFunction) tp_f_set_argument,
    METH_VARARGS,
    "set argument"
  },
  {
    (char *) "remove_argument",
    (PyCFunction) tp_f_remove_argument,
    METH_VARARGS,
    "remove argument"
  },
  {
    (char *) "get_arguments",
    (PyCFunction) tp_f_get_arguments,
    METH_VARARGS,
    "get arguments"
  },
  {NULL, NULL, 0, NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>hash</i> constructor
//! \param type type object
//! \param args argument list
//! \param kwds keywords dict
//! \return new <i>hash</i> object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
tp_new (PyTypeObject *type, PyObject *, PyObject *)
{
  forensics_turing_hash_o *self = (forensics_turing_hash_o *) type->tp_alloc (type, 0);

  if (self)
    self->obj = new mobius::forensics::turing::hash ();

  return (PyObject *) self;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief <i>hash</i> deallocator
//! \param self object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
tp_dealloc (forensics_turing_hash_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject forensics_turing_hash_t =
{
  PyVarObject_HEAD_INIT (NULL, 0)
  "mobius.forensics.turing.hash",          		// tp_name
  sizeof (forensics_turing_hash_o),        		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) tp_dealloc,                 		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "hash class",                            		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  tp_methods,                              		// tp_methods
  0,                                       		// tp_members
  tp_getset,                               		// tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  tp_new,                                  		// tp_new
  0,                                       		// tp_free
  0,                                       		// tp_is_gc
  0,                                       		// tp_bases
  0,                                       		// tp_mro
  0,                                       		// tp_cache
  0,                                       		// tp_subclasses
  0,                                       		// tp_weaklist
  0,                                       		// tp_del
  0,                                       		// tp_version_tag
};
