// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "partition_system_impl_base.h"
#include <algorithm>

namespace mobius::vfs
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Set entries starting_address, ending_address and size
//! \param sector_size Sector size
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
partition_system_impl_base::_set_addresses (
  std::vector <partition_system_entry>& entries,
  sector_type sector_size
) const
{
  for (auto& e : entries)
    {
      e.set_starting_address (e.get_starting_sector () * sector_size);
      e.set_ending_address ((e.get_ending_sector () + 1) * sector_size - 1);
      e.set_size (e.get_sectors () * sector_size);
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Add freespaces between non-contiguous entries
//! \param entries Entries
//! \param sectors Total number of sectors
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
partition_system_impl_base::_add_freespaces (
  std::vector <partition_system_entry>& entries,
  sector_type sectors
) const
{
  // sort entries
  auto sorted_entries = entries;
  std::sort (sorted_entries.begin (), sorted_entries.end ());

  // fill in freespace where it is necessary
  sector_type pos = 0;
  entries.clear ();

  for (const auto& e : sorted_entries)
    {
      if (pos < e.get_starting_sector ())
        {
          partition_system_entry f_entry;
          f_entry.set_starting_sector (pos);
          f_entry.set_ending_sector (e.get_starting_sector () - 1);
          f_entry.set_sectors (f_entry.get_ending_sector () - f_entry.get_starting_sector () + 1);
          f_entry.set_type ("freespace");
          f_entry.set_description ("Freespace");
          entries.push_back (f_entry);
        }

      entries.push_back (e);
      pos = e.get_ending_sector () + 1;
    }

  auto ending_sector = sectors - 1;

  if (pos < ending_sector)
    {
      partition_system_entry f_entry;
      f_entry.set_starting_sector (pos);
      f_entry.set_ending_sector (ending_sector);
      f_entry.set_sectors (ending_sector - pos + 1);
      f_entry.set_type ("freespace");
      f_entry.set_description ("Freespace");
      entries.push_back (f_entry);
    }
}

} // namespace mobius::vfs
