// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief  C++ API module wrapper
//! \author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <pymobius.h>
#include <pygil.h>
#include "disk/disk.h"
#include <mobius/partition/partition_system.h>

namespace
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Create tuple from partiton_system::entry object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
PyTuple_from_partition_system_entry (const mobius::partition::partition_system::entry& e)
{
  PyObject *ret = PyTuple_New (6);
  
  if (ret)
    {
      PyTuple_SetItem (ret, 0, PyLong_FromLong (e.starting_sector));
      PyTuple_SetItem (ret, 1, PyLong_FromLong (e.ending_sector));
      PyTuple_SetItem (ret, 2, PyLong_FromLong (e.sectors));
      PyTuple_SetItem (ret, 3, PyString2_from_std_string (e.type));
      PyTuple_SetItem (ret, 4, PyString2_from_std_string (e.flags));
      PyTuple_SetItem (ret, 5, PyString2_from_std_string (e.description));
    }

  return ret;
}

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get_partition_table function
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyObject*
func_get_partition_table (PyObject *, PyObject *args)
{
  // parse input args
  mobius::disk::disk arg_disk;

  try
    {
      arg_disk = mobius::py::get_arg_as_cpp (args, 0, PyMobius_Disk_Disk_as_cpp);
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_TypeError, e.what ());
      return nullptr;
    }

  // execute C++ code
  PyObject *ret = nullptr;

  try
    {
      mobius::partition::partition_system psystem (arg_disk);
      auto entries = mobius::py::GIL () (psystem.get_entries ());

      ret = mobius::py::pylist_from_cpp_container (
                 entries,
                 PyTuple_from_partition_system_entry
            );
    }
  catch (const std::exception& e)
    {
      PyErr_SetString (PyExc_Exception, e.what ());
    }

  return ret;
}
