// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "filesystem_impl_vfat.h"
#include <mobius/decoder/data_decoder.h>
#include <mobius/string_functions.h>

namespace mobius
{
namespace vfs
{
namespace filesystem
{
namespace
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Check if reader contains a VFAT boot sector
//! \param reader Reader object
//! \param offset Offset from the beginning of the stream
//! \return True/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static bool
check_vfat_boot_sector (mobius::io::reader reader, std::uint64_t offset)
{
  reader.seek (offset);
  auto data = reader.read (512);

  constexpr int VFAT_SIGNATURE_OFFSET = 0x1fe;

  return data[VFAT_SIGNATURE_OFFSET] == 0x55 &&
         data[VFAT_SIGNATURE_OFFSET + 1] == 0xaa &&
         (data.slice (54, 58) == "FAT12" ||
          data.slice (54, 58) == "FAT16" ||
          data.slice (82, 86) == "FAT32" ||
          data.slice (11, 18) == "\x00\x02\x01\x01\x00\x02\xe0\x00");
}

} // namespace

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Check if stream contains an instance of vfat filesystem
//! \param reader Reader object
//! \param offset Offset from the beginning of the stream
//! \return True/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
filesystem_impl_vfat::is_instance (mobius::io::reader reader, std::uint64_t offset)
{
  // check sector 0
  bool rc = check_vfat_boot_sector (reader, offset);

  // check sector 6 (FAT-32 usually has a backup boot sector there)
  if (!rc)
    rc = check_vfat_boot_sector (reader, offset + 6 * 512);

  // return result
  return rc;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Constructor
//! \param reader Reader object
//! \param offset Offset from the beginning of volume
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
filesystem_impl_vfat::filesystem_impl_vfat (
  const mobius::io::reader& reader,
  size_type offset
)
 : reader_ (reader),
   offset_ (offset),
   tsk_adaptor_ (reader, offset)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get metadata item
//! \param name Item name
//! \return Data object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::pod::data
filesystem_impl_vfat::get_metadata (const std::string& name) const
{
  _load_data ();
  return metadata_.get (name);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Get root folder
//! \return Root folder
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::io::folder
filesystem_impl_vfat::get_root_folder () const
{
  return tsk_adaptor_.get_root_folder ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief Load data on demand
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
filesystem_impl_vfat::_load_data () const
{
  if (data_loaded_)
    return;

  // get boot sector offset
  mobius::io::reader reader = reader_;
  std::uint64_t boot_offset = 0;

  if (check_vfat_boot_sector (reader, offset_))
    boot_offset = offset_;

  else if (check_vfat_boot_sector (reader, offset_ + 6 * 512))
    boot_offset = offset_ + 6 * 512;

  else          // force data decoding anyway
    boot_offset = offset_;

  // get VFAT type
  reader.seek (boot_offset);
  auto data = reader.read (512);
  std::uint32_t fat_entry_size = 0;

  if (data.compare (54, "FAT12"))
    fat_entry_size = 12;

  else if (data.compare (54, "FAT16"))
    fat_entry_size = 16;

  else if (data.compare (82, "FAT32"))
    fat_entry_size = 32;

  // decode data
  reader.seek (boot_offset);
  mobius::decoder::data_decoder decoder (reader);

  // FDC descriptor (ECMA 9.1)
  decoder.skip (3);
  auto oem_name = mobius::string::rstrip (decoder.get_string_by_size (8));
  auto sector_size = decoder.get_uint16_le ();
  auto sectors_per_cluster = decoder.get_uint8 ();
  auto reserved_sectors = decoder.get_uint16_le ();
  auto fats = decoder.get_uint8 ();
  auto root_dir_entries = decoder.get_uint16_le ();
  auto sectors = decoder.get_uint16_le ();
  auto media_descriptor = decoder.get_uint8 ();
  std::uint32_t sectors_per_fat = decoder.get_uint16_le ();
  auto sectors_per_track = decoder.get_uint16_le ();
  auto heads = decoder.get_uint16_le ();

  // common metadata
  auto hidden_sectors = decoder.get_uint32_le ();

  auto sectors_32bit = decoder.get_uint32_le ();
  if (sectors == 0)
    sectors = sectors_32bit;

  // FAT-32 specific metadata
  std::uint16_t flags = 0;
  std::uint16_t version = 0;
  std::uint32_t root_dir_cluster = 0;
  std::uint16_t fs_info_sector = 0;
  std::uint16_t backup_boot_sector = 0;

  if (fat_entry_size == 32)
    {
      sectors_per_fat = decoder.get_uint32_le ();
      flags = decoder.get_uint16_le ();
      version = decoder.get_uint16_le ();
      root_dir_cluster = decoder.get_uint32_le ();
      fs_info_sector = decoder.get_uint16_le ();
      backup_boot_sector = decoder.get_uint16_le ();
      decoder.skip (12);
    }

  // extended signature
  std::string volume_id;
  std::string volume_label;

  auto logical_drive_number = decoder.get_uint8 ();
  decoder.skip (1);             // reserved

  auto extended_signature = decoder.get_uint8 ();
  if (extended_signature)
    {
      volume_id = "0x" + mobius::string::to_hex (decoder.get_uint32_le (), 8);
      volume_label = mobius::string::rstrip (decoder.get_string_by_size (11));
      type_ = decoder.get_string_by_size (8);
    }
  else
    type_ = "fat" + std::to_string (fat_entry_size);

  // set metadata
  metadata_.set ("oem_name", oem_name);
  metadata_.set ("volume_id", volume_id);
  metadata_.set ("volume_label", volume_label);
  metadata_.set ("sector_size", sector_size);
  metadata_.set ("sectors_per_cluster", sectors_per_cluster);
  metadata_.set ("reserved_sectors", reserved_sectors);
  metadata_.set ("fats", fats);
  metadata_.set ("root_dir_entries", root_dir_entries);
  metadata_.set ("sectors", sectors);
  metadata_.set ("media_descriptor", media_descriptor);
  metadata_.set ("sectors_per_fat", sectors_per_fat);
  metadata_.set ("sectors_per_track", sectors_per_track);
  metadata_.set ("heads", heads);
  metadata_.set ("hidden_sectors", hidden_sectors);
  metadata_.set ("logical_drive_number", logical_drive_number);
  metadata_.set ("extended_signature", extended_signature);
  
  if (fat_entry_size == 32)
    {
      metadata_.set ("flags", flags);
      metadata_.set ("version", version);
      metadata_.set ("root_dir_cluster", root_dir_cluster);
      metadata_.set ("fs_info_sector", fs_info_sector);
      metadata_.set ("backup_boot_sector", backup_boot_sector);
    }

  // derived information
  size_ = sectors * sector_size;

  name_ = "VFAT-" + std::to_string (fat_entry_size);

  if (!volume_label.empty ())
    name_ += " (" + volume_label + ")";

  else if (!oem_name.empty ())
    name_ += " (" + oem_name + ")";

  // Set data loaded
  data_loaded_ = true;
}

} // namespace filesystem
} // namespace vfs
} // namespace mobius
