// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "entry.h"
#include "stream.h"
#include "entry_impl_null.h"

namespace mobius
{
namespace filesystem
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief default constructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
entry::entry ()
  : impl_ (std::make_shared <entry_impl_null> ())
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor from implementation pointer
//! \param impl implementation pointer
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
entry::entry (std::shared_ptr <entry_impl_base> impl)
  : is_valid_ (true), impl_ (impl)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get children
//! \return vector of children
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::vector <entry>
entry::get_children () const
{
  auto impl_children = impl_->get_children ();

  std::vector <entry> children;
  children.reserve (impl_children.size ());

  for (auto i : impl_children)
    children.push_back (entry (i));

  return children;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get parent
//! \return parent, if found
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
entry
entry::get_parent () const
{
  auto impl_parent = impl_->get_parent ();

  return (impl_parent) ? entry (impl_parent) : entry ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get child by name
//! \param name name
//! \param case_sensitive if name comparision is case sensitive or not
//! \return child, if found
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
entry
entry::get_child_by_name (const std::string& name, bool case_sensitive) const
{
  auto impl_child = impl_->get_child_by_name (name, case_sensitive);

  return (impl_child) ? entry (impl_child) : entry ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get child by path
//! \param path relative path
//! \param case_sensitive if name comparision is case sensitive or not
//! \return child, if found
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
entry
entry::get_child_by_path (const std::string& path, bool case_sensitive) const
{
  std::string::size_type pos = 0;
  std::string::size_type end = path.find ('/');
  auto impl_child = impl_;

  while (end != std::string::npos && bool (impl_child))
    {
      const std::string name = path.substr (pos, end - pos);
      impl_child = impl_child->get_child_by_name (name, case_sensitive);
      pos = end + 1;
      end = path.find ('/', pos);
    }

  if (impl_child)
    {
      const std::string name = path.substr (pos);
      impl_child = impl_child->get_child_by_name (name, case_sensitive);
    }

  return (impl_child) ? entry (impl_child) : entry ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get streams
//! \return vector of streams
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::vector <stream>
entry::get_streams () const
{
  auto impl_streams = impl_->get_streams ();

  std::vector <stream> streams;
  streams.reserve (impl_streams.size ());

  for (auto i : impl_streams)
    streams.push_back (stream (i));

  return streams;
}

} // namespace filesystem
} // namespace mobius
