# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import mobius
import struct
import datetime

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief get data as UNIX datetime
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def get_unix_datetime (data):
  timestamp = struct.unpack ('<I', data)[0]
  
  if timestamp > 0:
    delta = datetime.timedelta (seconds=timestamp)
    date = datetime.datetime (1970, 1, 1, 0, 0, 0) + delta
  else:
    date = None

  return date

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief get data as UTF-8 string (sanitize non UTF-8 strings)
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def get_utf8_string (data):
  try:
    s = unicode (data, 'utf-8')
  except UnicodeDecodeError, e:
    s = unicode (data, 'iso-8859-1', errors='ignore')

  return s.encode ('utf-8')    

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief decode metadata structure
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def decode (data):
  metadata = {}

  i = 0
  while i + 2 < len (data):

    type = ord (data[i])
    field_size = ord (data[i + 1]) | (ord (data[i + 2]) << 8)
    value = data[i + 3: i + 3 + field_size]

    if type == 1:
      metadata['path'] = value

    elif type == 2:
      metadata['title'] = get_utf8_string (value)

    elif type == 3:
      metadata['artist'] = get_utf8_string (value)

    elif type == 4:
      metadata['album'] = get_utf8_string (value)

    elif type == 5:
      metadata['category'] = get_utf8_string (value)

    elif type == 6:
      metadata['year'] = get_utf8_string (value)

    elif type == 7:
      metadata['vidinfo'] = value

    elif type == 8:
      metadata['language'] = get_utf8_string (value)

    elif type == 9:
      metadata['url'] = get_utf8_string (value)

    elif type == 10:
      metadata['comment'] = get_utf8_string (value)

    elif type == 11:
      metadata['download_completed_time'] = get_unix_datetime (value)

    elif type == 17:
      metadata['flag_corrupted'] = True

    elif type == 18:
      metadata['hash_of_phash'] = value

    else:
      mobius.core.log ('p2p.ares.decoder_share_metadata.py: Unknown field type (%d)' % type)
    
    i = i + 3 + field_size

  return metadata
