# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import mobius
import pymobius.ant.turing.add_passwords_to_database
import pymobius.ant.turing.chromium_passwords
import pymobius.ant.turing.credentials
import pymobius.ant.turing.credhist_hashes
import pymobius.ant.turing.dpapi_sys_mk
import pymobius.ant.turing.dpapi_user_mk
import pymobius.ant.turing.ie_passwords
import pymobius.ant.turing.registry_hashes
import pymobius.ant.turing.registry_passwords
import pymobius.ant.turing.wifi_passwords
import pymobius.ant.turing.test_case_passwords
import pymobius.ant.turing.test_hash_passwords

import binascii
import re

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ANT_ID = 'turing'
ANT_NAME = 'Turing'
ANT_VERSION = '1.0'

REGEX_WORDS = re.compile ("(\w[\w']*\w|\w)")

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Generic dataholder class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class dataholder (object):
  pass

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Ant: Turing
# @author Eduardo Aguiar
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Ant (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, item):
    self.name = 'Turing Cryptographic Agent'
    self.version = '1.0'

    self.__item = item
    self.__passwords = []
    self.__hashes = []
    self.__keys = []
 
    self.__unique_passwords = set ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Run ant
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def run (self):
    if not self.__item.datasource:
      return

    mobius.core.log ('INF ant.turing started')

    # create ants
    self.__ants = [
      pymobius.ant.turing.chromium_passwords.Ant (self.__item),
      pymobius.ant.turing.credentials.Ant (self.__item),
      pymobius.ant.turing.credhist_hashes.Ant (self.__item),
      pymobius.ant.turing.dpapi_sys_mk.Ant (self.__item),
      pymobius.ant.turing.dpapi_user_mk.Ant (self.__item),
      pymobius.ant.turing.ie_passwords.Ant (self.__item),
      pymobius.ant.turing.registry_hashes.Ant (self.__item),
      pymobius.ant.turing.registry_passwords.Ant (self.__item),
      pymobius.ant.turing.wifi_passwords.Ant (self.__item),
      pymobius.ant.turing.test_hash_passwords.Ant (self.__item),
      pymobius.ant.turing.test_case_passwords.Ant (self.__item),
      pymobius.ant.turing.add_passwords_to_database.Ant (self.__item)
    ]

    # run sub-ants
    for ant in self.__ants:
      ant.run ()

    # consolidate data
    for ant in self.__ants:
      ant.on_export_data (self)

    # terminate sub-ants
    for ant in self.__ants:
      ant.on_stop (self)

    # log info
    mobius.core.log ('INF ant.turing: passwords = %d' % len (self.__passwords))
    mobius.core.log ('INF ant.turing: hashes = %d' % len (self.__hashes))
    mobius.core.log ('INF ant.turing: keys = %d' % len (self.__keys))

    # save data
    self.__save_data ()

    mobius.core.log ('INF ant.turing finished')
    return

    self.__retrieve_sys_win_credential_passwords ()
    self.__retrieve_win_credentials_passwords ()
    self.__test_passwords ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Save data into model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __save_data (self):
    c = self.__item.case
    transaction = c.new_transaction ()

    # remove old data
    self.__item.remove_passwords ()
    self.__item.remove_password_hashes ()

    # save passwords
    passwords = [(p.type, p.description, p.value, p) for p in self.__passwords]

    for ptype, description, value, pobj in sorted (passwords):
      p = self.__item.new_password (ptype, value, description)
      p.set_attribute ('item', self.__item.name)

      for name, value in pobj.metadata:
        p.set_attribute (name, str (value))

    # save password hashes
    hashes = [(h.type, h.description, h.value, h) for h in self.__hashes]

    for htype, description, value, hobj in sorted (hashes):
      hash_value = binascii.hexlify (value)

      h = self.__item.new_password_hash (htype, hash_value, description)

      if hobj.password != None:
        h.password = hobj.password

      for name, value in hobj.metadata:
        h.set_attribute (name, value)

    self.__item.set_ant (ANT_ID, ANT_NAME, ANT_VERSION)
    transaction.commit ()
  
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add key to model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_key (self, key):
    self.__keys.append (key)

    for ant in self.__ants:
      f = getattr (ant, 'on_key', None)

      if f:
        f (self, key)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add password to model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_password (self, p):
    self.__passwords.append (p)

    for ant in self.__ants:
      f = getattr (ant, 'on_add_password', None)

      if f:
        f (self, p)

    self.test_password (p.value)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Test password
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def test_password (self, value):

    if value in self.__unique_passwords:
      return

    self.__unique_passwords.add (value)

    for ant in self.__ants:
      f = getattr (ant, 'on_test_password', None)

      if f:
        f (self, value)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Add hash to model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_hash (self, h):
    self.__hashes.append (h)

    for ant in self.__ants:
      f = getattr (ant, 'on_hash', None)

      if f:
        f (self, h)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Test passwords
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __test_passwords (self):

    case = self.__item.case
    keywords = set (p.value for p in self.__passwords)
    
    for h in self.__password_hashes:
      metadata = dict (h.metadata)
      keywords.update (REGEX_WORDS.findall (metadata.get ('username', '')))
      keywords.update (REGEX_WORDS.findall (metadata.get ('fullname', '')))
      keywords.update (REGEX_WORDS.findall (metadata.get ('admin_comment', '')))
      keywords.update (REGEX_WORDS.findall (metadata.get ('user_comment', '')))

    for h in case.get_password_hashes ():
      keywords.update (REGEX_WORDS.findall (h.get_attribute ('username')))
      keywords.update (REGEX_WORDS.findall (h.get_attribute ('fullname')))
      keywords.update (REGEX_WORDS.findall (h.get_attribute ('admin_comment')))
      keywords.update (REGEX_WORDS.findall (h.get_attribute ('user_comment')))

    keywords = list (keywords)

    # Test keywords
    i = 0
    l = len (keywords)

    while i < l and hashes:
      keyword = keywords[i]
      nt_value = binascii.hexlify (mobius.os.win.hash_nt (keyword))
      lm_value = binascii.hexlify (mobius.os.win.hash_lm (keyword))
      h_tmp = []

      for h in hashes:
        attrs = dict (h.metadata)

        if h.type == 'nt':
          value = nt_value
          
        elif h.type == 'lm':
          value = lm_value

        elif h.type == 'msdcc1':
          value = binascii.hexlify (mobius.os.win.hash_msdcc1 (keyword, attrs['username']))

        elif h.type == 'msdcc2':
          value = binascii.hexlify (mobius.os.win.hash_msdcc2 (keyword, attrs['username'], int (attrs['iterations'])))

        if value == h.value:
          h.password = keyword

        else:
          h_tmp.append (h)

      if len (h_tmp) < len (hashes):
        hashes = h_tmp

      i = i + 1
