// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "imagefile_impl_solo.h"
#include <mobius/io/file.h>
#include <mobius/charset.h>
#include <mobius/exception.inc>
#include <mobius/string_functions.h>
#include <cctype>
#include <regex>
#include <stdexcept>

namespace mobius
{
namespace imagefile
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get URL for first segment file
//! \param url imagefile URL
//! \return first segment file URL
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static const std::string
get_first_segment_url (const std::string& url)
{
  return url.substr (0, url.length () - 4) + ".001";
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief check if URL is an instance of imagefile solo
//! \return true/false
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
imagefile_impl_solo::is_instance (const std::string& url)
{
  bool instance = false;

  mobius::io::file f (url);

  if (f && f.exists ())
    {
      auto reader = f.new_reader ();
      mobius::bytearray data = reader.read (14);

      instance = data == "[SEIZE HEADER]";
    }

  return instance;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief construct object
//! \param url imagefile URL
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
imagefile_impl_solo::imagefile_impl_solo (const std::string& url)
  : url_ (url),
    split_imagefile_ (get_first_segment_url (url))
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new reader for imagefile
//! \return reader
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::io::reader
imagefile_impl_solo::new_reader () const
{
  return split_imagefile_.new_reader ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new writer for imagefile
//! \return writer
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::io::writer
imagefile_impl_solo::new_writer () const
{
  throw std::runtime_error (MOBIUS_EXCEPTION_MSG ("writer not implemented"));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get metadata
//! \return imagefile metadata
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::metadata
imagefile_impl_solo::get_metadata () const
{
  return mobius::metadata
  {
    {
      "url",
      "URL",
      "std::string",
      get_url ()
    },
    {
      "type",
      "type",
      "std::string",
      get_type ()
    },
    {
      "size",
      "size",
      "size_type",
      std::to_string (get_size ()) + " bytes"
    },
    {
      "sectors",
      "number of sectors",
      "size_type",
      std::to_string (get_sectors ())
    },
    {
      "sector_size",
      "sector size",
      "size_type",
      std::to_string (get_sector_size ()) + " bytes"
    },
    {
      "segments",
      "segments",
      "size_type",
      std::to_string (get_segments ())
    },
    {
      "segment_size",
      "segment_size",
      "size_type",
      std::to_string (get_segment_size ()) + " bytes"
    },
    {
      "drive_vendor",
      "drive vendor",
      "std::string",
      get_drive_vendor ()
    },
    {
      "drive_model",
      "drive model",
      "std::string",
      get_drive_model ()
    },
    {
      "drive_serial_number",
      "drive serial number",
      "std::string",
      get_drive_serial_number ()
    },
    {
      "acquisition_user",
      "acquisition user name",
      "std::string",
      get_acquisition_user ()
    },
    {
      "acquisition_time",
      "acquisition date/time",
      "mobius::datetime::datetime",
      to_string (get_acquisition_time ())
    },
    {
      "acquisition_tool",
      "acquisition tool",
      "std::string",
      get_acquisition_tool ()
    },
    {
      "acquisition_platform",
      "acquisition platform",
      "std::string",
      get_acquisition_platform ()
    },
    {
      "hash_md5",
      "MD5 hash",
      "std::string",
      get_hash_md5 ()
    },
  };
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief load metadata on demand
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
imagefile_impl_solo::_load_metadata () const
{
  if (metadata_loaded_)
    return;

  // load metadata
  mobius::io::file f (url_);
  constexpr mobius::io::file::size_type LOG_MAX_SIZE = 1048576;  // 1 MiB

  if (f.exists () && f.get_size () < LOG_MAX_SIZE)
    {
      std::regex REGEX_SEIZE_TIME ("\\nSeize Time = *([0-9]+):([0-9]+):([0-9]+)\\n");
      std::regex REGEX_SEIZE_DATE ("\\nSeize Date = *([0-9]+)/([0-9]+)/([0-9]+)\\n");
      std::regex REGEX_TOTAL_SECTORS ("\\nSeize Size = ([0-9]+)\\n");
      std::regex REGEX_MD5 ("\\nMD5: (.+)\\n");
      std::regex REGEX_SUSPECT ("Suspect: Model: ([^.]+?)\\. Serial Number: ([^.]+?)\\..*Block size: ([0-9]+)\\..*\\. Evidence");
      std::regex REGEX_SOLO_PRODUCT ("\\nProduct = (.+)\\n");
      std::regex REGEX_SOLO_SERIAL ("\\nSerial #: (.+)\\n");
      std::regex REGEX_SOLO_SOFTWARE ("\\nSoftware Version (.+)\\n");

      // parse .txt file
      auto reader = f.new_reader ();
      mobius::bytearray data = reader.read (f.get_size ());
      const std::string text = conv_charset_to_utf8 (data, "ASCII");
      std::smatch match;
      mobius::datetime::date acquisition_date;
      std::string product;
      std::string serial;

      if (std::regex_search (text, match, REGEX_SUSPECT))
        {
          drive_model_ = match[1].str ();
          drive_serial_number_ = match[2].str ();
          sector_size_ = stoi (match[3].str ());
        }

      if (std::regex_search (text, match, REGEX_SEIZE_DATE))
        {
          acquisition_date = mobius::datetime::date (
                                stoi (match[3].str ()),
                                stoi (match[1].str ()),
                                stoi (match[2].str ()));

        }
     
      if (std::regex_search (text, match, REGEX_SEIZE_TIME))
        {
          mobius::datetime::time time = mobius::datetime::time (
                                          stoi (match[1].str ()),
                                          stoi (match[2].str ()),
                                          stoi (match[3].str ()));

          if (acquisition_date)
            acquisition_time_ = mobius::datetime::datetime (acquisition_date, time);
        }

      if (std::regex_search (text, match, REGEX_TOTAL_SECTORS))
        {
          sectors_ = stoll (match[1].str ());
          size_ = sectors_ * sector_size_;
        }

      if (std::regex_search (text, match, REGEX_MD5))
        {
          std::string value = mobius::string::remove_char (match[1].str (), ' ');
          hash_md5_ = mobius::string::tolower (value);
        }

      if (std::regex_search (text, match, REGEX_SOLO_PRODUCT))
        product = match[1].str ();

      if (std::regex_search (text, match, REGEX_SOLO_SERIAL) && !product.empty ())
        acquisition_platform_ = product + " (s/n: " + match[1].str () + ')';

      if (std::regex_search (text, match, REGEX_SOLO_SOFTWARE))
        acquisition_tool_ = "Solo software v" + match[1].str ();

      segments_ = split_imagefile_.get_segments ();
      segment_size_ = split_imagefile_.get_segment_size ();
      acquisition_user_ = f.get_user_name ();

      normalize_drive_info (drive_vendor_, drive_model_, drive_serial_number_);
    }

  // set metadata loaded
  metadata_loaded_ = true;
}

} // namespace imagefile
} // namespace mobius
