// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief  C++ API module wrapper
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <Python.h>
#include "api_reader.h"

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: static prototypes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void api_reader_tp_dealloc (api_reader_o *);
static PyObject *api_reader_get_is_seekable (api_reader_o *);
static PyObject *api_reader_get_is_rewindable (api_reader_o *);
static PyObject *api_reader_get_size (api_reader_o *);
static PyObject *api_reader_f_read (api_reader_o *, PyObject *);
static PyObject *api_reader_f_tell (api_reader_o *, PyObject *);
static PyObject *api_reader_f_seek (api_reader_o *, PyObject *);
static PyObject *api_reader_f_rewind (api_reader_o *, PyObject *);
static PyObject *api_reader_f_skip (api_reader_o *, PyObject *);

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: getters and setters structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyGetSetDef api_reader_getsetters[] =
{
  {
    (char *) "is_seekable",
    (getter) api_reader_get_is_seekable,
    (setter) 0,
    (char *) "check if reader is seekable", NULL
  },
  {
    (char *) "is_rewindable",
    (getter) api_reader_get_is_rewindable,
    (setter) 0,
    (char *) "check if reader is rewindable", NULL
  },
  {
    (char *) "size",
    (getter) api_reader_get_size,
    (setter) 0,
    (char *) "data size in bytes", NULL
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: methods structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyMethodDef api_reader_methods[] =
{
  {
    (char *) "read",
    (PyCFunction) api_reader_f_read,
    METH_VARARGS,
    "read bytes from reader"
  },
  {
    (char *) "tell",
    (PyCFunction) api_reader_f_tell,
    METH_VARARGS,
    "get current read position"
  },
  {
    (char *) "seek",
    (PyCFunction) api_reader_f_seek,
    METH_VARARGS,
    "set current read position"
  },
  {
    (char *) "rewind",
    (PyCFunction) api_reader_f_rewind,
    METH_VARARGS,
    "set current read position to the beginning of data"
  },
  {
    (char *) "skip",
    (PyCFunction) api_reader_f_skip,
    METH_VARARGS,
    "set read position n bytes ahead"
  },
  {NULL} // sentinel
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: type structure
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
PyTypeObject api_reader_t =
{
  PyObject_HEAD_INIT (0)
  0,                                       		// ob_size
  "api.reader",                             		// tp_name
  sizeof (api_reader_o),                    		// tp_basicsize
  0,                                       		// tp_itemsize
  (destructor) api_reader_tp_dealloc,       		// tp_dealloc
  0,                                       		// tp_print
  0,                                       		// tp_getattr
  0,                                       		// tp_setattr
  0,                                       		// tp_compare
  0,                                       		// tp_repr
  0,                                       		// tp_as_number
  0,                                       		// tp_as_sequence
  0,                                       		// tp_as_mapping
  0,                                       		// tp_hash
  0,                                       		// tp_call
  0,                                       		// tp_str
  0,                                       		// tp_getattro
  0,                                       		// tp_setattro
  0,                                       		// tp_as_buffer
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE,		// tp_flags
  "reader for mobius.io.file",             		// tp_doc
  0,                                       		// tp_traverse
  0,                                       		// tp_clear
  0,                                       		// tp_richcompare
  0,                                       		// tp_weaklistoffset
  0,                                       		// tp_iter
  0,                                       		// tp_iternext
  api_reader_methods,                       		// tp_methods
  0,                                       		// tp_members
  api_reader_getsetters,                    		// tp_getset
  0,                                       		// tp_base
  0,                                       		// tp_dict
  0,                                       		// tp_descr_get
  0,                                       		// tp_descr_set
  0,                                       		// tp_dictoffset
  0,                                       		// tp_init
  0,                                       		// tp_alloc
  0                                        		// tp_new
};

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: tp_alloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
api_reader_o *
api_reader_tp_alloc ()
{
  return (api_reader_o *) api_reader_t.tp_alloc (&api_reader_t, 0);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: tp_dealloc
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static void
api_reader_tp_dealloc (api_reader_o *self)
{
  delete self->obj;
  self->ob_type->tp_free ((PyObject*) self);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: is_seekable getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_reader_get_is_seekable (api_reader_o *self)
{
  return PyBool_FromLong (self->obj->is_seekable ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: is_rewindable getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_reader_get_is_rewindable (api_reader_o *self)
{
  return PyBool_FromLong (self->obj->is_rewindable ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: size getter
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_reader_get_size (api_reader_o *self)
{
  return PyInt_FromLong (self->obj->get_size ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: read
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_reader_f_read (api_reader_o *self, PyObject *args)
{
  // parse input args
  std::uint64_t arg_size;

  if (!PyArg_ParseTuple (args, "K", &arg_size))
    return NULL;

  // build and return value
  try
    {
      auto ret_value = self->obj->read (arg_size);

      return PyString_FromStringAndSize ((const char *) ret_value.data (), ret_value.size ());
    }
  catch (const std::runtime_error& e)
    {
      PyErr_SetString (PyExc_IOError, e.what ());
      return NULL;
    }
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: tell
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_reader_f_tell (api_reader_o *self, PyObject *args)
{
  return PyInt_FromLong (self->obj->tell ());
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: seek
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_reader_f_seek (api_reader_o *self, PyObject *args)
{
  // parse input args
  std::int64_t arg_offset;
  int arg_whence = 0;

  if (!PyArg_ParseTuple (args, "L|i", &arg_offset, &arg_whence))
    return NULL;

  // execute C++ function
  mobius::io::reader::whence_type w;

  if (arg_whence == 0)
    w = mobius::io::reader::whence_type::beginning;

  else if (arg_whence == 1)
    w = mobius::io::reader::whence_type::current;

  else if (arg_whence == 2)
    w = mobius::io::reader::whence_type::end;

  self->obj->seek (arg_offset, w);

  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: rewind
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_reader_f_rewind (api_reader_o *self, PyObject *args)
{
  self->obj->rewind ();

  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api.reader: skip
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
static PyObject *
api_reader_f_skip (api_reader_o *self, PyObject *args)
{
  // parse input args
  std::int64_t arg_size;

  if (!PyArg_ParseTuple (args, "L", &arg_size))
    return NULL;

  // execute C++ function
  self->obj->skip (arg_size);

  Py_INCREF (Py_None);
  return Py_None;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief api_reader_new
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
api_reader_o *
api_reader_new ()
{
  return (api_reader_o *) _PyObject_New (&api_reader_t);
}

