// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/crypt/hash_zip.h>
#include <mobius/crypt/hash_crc32.h>

namespace mobius
{
namespace crypt
{
static constexpr std::uint32_t ZIPCONST = 0x08088405U;
static constexpr std::uint32_t INITK0 = 0x12345678;
static constexpr std::uint32_t INITK1 = 0x23456789;
static constexpr std::uint32_t INITK2 = 0x34567890;

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief default constructor
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
hash_zip::hash_zip () noexcept
  : k0_ (INITK0), k1_ (INITK1), k2_ (INITK2)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief reset hash value
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
hash_zip::reset () noexcept
{
  k0_ = INITK0;
  k1_ = INITK1;
  k2_ = INITK2;
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief update hash one byte
//!\param b byte
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
hash_zip::update_byte (std::uint8_t b) noexcept
{
  k0_ = crc32 (k0_, b);
  k1_ = (k1_ + (k0_ & 0xff)) * ZIPCONST + 1;
  k2_ = crc32 (k2_, k1_ >> 24);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief return hash digest
//!\return hash digest
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const bytearray
hash_zip::get_digest ()
{
  mobius::bytearray d (12);
  d[0] = (k0_ >> 24);
  d[1] = (k0_ >> 16) & 0xff;
  d[2] = (k0_ >> 8) & 0xff;
  d[3] = k0_ & 0xff;
  d[4] = (k1_ >> 24);
  d[5] = (k1_ >> 16) & 0xff;
  d[6] = (k1_ >> 8) & 0xff;
  d[7] = k1_ & 0xff;
  d[8] = (k2_ >> 24);
  d[9] = (k2_ >> 16) & 0xff;
  d[10] = (k2_ >> 8) & 0xff;
  d[11] = k2_ & 0xff;

  return d;
}

} // crypt namespace
} // mobius namespace
