# -*- encoding: utf-8 -*-
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import os
import os.path
import re
import shutil

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
VALUE_EXP = re.compile (r'\$\{(.*?)\}')
STATEMENT_VARS = {'text' : {'text' : '', 'newline' : True},
    'verbatim' : {'text' : '', 'newline' : True},
    'for' : {'vars' : '', 'values' : '', 'code' : '@BLOCK@'},
    'if' : {'condition' : '', 'if_code' : '@BLOCK@', 'else_code' : '@BLOCK@', 'elifs' : '@LIST@'},
    'elif' : {'condition' : '', 'code' : '@BLOCK@'},
    'call' : {'report' : '', 'args' : ''},
    'output' : {'filename' : ''},
    'block' : {'statements' : '@LIST@'},
    'report' : {'name' : '', 'code' : '@BLOCK@', 'args' : ''}}

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief generic generator for python source
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class PySource (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief create sourcefile
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, path):
    self.pos = 0
    self.fp = open (path, 'w')
    self.path = path

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief close sourcefile
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def close (self):
    self.fp.close ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief indent one level
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def indent (self):
    self.pos += 1

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief dedent one level
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def dedent (self):
    self.pos -= 1

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief write string to sourcefile
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def write (self, str='', indentation=True, newline=True, escaped=False):
    if indentation:
      self.fp.write ('  ' * self.pos)

    if escaped:
      str = str.replace ('\\', '\\\\')
      str = str.replace ("'", "\\'")
      str = str.replace (r"%", r"%%")
      str = str.replace ('\n', '\\n')
      str = str.replace ('\t', '\\t')

    self.fp.write (str or '')

    if newline:
      self.fp.write ('\n')

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief generate python code from report template
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Generator (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate code
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate (self, report_id, pyfile, mediator):
    self.mediator = mediator

    # create cache subdir, if necessary
    pydir = os.path.dirname (pyfile)
    if not os.path.exists (pydir):
      os.makedirs (pydir)

    # create .dep file
    deppath = os.path.join (pydir, '%s.dep' % report_id)
    depfile = open (deppath, 'w')

    # generate source
    try:
      source = PySource (pyfile)
      self.generate_py_header (source)
      self.generate_py_write_escape (source)
      self.generate_py_main_method (source, report_id, depfile)
      self.generate_py_report_methods (source, depfile)
      self.generate_py_footer (source)
    except:
      depfile.close ()
      os.remove (deppath)
      raise

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate source header
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_py_header (self, source):
    source.write ('# -*- encoding: utf-8 -*-')
    source.write ('#%s' % ('-=' * 39))
    source.write ('# Generated by Mobius Forensic Toolkit. Do not edit this file')
    source.write ('#%s' % ('-=' * 39))
    source.write ('class Generator (object):')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate source write_escape method
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_py_write_escape (self, source):
    source.write ()
    source.write ('  def write_escape (self, output, s):')
    source.write ("    s = s.replace ('\\\\', '\\\\backslash{}')")
    source.write ("    s = s.replace ('_', '\\\\_')")
    source.write ("    s = s.replace ('#', '\\\\#')")
    source.write ("    s = s.replace ('&', '\\\\&')")
    source.write ("    s = s.replace ('%', '\\\\%')")
    source.write ("    s = s.replace ('$', '\\\\$')")
    source.write ("    s = s.replace ('°C', '\\\\textcelsius{}')")
    source.write ("    output.write (s or '')")

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate main method
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_py_main_method (self, source, report_id, depfile):
    self.references = []
    self.generated = set ()

    report = self.mediator.call ('report.load', report_id)
    args = 'self'
    if report.args:
      args = args + ', ' + report.args

    source.write ()
    source.write ('  def generate (%s):' % args)
    source.write ('    output = None')
    source.indent ()
    self.generate_statement (source, report.code)
    source.dedent ()

    depfile.write ('%s\n' % report_id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate nested report
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_py_report_methods (self, source, depfile):

    # generate nested reports methods
    while self.references:
      reference_id = self.references[0]
      self.generated.add (reference_id)
      report = self.mediator.call ('report.load', reference_id)

      args = 'self, output'
      if report.args:
        args = args + ', ' + report.args

      source.write ()
      source.write ('  def generate_%s (%s):' % (self.get_underscore_id (reference_id), args))
      source.indent ()
      self.generate_statement (source, report.code)
      source.dedent ()

      self.references = self.references[1:]
      depfile.write ('%s\n' % reference_id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate source footer
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_py_footer (self, source):
    source.write ()
    source.write ('generator = Generator ()')
    source.write ('generator.generate (*args)')
    source.close ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate statement
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_statement (self, source, obj):
    method_name = 'generate_%s' % obj.statement.replace ('-', '_')
    statement_method = getattr (self, method_name, None)

    if statement_method:
      statement_method (source, obj)

    else:
      print 'unknown statement', obj.statement

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate block
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_block (self, source, obj):
    source.indent ()

    if not obj.statements:
      source.write ('pass')
    else:
      for statement in obj.statements:
        self.generate_statement (source, statement)

    source.dedent ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate verbatim
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_verbatim (self, source, obj):
    source.write ('output.write ("', newline=False)
    source.write (obj.text, escaped=True, indentation=False, newline=False)
    if obj.newline:
      source.write ('\\n', indentation=False, newline=False)
    source.write ('")', indentation=False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate text
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_text (self, source, obj):
    format, args = self.get_expand_vars (obj.text)
    source.write ("self.write_escape (output, '%s" % format, newline=False)

    if obj.newline:
      source.write ('\\n', indentation=False, newline=False)
    source.write ("'", indentation=False, newline=False)

    if args:
      source.write (" %% (%s)" % ', '.join (args), newline=False, indentation=False)

    source.write (")", indentation=False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate output
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_output (self, source, obj):
    format, args = self.get_expand_vars (obj.filename)
    source.write ("output = open ('%s'" % format, newline=False)

    if args:
      source.write (" %% (%s)" % ', '.join (args), newline=False, indentation=False)

    source.write (", 'w')", indentation=False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate call
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_call (self, source, obj):
    if obj.args:
      args = 'output, ' + obj.args
    else:
      args = 'output'

    source.write ('self.generate_%s (%s)' % (self.get_underscore_id (obj.report), args))

    if obj.report not in self.generated:
      self.references.append (obj.report)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate for
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_for (self, source, obj):
    source.write ('for %s in %s:' % (obj.vars, obj.values))
    self.generate_statement (source, obj.code)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate if
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_if (self, source, obj):
    source.write ('if %s:' % obj.condition)
    self.generate_statement (source, obj.if_code)

    for item in obj.elifs:
      source.write ('elif %s:' % item.condition)
      self.generate_statement (source, item.code)

    source.write ('else:')
    self.generate_statement (source, obj.else_code)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief generate switch
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def generate_switch (self, source, obj):
    first = True
    for case in obj.cases:
      if first:
        stmt = 'if'
        first = False
      else:
        stmt = 'elif'
      source.write ('%s %s == %s:' % (stmt, obj.expression, case.value))
      source.indent ()
      self.generate_statement (source, case.code)
      source.dedent ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief return underscored report.id string
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_underscore_id (self, report_id):
    report_id = report_id.replace ('-', '_')
    report_id = report_id.replace ('.', '_')
    return report_id

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief return format string and args for ${} strings
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_expand_vars (self, text):
    pos = 0
    format = ''
    args = []

    for exp in VALUE_EXP.finditer (text):
      format += self.py_escape (text[pos:exp.start (0)]) + r'%s'
      args.append (exp.group (1))
      pos = exp.end (0)

    format += self.py_escape (text[pos:])

    return format, args

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief escape python controle chars
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def py_escape (self, str):
    str = str.replace ('\\', '\\\\')
    str = str.replace ("'", "\\'")
    str = str.replace (r"%", r"%%")
    str = str.replace ('\n', '\\n')
    str = str.replace ('\t', '\\t')
    return str

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Report statement
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Statement (object):
  def __init__ (self):
    self.statement = ''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Icon
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'report-model'
    self.name = 'Report Model'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Report model and services'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.advertise ('report.iter', self.svc_report_iter)
    self.mediator.advertise ('report.new', self.svc_report_new)
    self.mediator.advertise ('report.load', self.svc_report_load)
    self.mediator.advertise ('report.save', self.svc_report_save)
    self.mediator.advertise ('report.remove', self.svc_report_remove)
    self.mediator.advertise ('report.new-statement', self.svc_report_new_statement)
    self.mediator.advertise ('report.create-folder', self.svc_report_create_folder)
    self.mediator.advertise ('report.remove-folder', self.svc_report_remove_folder)
    self.mediator.advertise ('report.run', self.svc_report_run)
    self.cachedir = self.mediator.call ('app.get-path', 'cache', 'report')
    self.reportdir = self.mediator.call ('app.get-path', 'report')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: report.new
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_report_new (self):
    report = self.svc_report_new_statement ('report')
    return report

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: report.new-statement
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_report_new_statement (self, stmt_id):
    stmt = Statement ()
    stmt.statement = stmt_id

    # fill statement vars
    stmt_vars = STATEMENT_VARS.get (stmt_id, {})

    for name, value in stmt_vars.items ():
      if value == '@BLOCK@':
        value = self.svc_report_new_statement ('block')

      elif value == '@LIST@':
        value = []

      setattr (stmt, name, value)

    return stmt

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: report.load
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_report_load (self, report_id):
    path = self.get_report_path (report_id)

    if os.path.exists (path):
      report = self.mediator.call ('xml.unpickle', path)
    else:
      report = None

    return report

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: report.save
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_report_save (self, report, report_id):
    path = self.get_report_path (report_id)

    dirname = os.path.dirname (path)
    if not os.path.exists (dirname):
      os.makedirs (dirname)

    self.mediator.call ('xml.pickle', path, report)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: report.remove
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_report_remove (self, report_id):
    path = self.get_report_path (report_id)

    if os.path.exists (path):
      os.remove (path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: report.iter
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_report_iter (self):
    pos = len (self.reportdir) + 1

    for root, dirs, files in os.walk (self.reportdir, topdown=True):
      for name in dirs:
        abs_path = os.path.join (root, name)
        rel_path = abs_path[pos:]
        name = os.path.splitext (rel_path)[0].replace ('/', '.')
        if name:
          yield 'FOLDER', name
       
      for name in files:
        abs_path = os.path.join (root, name)
        rel_path = abs_path[pos:]
        name = os.path.splitext (rel_path)[0].replace ('/', '.')
        yield 'REPORT', name

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: report.create-folder
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_report_create_folder (self, folder_id):
    path = os.path.join (self.reportdir, folder_id.replace ('.', '/'))
    os.makedirs (path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: report.remove-folder
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_report_remove_folder (self, folder_id):
    path = os.path.join (self.reportdir, folder_id.replace ('.', '/'))

    if os.path.isdir (path):
      shutil.rmtree (path)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief service: report.run
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def svc_report_run (self, report_id, *args):
    pyfile = os.path.join (self.cachedir, '%s.py' % report_id)

    # generate report code if necessary
    if not self.get_report_is_cached (report_id):
      generator = Generator ()
      generator.generate (report_id, pyfile, self.mediator)

    # execute report
    code = open (pyfile, 'r')
    data = {'args' : args}
    exec (code, globals (), data)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief return report path
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_report_path (self, report_id):
    return os.path.join (self.reportdir, '%s.report' % report_id.replace ('.', '/'))

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief check if template is cached
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_report_is_cached (self, report_id):
    depfile = os.path.join (self.cachedir, '%s.dep' % report_id)

    if not os.path.exists (depfile):
      return False

    # get depfile modification time
    st_depfile = os.stat (depfile)

    # compare with references' modification time
    for line in open (depfile):
      reference_file = self.get_report_path (line.rstrip ())

      if not os.path.exists (reference_file):
        return False

      st_ref = os.stat (reference_file)

      if st_ref.st_mtime > st_depfile.st_mtime:
        return False

    return True
